#include "porto_layers_ready_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoLayersReadyNode {

Y_UNIT_TEST_SUITE(PortoLayersReadyNodeSuite) {

static TLogger logger({});

Y_UNIT_TEST(TestLayersReady) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override {
            ++Calls_;
            Place_ = place;
            Mask_ = mask;

            TVector<TPortoLayer> layers(2);
            layers[0].set_name("some_layer");
            layers[0].set_private_value(":some_hash");
            layers[1].set_name("other_layer");
            layers[1].set_private_value(":other_hash");

            return layers;
        }

        size_t Calls_ = 0;
        TString Place_ = "";
        TString Mask_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString place = "/some_place";
    const TVector<TString> layers = {"some_layer", "other_layer"};
    const TVector<TString> hashes = {"some_hash", "other_hash"};

    TPortoLayersReadyNodePtr node = new TPortoLayersReadyNode(TBasicTreeNodeDescriptor{1, "title"}, porto, place, layers, hashes);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoClient*)myPorto.Get())->Place_);
    UNIT_ASSERT_EQUAL("", ((TMyPortoClient*)myPorto.Get())->Mask_);
}

Y_UNIT_TEST(TestLayerNotExists) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override {
            ++Calls_;
            Place_ = place;
            Mask_ = mask;

            TVector<TPortoLayer> layers(2);
            layers[0].set_name("some_layer");
            layers[0].set_private_value(":some_hash");
            layers[1].set_name("second_layer");
            layers[1].set_private_value(":other_hash");

            return layers;
        }

        size_t Calls_ = 0;
        TString Place_ = "";
        TString Mask_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString place = "/some_place";
    const TVector<TString> layers = {"some_layer", "other_layer"};
    const TVector<TString> hashes = {"some_hash", "other_hash"};

    TPortoLayersReadyNodePtr node = new TPortoLayersReadyNode(TBasicTreeNodeDescriptor{1, "title"}, porto, place, layers, hashes);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoClient*)myPorto.Get())->Place_);
    UNIT_ASSERT_EQUAL("", ((TMyPortoClient*)myPorto.Get())->Mask_);
}

Y_UNIT_TEST(TestLayerWrongHash) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override {
            ++Calls_;
            Place_ = place;
            Mask_ = mask;

            TVector<TPortoLayer> layers(2);
            layers[0].set_name("some_layer");
            layers[0].set_private_value(":some_hash");
            layers[1].set_name("second_layer");
            layers[1].set_private_value(":other_hash");

            return layers;
        }

        size_t Calls_ = 0;
        TString Place_ = "";
        TString Mask_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString place = "/some_place";
    const TVector<TString> layers = {"some_layer", "second_layer"};
    const TVector<TString> hashes = {"some_hash", "second_hash"};

    TPortoLayersReadyNodePtr node = new TPortoLayersReadyNode(TBasicTreeNodeDescriptor{1, "title"}, porto, place, layers, hashes);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoClient*)myPorto.Get())->Place_);
    UNIT_ASSERT_EQUAL("", ((TMyPortoClient*)myPorto.Get())->Mask_);
}

Y_UNIT_TEST(TestLayerIncorrectHash) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override {
            ++Calls_;
            Place_ = place;
            Mask_ = mask;

            TVector<TPortoLayer> layers(2);
            layers[0].set_name("some_layer");
            layers[0].set_private_value("some_hash");
            layers[1].set_name("other_layer");
            layers[1].set_private_value(":other_hash");

            return layers;
        }

        size_t Calls_ = 0;
        TString Place_ = "";
        TString Mask_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString place = "/some_place";
    const TVector<TString> layers = {"some_layer", "other_layer"};
    const TVector<TString> hashes = {"some_hash", "other_hash"};

    TPortoLayersReadyNodePtr node = new TPortoLayersReadyNode(TBasicTreeNodeDescriptor{1, "title"}, porto, place, layers, hashes);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoClient*)myPorto.Get())->Place_);
    UNIT_ASSERT_EQUAL("", ((TMyPortoClient*)myPorto.Get())->Mask_);
}

Y_UNIT_TEST(TestLayerExistsWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<TVector<TPortoLayer>, TPortoError> ListLayers(const TString& place, const TString& mask) override {
            ++Calls_;
            Place_ = place;
            Mask_ = mask;

            return TPortoError{EPortoError::Unknown, "ListVolumes", "NO"};
        }

        size_t Calls_ = 0;
        TString Place_ = "";
        TString Mask_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString place = "/some_place";
    const TVector<TString> layers = {"some_layer", "other_layer"};
    const TVector<TString> hashes = {"some_hash", "other_hash"};

    TPortoLayersReadyNodePtr node = new TPortoLayersReadyNode(TBasicTreeNodeDescriptor{1, "title"}, porto, place, layers, hashes);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "ListVolumes");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoFailingClient*)myPorto.Get())->Place_);
    UNIT_ASSERT_EQUAL("", ((TMyPortoFailingClient*)myPorto.Get())->Mask_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoLayersReadyNode
