#include "porto_link_volume_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/volume_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoLinkVolumeNode {

Y_UNIT_TEST_SUITE(PortoLinkVolumeNodeSuite) {

static TLogger logger({});

Y_UNIT_TEST(TestLinkVolume) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> LinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool readOnly, bool required) override {
            ++Calls_;
            Path_ = path;
            Container_ = container;
            Target_ = target;
            ReadOnly_ = readOnly;
            Required_ = required;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TString Path_ = "";
        TPortoContainerName Container_ = {""};
        TString Target_ = "";
        bool ReadOnly_ = false;
        bool Required_ = false;
    };

    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString path = "/some_path";
    const TPortoContainerName container("some_container");
    const TString target = "some_target";
    const bool readOnly = false;
    const bool required = false;

    TPortoLinkVolumeNodePtr node = new TPortoLinkVolumeNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , holder
        , id
        , path
        , container
        , target
        , readOnly
        , required
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C((bool)result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoClient*)myPorto.Get())->Path_);
    UNIT_ASSERT_EQUAL(container, ((TMyPortoClient*)myPorto.Get())->Container_);
    UNIT_ASSERT_EQUAL(target, ((TMyPortoClient*)myPorto.Get())->Target_);
    UNIT_ASSERT_EQUAL(readOnly, ((TMyPortoClient*)myPorto.Get())->ReadOnly_);
    UNIT_ASSERT_EQUAL(required, ((TMyPortoClient*)myPorto.Get())->Required_);
}

Y_UNIT_TEST(TestLinkVolumeWithFailingLink) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> LinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool readOnly, bool required) override {
            ++Calls_;
            Path_ = path;
            Container_ = container;
            Target_ = target;
            ReadOnly_ = readOnly;
            Required_ = required;

            return TPortoError{EPortoError::Busy, "LinkVolume", "NO"};
        }

        size_t Calls_ = 0;
        TString Path_ = "";
        TPortoContainerName Container_ = {""};
        TString Target_ = "";
        bool ReadOnly_ = false;
        bool Required_ = false;
    };

    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString path = "/some_path";
    const TPortoContainerName container("some_container");
    const TString target = "some_target";
    const bool readOnly = false;
    const bool required = false;

    TPortoLinkVolumeNodePtr node = new TPortoLinkVolumeNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , holder
        , id
        , path
        , container
        , target
        , readOnly
        , required
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C((bool)result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "LinkVolume");
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).failed().message(), result.Success().Message);
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), 1);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoFailingClient*)myPorto.Get())->Path_);
    UNIT_ASSERT_EQUAL(container, ((TMyPortoFailingClient*)myPorto.Get())->Container_);
    UNIT_ASSERT_EQUAL(target, ((TMyPortoFailingClient*)myPorto.Get())->Target_);
    UNIT_ASSERT_EQUAL(readOnly, ((TMyPortoFailingClient*)myPorto.Get())->ReadOnly_);
    UNIT_ASSERT_EQUAL(required, ((TMyPortoFailingClient*)myPorto.Get())->Required_);
}

Y_UNIT_TEST(TestLinkVolumeWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> LinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool readOnly, bool required) override {
            ++Calls_;
            Path_ = path;
            Container_ = container;
            Target_ = target;
            ReadOnly_ = readOnly;
            Required_ = required;

            return TPortoError{EPortoError::Unknown, "LinkVolume", "NO"};
        }

        size_t Calls_ = 0;
        TString Path_ = "";
        TPortoContainerName Container_ = {""};
        TString Target_ = "";
        bool ReadOnly_ = false;
        bool Required_ = false;
    };

    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    const TString path = "/some_path";
    const TPortoContainerName container("some_container");
    const TString target = "some_target";
    const bool readOnly = false;
    const bool required = false;

    TPortoLinkVolumeNodePtr node = new TPortoLinkVolumeNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , holder
        , id
        , path
        , container
        , target
        , readOnly
        , required
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "LinkVolume");
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).failed().message(), result.Error().Message);
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), 1);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoFailingClient*)myPorto.Get())->Path_);
    UNIT_ASSERT_EQUAL(container, ((TMyPortoFailingClient*)myPorto.Get())->Container_);
    UNIT_ASSERT_EQUAL(target, ((TMyPortoFailingClient*)myPorto.Get())->Target_);
    UNIT_ASSERT_EQUAL(readOnly, ((TMyPortoFailingClient*)myPorto.Get())->ReadOnly_);
    UNIT_ASSERT_EQUAL(required, ((TMyPortoFailingClient*)myPorto.Get())->Required_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoLinkVolumeNode
