#include "porto_remove_layer_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoRemoveLayerNode {

Y_UNIT_TEST_SUITE(PortoRemoveLayerNodeSuite) {

static TLogger logger({});

Y_UNIT_TEST(TestRemoveLayer) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> RemoveLayer(const TString& layer, const TString& place) override {
            ++Calls_;
            LastLayer_ = layer;
            LastPlace_ = place;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TString LastLayer_ = "";
        TString LastPlace_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerId = "some_layer_id";
    const TString layerDownloadHash = "some_layer_download_hash";
    const TString layerName = "some_layer_name";
    const TString layerPlace = "/some_place";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerDownloadHash));

    TPortoRemoveLayerNodePtr node = new TPortoRemoveLayerNode(TBasicTreeNodeDescriptor{1, "title"}, porto, statusRepository, layerDownloadHash, layerName, layerPlace);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(0, statusRepository->GetObjectStatus(layerId).fail_counter());
    UNIT_ASSERT_EQUAL("", statusRepository->GetObjectStatus(layerId).failed().message());
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(layerName, ((TMyPortoClient*)myPorto.Get())->LastLayer_);
    UNIT_ASSERT_EQUAL(layerPlace, ((TMyPortoClient*)myPorto.Get())->LastPlace_);
}

Y_UNIT_TEST(TestImportLayerWithFailingRemove) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> RemoveLayer(const TString& layer, const TString& place) override {
            ++Calls_;
            LastLayer_ = layer;
            LastPlace_ = place;

            return TPortoError{EPortoError::LayerNotFound, "RemoveLayer", "NO"};
        }

        size_t Calls_ = 0;
        TString LastLayer_ = "";
        TString LastPlace_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerId = "some_layer_id";
    const TString layerDownloadHash = "some_layer_download_hash";
    const TString layerName = "some_layer_name";
    const TString layerPlace = "/some_place";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerDownloadHash));

    TPortoRemoveLayerNodePtr node = new TPortoRemoveLayerNode({1, "title"}, porto, statusRepository, layerDownloadHash, layerName, layerPlace);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "RemoveLayer");
    UNIT_ASSERT_EQUAL(1, statusRepository->GetObjectStatus(layerId).fail_counter());
    UNIT_ASSERT_EQUAL(result.Success().Message, statusRepository->GetObjectStatus(layerId).failed().message());
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(layerName, ((TMyPortoFailingClient*)myPorto.Get())->LastLayer_);
    UNIT_ASSERT_EQUAL(layerPlace, ((TMyPortoFailingClient*)myPorto.Get())->LastPlace_);
}

Y_UNIT_TEST(TestRemoveLayerWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> RemoveLayer(const TString& layer, const TString& place) override {
            ++Calls_;
            LastLayer_ = layer;
            LastPlace_ = place;

            return TPortoError{EPortoError::Unknown, "RemoveLayer", "NO"};
        }

        size_t Calls_ = 0;
        TString LastLayer_ = "";
        TString LastPlace_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    const TString layerId = "some_layer_id";
    const TString layerDownloadHash = "some_layer_download_hash";
    const TString layerName = "some_layer_name";
    const TString layerPlace = "/some_place";
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(layerId, layerDownloadHash));

    TPortoRemoveLayerNodePtr node = new TPortoRemoveLayerNode({1, "title"}, porto, statusRepository, layerDownloadHash, layerName, layerPlace);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "RemoveLayer");
    UNIT_ASSERT_EQUAL(1, statusRepository->GetObjectStatus(layerId).fail_counter());
    UNIT_ASSERT_EQUAL(result.Error().Message, statusRepository->GetObjectStatus(layerId).failed().message());
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(layerName, ((TMyPortoFailingClient*)myPorto.Get())->LastLayer_);
    UNIT_ASSERT_EQUAL(layerPlace, ((TMyPortoFailingClient*)myPorto.Get())->LastPlace_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoRemoveLayerNode
