#include "porto_remove_storage_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoRemoveStorageNode {

Y_UNIT_TEST_SUITE(PortoRemoveStorageNodeSuite) {

static TLogger logger({});

Y_UNIT_TEST(TestRemoveStorage) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> RemoveStorage(const TString& name, const TString& place) override {
            ++Calls_;
            LastStorage_ = name;
            LastPlace_ = place;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TString LastStorage_ = "";
        TString LastPlace_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TString storage = "some_storage";
    TString place = "some_place";

    TPortoRemoveStorageNodePtr node = new TPortoRemoveStorageNode(TBasicTreeNodeDescriptor{1, "title"}, porto, storage, place);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(storage, ((TMyPortoClient*)myPorto.Get())->LastStorage_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoClient*)myPorto.Get())->LastPlace_);
}

Y_UNIT_TEST(TestRemoveStorageWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> RemoveStorage(const TString& name, const TString& place) override {
            ++Calls_;
            LastStorage_ = name;
            LastPlace_ = place;

            return TPortoError{EPortoError::Unknown, "RemoveStorage", "NO"};
        }

        size_t Calls_ = 0;
        TString LastStorage_ = "";
        TString LastPlace_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TString storage = "some_storage";
    TString place = "some_place";

    TPortoRemoveStorageNodePtr node = new TPortoRemoveStorageNode({1, "title"}, porto, storage, place);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "RemoveStorage");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(storage, ((TMyPortoFailingClient*)myPorto.Get())->LastStorage_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoFailingClient*)myPorto.Get())->LastPlace_);
}

Y_UNIT_TEST(TestRemoveNonexistentStorage) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> RemoveStorage(const TString& name, const TString& place) override {
            ++Calls_;
            LastStorage_ = name;
            LastPlace_ = place;

            return TPortoError{EPortoError::VolumeNotFound, "", ""};
        }

        size_t Calls_ = 0;
        TString LastStorage_ = "";
        TString LastPlace_ = "";
    };

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TString storage = "some_storage";
    TString place = "some_place";

    TPortoRemoveStorageNodePtr node = new TPortoRemoveStorageNode({1, "title"}, porto, storage, place);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(storage, ((TMyPortoFailingClient*)myPorto.Get())->LastStorage_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoFailingClient*)myPorto.Get())->LastPlace_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoRemoveStorageNode
