#include "porto_set_container_private_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoSetContainerPrivateNode {

Y_UNIT_TEST_SUITE(PortoSetContainerPrivateNodeSuite) {

static TLogger logger({});

TPortoSetContainerPrivateNodePtr PortoSetPropertyer(
    TPortoClientPtr porto
    , TWorkloadStatusRepositoryPtr statusRepository
    , const TString& workloadId
    , const TPortoContainerName& containerName
    , const TString& treeHash
    , EContainerPrivate value
) {
    return new TPortoSetContainerPrivateNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , statusRepository
        , NStatusRepositoryTypes::TContainerDescription(
            workloadId
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
        )
        , containerName
        , treeHash
        , value
    );
}

Y_UNIT_TEST(TestSetProperty) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetProperty(const TPortoContainerName& name, EPortoContainerProperty property, const TString& /*value*/) override {
            ++Calls_;
            LastName_ = name;
            LastProperty_ = property;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        EPortoContainerProperty LastProperty_ = EPortoContainerProperty::State;
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString workloadId = "my_workload";
    TPortoContainerName containerName = {"some_name"};
    TString treeHash = "tree_hash";
    EContainerPrivate value = CP_READY;

    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto node = PortoSetPropertyer(porto, statusRepository, workloadId, containerName, treeHash, value);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().current().fail_reason(), "", statusRepository->GetObjectStatus(workloadId).start().current().fail_reason());
    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 0, statusRepository->GetObjectStatus(workloadId).start().system_failure_counter());

    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->LastName_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoClient*)porto.Get())->LastProperty_);
}

Y_UNIT_TEST(TestSetPropertyWithFailingSet) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetProperty(const TPortoContainerName& name, EPortoContainerProperty property, const TString& /*value*/) override {
            ++Calls_;
            LastName_ = name;
            LastProperty_ = property;

            return TPortoError{EPortoError::ContainerDoesNotExist, "SetProperty", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        EPortoContainerProperty LastProperty_ = EPortoContainerProperty::State;
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();

    const TString workloadId = "my_workload";
    TPortoContainerName containerName = {"some_name"};
    TString treeHash = "tree_hash";
    EContainerPrivate value = CP_READY;

    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto node = PortoSetPropertyer(porto, statusRepository, workloadId, containerName, treeHash, value);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "SetProperty");

    UNIT_ASSERT_EQUAL_C(
        statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
        , "ContainerDoesNotExist(4):SetProperty:NO:"
        , statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
    );
    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 1, statusRepository->GetObjectStatus(workloadId).start().system_failure_counter());
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().state(), API::EContainerState_UNKNOWN);

    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoFailingClient*)porto.Get())->LastName_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoFailingClient*)porto.Get())->LastProperty_);
}

Y_UNIT_TEST(TestSetPropertyWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetProperty(const TPortoContainerName& name, EPortoContainerProperty property, const TString& /*value*/) override {
            ++Calls_;
            LastName_ = name;
            LastProperty_ = property;

            return TPortoError{EPortoError::Unknown, "SetProperty", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        EPortoContainerProperty LastProperty_ = EPortoContainerProperty::State;
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();

    const TString workloadId = "my_workload";
    TPortoContainerName containerName = {"some_name"};
    TString treeHash = "tree_hash";
    EContainerPrivate value = CP_READY;

    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto node = PortoSetPropertyer(porto, statusRepository, workloadId, containerName, treeHash, value);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "SetProperty");

    UNIT_ASSERT_EQUAL_C(
        statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
        , "Unknown(1):SetProperty:NO:"
        , statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
    );
    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 1, statusRepository->GetObjectStatus(workloadId).start().system_failure_counter());
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().state(), API::EContainerState_UNKNOWN);

    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoFailingClient*)porto.Get())->LastName_);
    UNIT_ASSERT_EQUAL(EPortoContainerProperty::Private, ((TMyPortoFailingClient*)porto.Get())->LastProperty_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoSetContainerPrivateNode
