#include "porto_set_layer_private_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoSetLayerPrivateNode {

Y_UNIT_TEST_SUITE(PortoSetLayerPrivateNodeSuite) {

static TLogger logger({});

TPortoSetLayerPrivateNodePtr CreateNode(
    TPortoClientPtr porto
    , const TString& name
    , const TString& place
    , const TString& value
) {
    return new TPortoSetLayerPrivateNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , name
        , place
        , value
    );
}

Y_UNIT_TEST(TestSetProperty) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetLayerPrivate(const TString& privateValue, const TString& layer, const TString& place) override {
            ++Calls_;
            LastName_ = layer;
            LastPlace_ = place;
            LastValue_ = privateValue;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TString LastName_ = "";
        TString LastPlace_ = "";
        TString LastValue_ = "";
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString name = "some_name";
    const TString place = "/some_place";
    const TString value = "my_value";

    auto node = CreateNode(porto, name, place, value);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)porto.Get())->LastName_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoClient*)porto.Get())->LastPlace_);
    UNIT_ASSERT_EQUAL(value, ((TMyPortoClient*)porto.Get())->LastValue_);
}

Y_UNIT_TEST(TestSetPropertyWithFailingSet) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetLayerPrivate(const TString& privateValue, const TString& layer, const TString& place) override {
            ++Calls_;
            LastName_ = layer;
            LastPlace_ = place;
            LastValue_ = privateValue;
            return TPortoError{EPortoError::InvalidValue, "SetLayerPrivate", "NO"};
        }
        size_t Calls_ = 0;
        TString LastName_ = "";
        TString LastPlace_ = "";
        TString LastValue_ = "";
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();

    const TString name = "some_name";
    const TString place = "/some_place";
    const TString value = "my_value";

    auto node = CreateNode(porto, name, place, value);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "SetLayerPrivate");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoFailingClient*)porto.Get())->LastName_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoFailingClient*)porto.Get())->LastPlace_);
    UNIT_ASSERT_EQUAL(value, ((TMyPortoFailingClient*)porto.Get())->LastValue_);
}

Y_UNIT_TEST(TestSetPropertyWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetLayerPrivate(const TString& privateValue, const TString& layer, const TString& place) override {
            ++Calls_;
            LastName_ = layer;
            LastPlace_ = place;
            LastValue_ = privateValue;
            return TPortoError{EPortoError::Unknown, "SetLayerPrivate", "NO"};
        }
        size_t Calls_ = 0;
        TString LastName_ = "";
        TString LastPlace_ = "";
        TString LastValue_ = "";
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();

    const TString name = "some_name";
    const TString place = "/some_place";
    const TString value = "my_value";

    auto node = CreateNode(porto, name, place, value);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "SetLayerPrivate");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoFailingClient*)porto.Get())->LastName_);
    UNIT_ASSERT_EQUAL(place, ((TMyPortoFailingClient*)porto.Get())->LastPlace_);
    UNIT_ASSERT_EQUAL(value, ((TMyPortoFailingClient*)porto.Get())->LastValue_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoSetLayerPrivateNode
