#include "porto_set_properties_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoSetPropertiesNode {

Y_UNIT_TEST_SUITE(PortoSetPropertiesNodeSuite) {

static TLogger logger({});

TPortoSetPropertiesNodePtr PortoSetPropertyer(
    TPortoClientPtr porto
    , TWorkloadStatusRepositoryPtr statusRepository
    , const TString& workloadId
    , const TPortoContainerName& containerName
    , const TMap<EPortoContainerProperty, TString>& properties
) {
    return new TPortoSetPropertiesNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , statusRepository
        , NStatusRepositoryTypes::TContainerDescription(
            workloadId
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
        )
        , containerName
        , properties
    );
}

Y_UNIT_TEST(TestSetProperties) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetProperties(const TPortoContainerName& name, const TMap<EPortoContainerProperty, TString>& properties) override {
            ++Calls_;
            LastName_ = name;
            LastProperties_ = properties;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        TMap<EPortoContainerProperty, TString> LastProperties_ = {};
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString workloadId = "my_workload";
    TPortoContainerName containerName = {"some_name"};

    TMap<EPortoContainerProperty, TString> properties = {
        {EPortoContainerProperty::Command, "my_command"},
        {EPortoContainerProperty::CoreCommand, "my_core_command"}
    };

    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto node = PortoSetPropertyer(porto, statusRepository, workloadId, containerName, properties);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().current().fail_reason(), "", statusRepository->GetObjectStatus(workloadId).start().current().fail_reason());
    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 0, statusRepository->GetObjectStatus(workloadId).start().system_failure_counter());

    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->LastName_);
    UNIT_ASSERT_EQUAL(properties, ((TMyPortoClient*)porto.Get())->LastProperties_);
}

Y_UNIT_TEST(TestSetPropertiesWithFailingSet) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetProperties(const TPortoContainerName& name, const TMap<EPortoContainerProperty, TString>& properties) override {
            ++Calls_;
            LastName_ = name;
            LastProperties_ = properties;

            return TPortoError{EPortoError::ContainerDoesNotExist, "SetProperties", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        TMap<EPortoContainerProperty, TString> LastProperties_ = {};
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();

    const TString workloadId = "my_workload";
    TPortoContainerName containerName = {"some_name"};
    TMap<EPortoContainerProperty, TString> properties = {
        {EPortoContainerProperty::Command, "my_command"},
        {EPortoContainerProperty::CoreCommand, "my_core_command"}
    };

    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto node = PortoSetPropertyer(porto, statusRepository,  workloadId, containerName, properties);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "SetProperties");

    UNIT_ASSERT_EQUAL_C(
        statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
        , "ContainerDoesNotExist(4):SetProperties:NO:"
        , statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
    );
    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 1, statusRepository->GetObjectStatus(workloadId).start().system_failure_counter());
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().state(), API::EContainerState_UNKNOWN);

    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoFailingClient*)porto.Get())->LastName_);
    UNIT_ASSERT_EQUAL(properties, ((TMyPortoFailingClient*)porto.Get())->LastProperties_);
}

Y_UNIT_TEST(TestSetPropertiesWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetProperties(const TPortoContainerName& name, const TMap<EPortoContainerProperty, TString>& properties) override {
            ++Calls_;
            LastName_ = name;
            LastProperties_ = properties;

            return TPortoError{EPortoError::Unknown, "SetProperties", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        TMap<EPortoContainerProperty, TString> LastProperties_ = {};
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();

    const TString workloadId = "my_workload";
    TPortoContainerName containerName = {"some_name"};
    TMap<EPortoContainerProperty, TString> properties = {
        {EPortoContainerProperty::Command, "my_command"},
        {EPortoContainerProperty::CoreCommand, "my_core_command"}
    };

    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto node = PortoSetPropertyer(porto, statusRepository,  workloadId, containerName, properties);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "SetProperties");

    UNIT_ASSERT_EQUAL_C(
        statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
        , "Unknown(1):SetProperties:NO:"
        , statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
    );
    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 1, statusRepository->GetObjectStatus(workloadId).start().system_failure_counter());
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().state(), API::EContainerState_UNKNOWN);

    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)porto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoFailingClient*)porto.Get())->LastName_);
    UNIT_ASSERT_EQUAL(properties, ((TMyPortoFailingClient*)porto.Get())->LastProperties_);
}

Y_UNIT_TEST(TestSetPropertiesWithEmptyProperty) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> SetProperties(const TPortoContainerName& name, const TMap<EPortoContainerProperty, TString>& properties) override {
            ++Calls_;
            LastName_ = name;
            LastProperties_ = properties;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        TMap<EPortoContainerProperty, TString> LastProperties_ = {};
    };

    TPortoClientPtr porto = new TMyPortoFailingClient();

    const TString workloadId = "my_workload";
    TPortoContainerName containerName = {"some_name"};
    TMap<EPortoContainerProperty, TString> properties = {
        {EPortoContainerProperty::Command, "my_command"},
        {EPortoContainerProperty::CoreCommand, ""}
    };

    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    TPortoSetPropertiesNodePtr node;
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        node = PortoSetPropertyer(porto, statusRepository,  workloadId, containerName, properties)
        , yexception
        , "TPortoSetPropertiesNode must not have empty properties, but core_command is empty"
    );

    UNIT_ASSERT_EQUAL_C(
        statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
        , ""
        , statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
    );
    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 0, statusRepository->GetObjectStatus(workloadId).start().system_failure_counter());
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().state(), API::EContainerState_UNKNOWN);

    UNIT_ASSERT_EQUAL(0, ((TMyPortoFailingClient*)porto.Get())->Calls_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoSetPropertiesNode