#include "porto_start_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestPortoStartNode {

Y_UNIT_TEST_SUITE(PortoStartNodeSuite) {

static TLogger logger({});

TPortoStartNodePtr CreateNode(
    TPortoClientPtr porto
    , const TPortoContainerName& containerName
    , TWorkloadStatusRepositoryPtr statusRepository
    , const TString& workloadId
) {
    return new TPortoStartNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , statusRepository
        , NStatusRepositoryTypes::TContainerDescription(
            workloadId
            , NStatusRepositoryTypes::EObjectType::WORKLOAD
            , NStatusRepositoryTypes::TContainerDescription::EContainerType::START
        )
        , containerName
    );
}

Y_UNIT_TEST(TestStartOk) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> Start(const TPortoContainerName& name) override {
            Name_ = name;
            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString workloadId = "my_workload";
    TPortoContainerName containerName = {"container_0"};

    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto node = CreateNode(porto, containerName, statusRepository, workloadId);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), result.Success());

    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().current().fail_reason(), "", statusRepository->GetObjectStatus(workloadId).start().current().fail_reason());
    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 0, statusRepository->GetObjectStatus(workloadId).start().system_failure_counter());

    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->Name_);
}

Y_UNIT_TEST(TestStartError) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> Start(const TPortoContainerName& name) override {
            Name_ = name;
            return TPortoError{EPortoError::Unknown, "FAIL0", "FAIL1", "FAIL2"};
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr porto = new TMyPortoClient();

    const TString workloadId = "my_workload";
    TPortoContainerName containerName = {"container_0"};

    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto node = CreateNode(porto, containerName, statusRepository, workloadId);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL_C(
        statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
        , "Unknown(1):FAIL0:FAIL1:FAIL2"
        , statusRepository->GetObjectStatus(workloadId).start().current().fail_reason()
    );

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);

    UNIT_ASSERT_EQUAL_C(statusRepository->GetObjectStatus(workloadId).start().system_failure_counter(), 1, statusRepository->GetObjectStatus(workloadId).start().system_failure_counter());
    UNIT_ASSERT_EQUAL(statusRepository->GetObjectStatus(workloadId).start().current().state(), API::EContainerState_UNKNOWN);

    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)porto.Get())->Name_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoStartNode
