#include "porto_static_resource_verify_container_time_expired_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoStaticResourceVerifyContainerTimeExpiredNode {

Y_UNIT_TEST_SUITE(PortoStaticResourceVerifyContainerTimeExpiredNodeSuite) {

static TLogger logger({});

TPortoStaticResourceVerifyContainerTimeExpiredNodePtr CreateNode(
    TAsyncPortoClientPtr porto
    , TStaticResourceStatusRepositoryPtr statusRepository
    , const TString& staticResourceDownloadHash
    , const TPortoContainerName& containerName
    , EPortoContainerProperty initialTime
) {
    return TPortoStaticResourceVerifyContainerTimeExpiredNodePtr(new TPortoStaticResourceVerifyContainerTimeExpiredNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , statusRepository
        , staticResourceDownloadHash
        , containerName
        , initialTime
    ));
}

void AddObjectWithCheckPeriod(
    TStaticResourceStatusRepositoryPtr holder
    , const TString& id
    , const TString& hash
    , ui64 checkPeriodMs
) {
    holder->AddObject(
        TStaticResourceMeta(
            id
            , 0
            , 0
            , hash
            , checkPeriodMs
        )
    );
}

Y_UNIT_TEST(TestIncorrectInitialTimeArgument) {
    TPortoClientPtr myPorto = new TMockPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id = "my_static_resource";
    const TString hash = "my_static_resource_dowonload_hash";
    const ui64 checkPeriodMs = 60 * 1000;
    AddObjectWithCheckPeriod(holder, id, hash, checkPeriodMs);

    UNIT_ASSERT_EXCEPTION(CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::State), yexception);
}

Y_UNIT_TEST(TestStartTimeExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("10");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id = "my_static_resource";
    const TString hash = "my_static_resource_dowonload_hash";
    const ui64 checkPeriodMs = 60 * 1000;
    AddObjectWithCheckPeriod(holder, id, hash, checkPeriodMs);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::StartTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return TString("10");
            }
            return TPortoError();
        }
        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id = "my_static_resource";
    const TString hash = "my_static_resource_dowonload_hash";
    const ui64 checkPeriodMs = 60 * 1000;
    AddObjectWithCheckPeriod(holder, id, hash, checkPeriodMs);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::CreationTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestStartTimeNotExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("4000000000");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id = "my_static_resource";
    const TString hash = "my_static_resource_dowonload_hash";
    const ui64 checkPeriodMs = 60 * 1000;
    AddObjectWithCheckPeriod(holder, id, hash, checkPeriodMs);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::StartTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeNotExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return TString("4000000000");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id = "my_static_resource";
    const TString hash = "my_static_resource_dowonload_hash";
    const ui64 checkPeriodMs = 60 * 1000;
    AddObjectWithCheckPeriod(holder, id, hash, checkPeriodMs);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::CreationTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestStartTimeNowNotExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TMyPortoClient(const TString& time) : Time_(time) {}
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return Time_;
            }
            return TPortoError();
        }

        TString Time_;
        TPortoContainerName Name_ = TString("");
    };

    TInstant now = Now();
    TPortoClientPtr myPorto = new TMyPortoClient(ToString(now.Seconds()));
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id = "my_static_resource";
    const TString hash = "my_static_resource_dowonload_hash";
    const ui64 checkPeriodMs = 60 * 1000;
    AddObjectWithCheckPeriod(holder, id, hash, checkPeriodMs);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::StartTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeNowNotExpired) {
    struct TMyPortoClient : public TMockPortoClient {
        TMyPortoClient(const TString& time) : Time_(time) {}

        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return Time_;
            }
            return TPortoError();
        }

        TString Time_;
        TPortoContainerName Name_ = TString("");
    };

    TInstant now = Now();
    TPortoClientPtr myPorto = new TMyPortoClient(ToString(now.Seconds()));
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id = "my_static_resource";
    const TString hash = "my_static_resource_dowonload_hash";
    const ui64 checkPeriodMs = 60 * 1000;
    AddObjectWithCheckPeriod(holder, id, hash, checkPeriodMs);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::CreationTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestStartTimeBadSyntax) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("bad syntax");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id = "my_static_resource";
    const TString hash = "my_static_resource_dowonload_hash";
    const ui64 checkPeriodMs = 60 * 1000;
    AddObjectWithCheckPeriod(holder, id, hash, checkPeriodMs);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::StartTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "bad syntax");
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeBadSyntax) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return TString("bad syntax");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id = "my_static_resource";
    const TString hash = "my_static_resource_dowonload_hash";
    const ui64 checkPeriodMs = 60 * 1000;
    AddObjectWithCheckPeriod(holder, id, hash, checkPeriodMs);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::CreationTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "bad syntax");
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestStartTimeBadStaticResource) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("4000000000");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString hash = "my_static_resource_dowonload_hash";

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::StartTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "not found");
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeBadStaticResource) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return TString("4000000000");
            }
            return TPortoError();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString hash = "my_static_resource_dowonload_hash";

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::CreationTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "not found");
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestStartTimeUseMinPeriod) {
    struct TMyPortoClient : public TMockPortoClient {
        TMyPortoClient(const TString& time) : Time_(time) {}

        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::StartTimeRaw) {
                return Time_;
            }
            return TPortoError();
        }

        TString Time_;
        TPortoContainerName Name_ = TString("");
    };

    const ui64 checkPeriodMs1 = 6000 * 1000;
    const ui64 checkPeriodMs2 = 5 * 1000;
    UNIT_ASSERT(4 * checkPeriodMs2 < checkPeriodMs1); // important for test

    TInstant now = Now() - TDuration::MilliSeconds(2 * checkPeriodMs2);
    TPortoClientPtr myPorto = new TMyPortoClient(ToString(now.Seconds()));
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id1 = "my_static_resource1";
    const TString id2 = "my_static_resource2";
    const TString hash = "my_static_resource_dowonload_hash";
    AddObjectWithCheckPeriod(holder, id1, hash, checkPeriodMs1);
    AddObjectWithCheckPeriod(holder, id2, hash, checkPeriodMs2);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::StartTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

Y_UNIT_TEST(TestCreationTimeUseMinPeriod) {
    struct TMyPortoClient : public TMockPortoClient {
        TMyPortoClient(const TString& time) : Time_(time) {}

        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int /*flags*/) override {
            Name_ = name;

            if (property == EPortoContainerProperty::CreationTimeRaw) {
                return Time_;
            }
            return TPortoError();
        }

        TString Time_;
        TPortoContainerName Name_ = TString("");
    };

    const ui64 checkPeriodMs1 = 6000 * 1000;
    const ui64 checkPeriodMs2 = 5 * 1000;
    UNIT_ASSERT(4 * checkPeriodMs2 < checkPeriodMs1); // important for test

    TInstant now = Now() - TDuration::MilliSeconds(2 * checkPeriodMs2);
    TPortoClientPtr myPorto = new TMyPortoClient(ToString(now.Seconds()));
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    TPortoContainerName containerName = {"container_0"};
    const TString id1 = "my_static_resource1";
    const TString id2 = "my_static_resource2";
    const TString hash = "my_static_resource_dowonload_hash";
    AddObjectWithCheckPeriod(holder, id1, hash, checkPeriodMs1);
    AddObjectWithCheckPeriod(holder, id2, hash, checkPeriodMs2);

    auto node = CreateNode(porto, holder, hash, containerName, EPortoContainerProperty::CreationTimeRaw);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->Name_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoStaticResourceVerifyContainerTimeExpiredNode
