#include "porto_stop_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestPortoStopNode {

Y_UNIT_TEST_SUITE(PortoStopNodeSuite) {

static TLogger logger({});

TPortoStopNodePtr CreateNode(TPortoClientPtr porto, const TPortoContainerName& name) {
    return new TPortoStopNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , new TAsyncPortoClient(porto, new TFakeThreadPool())
        , name
    );
}

Y_UNIT_TEST(TestStopOk) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> Stop(const TPortoContainerName& name, TDuration /*timeout*/) override {
            Name_ = name;
            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr porto = new TMyPortoClient();
    TPortoContainerName name = {"container_0"};

    auto node = CreateNode(porto, name);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), result.Success());
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)porto.Get())->Name_);
}

Y_UNIT_TEST(TestStopFailure) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> Stop(const TPortoContainerName& name, TDuration /*timeout*/) override {
            Name_ = name;
            return TPortoError{EPortoError::ContainerDoesNotExist, "Stop", "NO"};
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr porto = new TMyPortoClient();
    TPortoContainerName name = {"container_0"};

    auto node = CreateNode(porto, name);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)porto.Get())->Name_);
}

Y_UNIT_TEST(TestStopError) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> Stop(const TPortoContainerName& name, TDuration /*timeout*/) override {
            Name_ = name;
            return TPortoError{EPortoError::Unknown, "Stop", "NO"};
        }

        TPortoContainerName Name_ = TString("");
    };

    TPortoClientPtr porto = new TMyPortoClient();
    TPortoContainerName name = {"container_0"};

    auto node = CreateNode(porto, name);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "Stop");
    UNIT_ASSERT_EQUAL(name, ((TMyPortoClient*)porto.Get())->Name_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoStopNode
