#include "porto_unlink_volume_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/volume_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoUnlinkVolumeNode {

static TLogger logger({});

Y_UNIT_TEST_SUITE(PortoUnlinkVolumeNodeSuite) {

Y_UNIT_TEST(TestUnlinkVolume) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> UnlinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool strict) override {
            ++Calls_;
            Path_ = path;
            Container_ = container;
            Target_ = target;
            Strict_ = strict;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TString Path_ = "";
        TPortoContainerName Container_ = {""};
        TString Target_ = "";
        bool Strict_ = false;
    };

    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TString path = "/some_path";
    TPortoContainerName container("some_container");
    TString target = "some_target";
    bool strict = false;

    TPortoUnlinkVolumeNodePtr node = new TPortoUnlinkVolumeNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , holder
        , id
        , path
        , container
        , target
        , strict
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C((bool)result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoClient*)myPorto.Get())->Path_);
    UNIT_ASSERT_EQUAL(container, ((TMyPortoClient*)myPorto.Get())->Container_);
    UNIT_ASSERT_EQUAL(target, ((TMyPortoClient*)myPorto.Get())->Target_);
    UNIT_ASSERT_EQUAL(strict, ((TMyPortoClient*)myPorto.Get())->Strict_);
}

Y_UNIT_TEST(TestUnlinkVolumeWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> UnlinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool strict) override {
            ++Calls_;
            Path_ = path;
            Container_ = container;
            Target_ = target;
            Strict_ = strict;

            return TPortoError{EPortoError::Unknown, "UnlinkVolume", "NO"};
        }

        size_t Calls_ = 0;
        TString Path_ = "";
        TPortoContainerName Container_ = {""};
        TString Target_ = "";
        bool Strict_ = false;
    };

    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TString path = "/some_path";
    TPortoContainerName container("some_container");
    TString target = "some_target";
    bool strict = false;

    TPortoUnlinkVolumeNodePtr node = new TPortoUnlinkVolumeNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , holder
        , id
        , path
        , container
        , target
        , strict
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "UnlinkVolume");
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).failed().message(), result.Error().Message);
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), 1);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoFailingClient*)myPorto.Get())->Path_);
    UNIT_ASSERT_EQUAL(container, ((TMyPortoFailingClient*)myPorto.Get())->Container_);
    UNIT_ASSERT_EQUAL(target, ((TMyPortoFailingClient*)myPorto.Get())->Target_);
    UNIT_ASSERT_EQUAL(strict, ((TMyPortoFailingClient*)myPorto.Get())->Strict_);
}

Y_UNIT_TEST(TestUnlinkVolumeWithFailingUnlink) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> UnlinkVolume(const TString& path, const TPortoContainerName& container, const TString& target, bool strict) override {
            ++Calls_;
            Path_ = path;
            Container_ = container;
            Target_ = target;
            Strict_ = strict;

            return TPortoError{EPortoError::Busy, "UnlinkVolume", "NO"};
        }

        size_t Calls_ = 0;
        TString Path_ = "";
        TPortoContainerName Container_ = {""};
        TString Target_ = "";
        bool Strict_ = false;
    };

    TVolumeStatusRepositoryPtr holder = new TVolumeStatusRepository();
    const TString id = "my_volume";
    holder->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(id));

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TString path = "/some_path";
    TPortoContainerName container("some_container");
    TString target = "some_target";
    bool strict = false;

    TPortoUnlinkVolumeNodePtr node = new TPortoUnlinkVolumeNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , holder
        , id
        , path
        , container
        , target
        , strict
    );
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "UnlinkVolume");
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).failed().message(), result.Success().Message);
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), 1);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoFailingClient*)myPorto.Get())->Path_);
    UNIT_ASSERT_EQUAL(container, ((TMyPortoFailingClient*)myPorto.Get())->Container_);
    UNIT_ASSERT_EQUAL(target, ((TMyPortoFailingClient*)myPorto.Get())->Target_);
    UNIT_ASSERT_EQUAL(strict, ((TMyPortoFailingClient*)myPorto.Get())->Strict_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoUnlinkVolumeNode
