#include "porto_volume_exists_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoVolumeExistsNode {

Y_UNIT_TEST_SUITE(PortoVolumeExistsNodeSuite) {

static TLogger logger({});

Y_UNIT_TEST(TestVolumeExists) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<int, TPortoError> IsVolumeExists(const TString& path) override {
            ++Calls_;
            Path_ = path;

            return 1;
        }

        TString Path_;
        size_t Calls_ = 0;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TString path = "/some_path";

    TPortoVolumeExistsNodePtr node = new TPortoVolumeExistsNode(TBasicTreeNodeDescriptor{1, "title"}, porto, path);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(path, ((TMyPortoClient*)myPorto.Get())->Path_);
}

Y_UNIT_TEST(TestVolumeNotExists) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<int, TPortoError> IsVolumeExists(const TString& /*path*/) override {
            ++Calls_;
            return 0;
        }

        size_t Calls_ = 0;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TString path = "/some_path";

    TPortoVolumeExistsNodePtr node = new TPortoVolumeExistsNode({1, "title"}, porto, path);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
}

Y_UNIT_TEST(TestVolumeExistsWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<int, TPortoError> IsVolumeExists(const TString& /*path*/) override {
            ++Calls_;
            return TPortoError{EPortoError::Unknown, "ListVolumes", "NO"};
        }

        size_t Calls_ = 0;
    };

    TPortoClientPtr myPorto = new TMyPortoFailingClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());

    TString path = "/some_path";

    TPortoVolumeExistsNodePtr node = new TPortoVolumeExistsNode({1, "title"}, porto, path);
    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "ListVolumes");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoFailingClient*)myPorto.Get())->Calls_);
}

}

} // namespace NInfra::NPodAgent::NTestPortoVolumeExistsNode
