#include "porto_volume_is_link_of_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>


namespace NInfra::NPodAgent::NTestPortoVolumeIsLinkOfNode  {

static TLogger logger({});

TPortoVolumeIsLinkOfNodePtr GetPortoVolumeIsLinkOfNode(
    TAsyncPortoClientPtr porto
    , const TString& path
    , const TPortoContainerName& container
    , const TString& target
    , bool readOnly
    , bool required
) {
    return new TPortoVolumeIsLinkOfNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , path
        , container
        , target
        , readOnly
        , required
    );
}

class ITestPortoVolumeIsLinkOfNodeCanon {
public:
    ITestPortoVolumeIsLinkOfNodeCanon(TPortoClientPtr porto)
        : MyPorto_(porto)
        , Porto_(new TAsyncPortoClient(porto, new TFakeThreadPool()))
    {}

    virtual ~ITestPortoVolumeIsLinkOfNodeCanon() {
    }

    void DoTest() {
        Test();
    }

protected:
    virtual void Test() = 0;

protected:
    TPortoClientPtr MyPorto_;
    TAsyncPortoClientPtr Porto_;
};

Y_UNIT_TEST_SUITE(PortoVolumeIsLinkOfNodeSuite) {

Y_UNIT_TEST(TestNoVolume) {
    class TMyPortoClient : public TMockPortoClient {
    public:
        TMyPortoClient() : Calls_(0)
        {}

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& /*path*/, const TPortoContainerName& /*container*/) override {
            ++Calls_;
            return TVector<TPortoVolume>{TPortoVolume{}};
        }

        size_t GetCalls() {
            return Calls_;
        }

    private:
        size_t Calls_;
    };

    class TTest : public ITestPortoVolumeIsLinkOfNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoVolumeIsLinkOfNodeCanon(porto)
        {}

    protected:
        void Test() override {
            TPortoVolumeIsLinkOfNodePtr node = GetPortoVolumeIsLinkOfNode(Porto_, "path", TPortoContainerName("container"), "target", false, false);
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "Path volume not found");
            UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)MyPorto_.Get())->GetCalls());
        }
    };

    TTest test(new TMyPortoClient());
    test.DoTest();
}

Y_UNIT_TEST(TestListVolumeError) {
    class TMyPortoClient : public TMockPortoClient {
    public:
        TMyPortoClient() : Calls_(0)
        {}

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& /*path*/, const TPortoContainerName& /*container*/) override {
            ++Calls_;
            return TPortoError{EPortoError::Unknown, "Some error", "NO"};
        }

        size_t GetCalls() {
            return Calls_;
        }

    private:
        size_t Calls_;
    };

    class TTest : public ITestPortoVolumeIsLinkOfNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoVolumeIsLinkOfNodeCanon(porto)
        {}

    protected:
        void Test() override {
            TPortoVolumeIsLinkOfNodePtr node = GetPortoVolumeIsLinkOfNode(Porto_, "path", TPortoContainerName("container"), "target", false, false);
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "Some error");
            UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)MyPorto_.Get())->GetCalls());
        }
    };

    TTest test(new TMyPortoClient());
    test.DoTest();
}

Y_UNIT_TEST(TestBadTarget) {
    class TMyPortoClient : public TMockPortoClient {
    public:
        TMyPortoClient() : Calls_(0)
        {}

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& /*path*/, const TPortoContainerName& /*container*/) override {
            ++Calls_;

            TPortoVolume volume;
            volume.set_path("my_path");

            return TVector<TPortoVolume>{volume};
        }

        size_t GetCalls() {
            return Calls_;
        }

    private:
        size_t Calls_;
    };

    class TTest : public ITestPortoVolumeIsLinkOfNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoVolumeIsLinkOfNodeCanon(porto)
        {}

    protected:
        void Test() override {
            TPortoVolumeIsLinkOfNodePtr node = GetPortoVolumeIsLinkOfNode(Porto_, "my_path", TPortoContainerName("container"), "my_target", false, false);
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
            UNIT_ASSERT_EQUAL("Target volume 'my_target' not linked to path volume 'my_path'", result.Success().Message);
            UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)MyPorto_.Get())->GetCalls());
        }
    };

    TTest test(new TMyPortoClient());
    test.DoTest();
}

Y_UNIT_TEST(TestBadContainer) {
    class TMyPortoClient : public TMockPortoClient {
    public:
        TMyPortoClient() : Calls_(0)
        {}

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& /*path*/, const TPortoContainerName& /*container*/) override {
            ++Calls_;

            TPortoVolume volume;
            volume.set_path("my_path");
            TPortoVolumeLink *new_link = volume.mutable_links()->Add();
            new_link->set_container("bad_container");
            new_link->set_target("my_target");
            new_link->set_read_only(false);
            new_link->set_required(false);

            return TVector<TPortoVolume>{volume};
        }

        size_t GetCalls() {
            return Calls_;
        }

    private:
        size_t Calls_;
    };

    class TTest : public ITestPortoVolumeIsLinkOfNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoVolumeIsLinkOfNodeCanon(porto)
        {}

    protected:
        void Test() override {
            TPortoVolumeIsLinkOfNodePtr node = GetPortoVolumeIsLinkOfNode(Porto_, "my_path", TPortoContainerName("my_container"), "my_target", false, false);
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
            UNIT_ASSERT_EQUAL("Target volume 'my_target' linked to path volume 'my_path' but with wrong container", result.Success().Message);
            UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)MyPorto_.Get())->GetCalls());
        }
    };

    TTest test(new TMyPortoClient());
    test.DoTest();
}

Y_UNIT_TEST(TestBadReadOnly) {
    class TMyPortoClient : public TMockPortoClient {
    public:
        TMyPortoClient() : Calls_(0)
        {}

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& /*path*/, const TPortoContainerName& /*container*/) override {
            ++Calls_;

            TPortoVolume volume;
            volume.set_path("my_path");
            TPortoVolumeLink *new_link = volume.mutable_links()->Add();
            new_link->set_container("my_container");
            new_link->set_target("my_target");
            new_link->set_read_only(true);
            new_link->set_required(false);

            return TVector<TPortoVolume>{volume};
        }

        size_t GetCalls() {
            return Calls_;
        }

    private:
        size_t Calls_;
    };

    class TTest : public ITestPortoVolumeIsLinkOfNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoVolumeIsLinkOfNodeCanon(porto)
        {}

    protected:
        void Test() override {
            TPortoVolumeIsLinkOfNodePtr node = GetPortoVolumeIsLinkOfNode(Porto_, "my_path", TPortoContainerName("my_container"), "my_target", false, false);
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
            UNIT_ASSERT_EQUAL("Target volume 'my_target' linked to path volume 'my_path' but with wrong read mode", result.Success().Message);
            UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)MyPorto_.Get())->GetCalls());
        }
    };

    TTest test(new TMyPortoClient());
    test.DoTest();
}

Y_UNIT_TEST(TestBadRequired) {
    class TMyPortoClient : public TMockPortoClient {
    public:
        TMyPortoClient() : Calls_(0)
        {}

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& /*path*/, const TPortoContainerName& /*container*/) override {
            ++Calls_;

            TPortoVolume volume;
            volume.set_path("my_path");
            TPortoVolumeLink *new_link = volume.mutable_links()->Add();
            new_link->set_container("my_container");
            new_link->set_target("my_target");
            new_link->set_read_only(false);
            new_link->set_required(true);

            return TVector<TPortoVolume>{volume};
        }

        size_t GetCalls() {
            return Calls_;
        }

    private:
        size_t Calls_;
    };

    class TTest : public ITestPortoVolumeIsLinkOfNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoVolumeIsLinkOfNodeCanon(porto)
        {}

    protected:
        void Test() override {
            TPortoVolumeIsLinkOfNodePtr node = GetPortoVolumeIsLinkOfNode(Porto_, "my_path", TPortoContainerName("my_container"), "my_target", false, false);
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
            UNIT_ASSERT_EQUAL("Target volume 'my_target' linked to path volume 'my_path' but with wrong require mode", result.Success().Message);
            UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)MyPorto_.Get())->GetCalls());
        }
    };

    TTest test(new TMyPortoClient());
    test.DoTest();
}

Y_UNIT_TEST(TestSuccess) {
    class TMyPortoClient : public TMockPortoClient {
    public:
        TMyPortoClient() : Calls_(0)
        {}

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& /*path*/, const TPortoContainerName& /*container*/) override {
            ++Calls_;

            TPortoVolume volume;
            volume.set_path("my_path");
            TPortoVolumeLink *new_link = volume.mutable_links()->Add();
            new_link->set_container("my_container");
            new_link->set_target("my_target");
            new_link->set_read_only(false);
            new_link->set_required(false);

            return TVector<TPortoVolume>{volume};
        }

        size_t GetCalls() {
            return Calls_;
        }

    private:
        size_t Calls_;
    };

    class TTest : public ITestPortoVolumeIsLinkOfNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoVolumeIsLinkOfNodeCanon(porto)
        {}

    protected:
        void Test() override {
            TPortoVolumeIsLinkOfNodePtr node = GetPortoVolumeIsLinkOfNode(Porto_, "my_path", TPortoContainerName("my_container"), "my_target", false, false);
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
            UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)MyPorto_.Get())->GetCalls());
        }
    };

    TTest test(new TMyPortoClient());
    test.DoTest();
}

Y_UNIT_TEST(TestSuccessIgnoreContainerName) {
    class TMyPortoClient : public TMockPortoClient {
    public:
        TMyPortoClient() : Calls_(0)
        {}

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& /*path*/, const TPortoContainerName& /*container*/) override {
            ++Calls_;

            TPortoVolume volume;
            volume.set_path("my_path");
            TPortoVolumeLink *new_link = volume.mutable_links()->Add();
            new_link->set_container("bad_container");
            new_link->set_target("my_target");
            new_link->set_read_only(false);
            new_link->set_required(false);

            return TVector<TPortoVolume>{volume};
        }

        size_t GetCalls() {
            return Calls_;
        }

    private:
        size_t Calls_;
    };

    class TTest : public ITestPortoVolumeIsLinkOfNodeCanon {
    public:
        TTest(TPortoClientPtr porto) : ITestPortoVolumeIsLinkOfNodeCanon(porto)
        {}

    protected:
        void Test() override {
            TPortoVolumeIsLinkOfNodePtr node = GetPortoVolumeIsLinkOfNode(Porto_, "my_path", TPortoContainerName("IGNORE_CONTAINER_NAME"), "my_target", false, false);
            auto result = node->Tick(MockTickContext(logger));

            UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
            UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)MyPorto_.Get())->GetCalls());
        }
    };

    TTest test(new TMyPortoClient());
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTestPortoVolumeIsLinkOfNode
