#include "check_same_inode_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NCheckSameINodeNodeTest {

static TLogger logger({});

void CreateFile(const TString& path) {
    TOFStream flag(path);
    flag.Flush();
    flag.Finish();
}

void EnsureDoesNotExist(const TString& path) {
    if (NFs::Exists(path)) {
        try {
            NFs::RemoveRecursive(path);
        } catch(...) {
            NFs::Remove(path);
        }
    }
}

Y_UNIT_TEST_SUITE(CheckSaneINodeNodeSuite) {

Y_UNIT_TEST(CheckSameINodeSuccess) {
    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TPosixWorkerPtr posixWorker = new TPosixWorker(pool);

    const TString file = "file";
    const TString fileHardLink = "file_hardlink";
    EnsureDoesNotExist(file);
    EnsureDoesNotExist(fileHardLink);
    CreateFile(file);
    UNIT_ASSERT(NFs::HardLink(file, fileHardLink));

    TCheckSameINodeNodePtr node = new TCheckSameINodeNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , file
        , fileHardLink
    );

    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C((bool)result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
}

Y_UNIT_TEST(CheckSameINodeFailure) {
    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TPosixWorkerPtr posixWorker = new TPosixWorker(pool);

    const TString file = "file";
    const TString fileOther = "file_other";
    EnsureDoesNotExist(file);
    EnsureDoesNotExist(fileOther);
    CreateFile(file);
    CreateFile(fileOther);

    TCheckSameINodeNodePtr node = new TCheckSameINodeNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , file
        , fileOther
    );

    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C((bool)result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
}

Y_UNIT_TEST(CheckSameINodeError) {
    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TPosixWorkerPtr posixWorker = new TPosixWorker(pool);

    const TString nonExistentFile = "non_existent_file";
    EnsureDoesNotExist(nonExistentFile);

    TCheckSameINodeNodePtr node = new TCheckSameINodeNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , nonExistentFile
        , nonExistentFile
    );

    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT(!(bool)result);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "No such file or directory");
}

}

} // namespace NInfra::NPodAgent::NCheckSameINodeNodeTest

