#include "file_modified_before_porto_container_start_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestFileModifiedBeforePortoContainerStartNode {

Y_UNIT_TEST_SUITE(FileModifiedBeforePortoContainerStartNodeSuite) {

static TLogger logger({});

TFileModifiedBeforePortoContainerStartNodePtr CreateNode(TAsyncPortoClientPtr porto, TPosixWorkerPtr posixWorker, const TPortoContainerName& containerName, const TString& filePath) {
    return TFileModifiedBeforePortoContainerStartNodePtr(new TFileModifiedBeforePortoContainerStartNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , posixWorker
        , containerName
        , filePath
    ));
}

Y_UNIT_TEST(TestFileModifiedBeforeContainerCreation){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls;
            LastPath = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(400000);

            return result;
        }

        size_t Calls = 0;
        TString LastPath = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int ) override {
            ContainerName = name;
            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("500000");
            }
            return TPortoError();
        }

        TPortoContainerName ContainerName = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TPortoContainerName containerName = {"container_0"};
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, containerName, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->ContainerName);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->LastPath);
}

Y_UNIT_TEST(TestFileModifiedAfterContainerCreation){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls;
            LastPath = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(500000);

            return result;
        }

        size_t Calls = 0;
        TString LastPath = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int ) override {
            ContainerName = name;
            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("400000");
            }
            return TPortoError();
        }

        TPortoContainerName ContainerName = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TPortoContainerName containerName = {"container_0"};
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, containerName, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "modified after container start");
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->ContainerName);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->LastPath);
}

Y_UNIT_TEST(TestPortoReturnsBadSyntax){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls;
            LastPath = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(500000);

            return result;
        }

        size_t Calls = 0;
        TString LastPath = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int ) override {
            ContainerName = name;
            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("bad syntax");
            }
            return TPortoError();
        }

        TPortoContainerName ContainerName = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TPortoContainerName containerName = {"container_0"};
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, containerName, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "bad syntax");
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->ContainerName);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->LastPath);
}

Y_UNIT_TEST(TestPortoReturnsError){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls;
            LastPath = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(500000);

            return result;
        }

        size_t Calls = 0;
        TString LastPath = "";
    };

    struct TMyFailingPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty, int ) override {
            ContainerName = name;
            return TPortoError{EPortoError::InvalidValue, "", "", "porto error"};
        }

        TPortoContainerName ContainerName = TString("");
    };

    TPortoClientPtr myPorto = new TMyFailingPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TPortoContainerName containerName = {"container_0"};
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, containerName, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "porto error");
    UNIT_ASSERT_EQUAL(containerName, ((TMyFailingPortoClient*)myPorto.Get())->ContainerName);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->LastPath);
}

Y_UNIT_TEST(TestPosixWorkerReturnsError){
    struct TMyFailingPosixClient : public TPosixWorker {
        TMyFailingPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls;
            LastPath = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(TPosixError{-1, "posix error"});

            return result;
        }

        size_t Calls = 0;
        TString LastPath = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError> GetProperty(const TPortoContainerName& name, EPortoContainerProperty property, int ) override {
            ContainerName = name;
            if (property == EPortoContainerProperty::StartTimeRaw) {
                return TString("500000");
            }
            return TPortoError();
        }

        TPortoContainerName ContainerName = TString("");
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TPortoContainerName containerName = {"container_0"};
    TPosixWorkerPtr posixWorker = new TMyFailingPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, containerName, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "posix error");
    UNIT_ASSERT_EQUAL(containerName, ((TMyPortoClient*)myPorto.Get())->ContainerName);
    UNIT_ASSERT_EQUAL(1, ((TMyFailingPosixClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(filePath, ((TMyFailingPosixClient*)posixWorker.Get())->LastPath);
}

}

} // namespace NInfra::NPodAgent::NTestFileModifiedBeforePortoContainerStartNode
