#include "file_modified_before_porto_volume_build_node.h"

namespace NInfra::NPodAgent {

ENodeType TFileModifiedBeforePortoVolumeBuildNode::GetType() const {
    return NODE_TYPE;
}

TExpected<TFuture<TExpected<TCompareTimesResult, TModifiedTimeError>>, TTickResult> TFileModifiedBeforePortoVolumeBuildNode::PortoCall(TTickContextPtr /*context*/) {
    return GetFileModificationTime().Apply([this](const auto& fileModificationTimeResult) {
        return GetVolumeBuildTime().Apply([this, fileModificationTimeResult](const auto& volumeBuildTimeResult){
            return CompareTimes(volumeBuildTimeResult.GetValue(), fileModificationTimeResult.GetValue());
        });
    });
}

TExpected<TCompareTimesResult, TModifiedTimeError> TFileModifiedBeforePortoVolumeBuildNode::CompareTimes(
    const TExpected<time_t, TPortoError>& volumeBuildTimeResult
    , const TExpected<time_t, TPosixError>& fileModificationTimeResult
) {
    if (!(bool)(volumeBuildTimeResult))
        return TModifiedTimeError(volumeBuildTimeResult.Error());

    if (!(bool)(fileModificationTimeResult))
        return TModifiedTimeError(fileModificationTimeResult.Error());

    time_t volumeBuildTime = volumeBuildTimeResult.Success();
    time_t fileModificationTime = fileModificationTimeResult.Success();
    if (fileModificationTime <= volumeBuildTime)
        return TCompareTimesResult { true };
    else
        return TCompareTimesResult { false, TStringBuilder() << "volume build time: " << volumeBuildTime << ", file modification time: " << fileModificationTime };
}

TTickResult TFileModifiedBeforePortoVolumeBuildNode::ProcessPortoResult(TTickContextPtr /*context*/, TExpected<TCompareTimesResult, TModifiedTimeError>& result) {
    if (!result) {
        auto error = result.Error();
        if (std::holds_alternative<TPosixError>(error))
            return TNodeError{ToString(std::get<TPosixError>(error))};

        if (std::holds_alternative<TPortoError>(error))
            return TNodeError{ToString(std::get<TPortoError>(error))};

        return TNodeError{"unknown error"};
    }

    return result.Success().Status
           ? TNodeSuccess(ENodeStatus::SUCCESS)
           : TNodeSuccess(ENodeStatus::FAILURE, TStringBuilder() << "file modified after volume build: " << FilePath_ << ", " << result.Success().Message);
}

TFuture<TExpected<time_t, TPortoError>> TFileModifiedBeforePortoVolumeBuildNode::GetVolumeBuildTime() {
    return Porto_->ListVolumes(VolumePath_).Apply([this](const auto& listVolumesResult) {

        const auto& result = listVolumesResult.GetValue();
        if (result) {
            if (result.Success().size() != 1) {
                const auto error = TPortoError{EPortoError::InvalidValue, "", "", TStringBuilder() << "only 1 volume '" << VolumePath_ << "' expected, got " << ToString(result.Success().size())};
                return TExpected<time_t, TPortoError>(error);
            }
                
            const TPortoVolume& volume = result.Success().front();
            if (volume.path() != VolumePath_) {
                const auto error = TPortoError{EPortoError::InvalidValue, "", "", TStringBuilder() << "porto ListVolumes call doesn't contain '" << VolumePath_ << "' volume"};
                return TExpected<time_t, TPortoError>(error);
            }

            return TExpected<time_t, TPortoError>(static_cast<time_t>(volume.build_time()));
        }
        return TExpected<time_t, TPortoError>(result.Error());
    });
}

NThreading::TFuture<TExpected<time_t, TPosixError>> TFileModifiedBeforePortoVolumeBuildNode::GetFileModificationTime() {
    return Posix_->GetFileModificationTimeAsync(FilePath_);
}

}
