#include "file_modified_before_porto_volume_build_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestFileModifiedBeforePortoContainerStartNode {

Y_UNIT_TEST_SUITE(FileModifiedBeforePortoVolumeBuildNodeSuite) {

static TLogger logger({});

TFileModifiedBeforePortoVolumeBuildNodePtr CreateNode(TAsyncPortoClientPtr porto, TPosixWorkerPtr posixWorker, const TString& volumePath, const TString& filePath) {
    return TFileModifiedBeforePortoVolumeBuildNodePtr(new TFileModifiedBeforePortoVolumeBuildNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , porto
        , posixWorker
        , volumePath
        , filePath
    ));
}

Y_UNIT_TEST(TestFileModifiedBeforeVolumeBuild){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls_;
            FilePath_ = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(400000);

            return result;
        }

        size_t Calls_ = 0;
        TString FilePath_ = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName&) override {
            ++Calls_;
            VolumePath_ = path;

            TVector<TPortoVolume> result(1);
            result[0].set_path(path);
            result[0].set_build_time(500000);

            return result;
        }

        size_t Calls_ = 0;
        TString VolumePath_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TString volumePath = "volume_path";
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, volumePath, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(volumePath, ((TMyPortoClient*)myPorto.Get())->VolumePath_);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls_);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->FilePath_);
}

Y_UNIT_TEST(TestFileModifiedAfterVolumeBuild){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls_;
            FilePath_ = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(600000);

            return result;
        }

        size_t Calls_ = 0;
        TString FilePath_ = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName&) override {
            ++Calls_;
            VolumePath_ = path;

            TVector<TPortoVolume> result(1);
            result[0].set_path(path);
            result[0].set_build_time(500000);

            return result;
        }

        size_t Calls_ = 0;
        TString VolumePath_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TString volumePath = "volume_path";
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, volumePath, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(volumePath, ((TMyPortoClient*)myPorto.Get())->VolumePath_);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls_);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->FilePath_);
}


Y_UNIT_TEST(TestListVolumesReturnsSeveralVolumes){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls_;
            FilePath_ = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(400000);

            return result;
        }

        size_t Calls_ = 0;
        TString FilePath_ = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName&) override {
            ++Calls_;
            VolumePath_ = path;

            TVector<TPortoVolume> result(2);
            result[0].set_path(path);
            result[0].set_build_time(500000);

            result[1].set_path(path);
            result[1].set_build_time(500000);

            return result;
        }

        size_t Calls_ = 0;
        TString VolumePath_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TString volumePath = "volume_path";
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, volumePath, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, TStringBuilder() << "only 1 volume '" << volumePath << "' expected");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(volumePath, ((TMyPortoClient*)myPorto.Get())->VolumePath_);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls_);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->FilePath_);
}

Y_UNIT_TEST(TestListVolumesReturnslVolumeWithInvalidPath){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls_;
            FilePath_ = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(400000);

            return result;
        }

        size_t Calls_ = 0;
        TString FilePath_ = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString&, const TPortoContainerName&) override {
            ++Calls_;

            TVector<TPortoVolume> result(1);
            result[0].set_path("invalid_volume_path");
            result[0].set_build_time(500000);

            return result;
        }

        size_t Calls_ = 0;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TString volumePath = "volume_path";
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, volumePath, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, TStringBuilder() << "porto ListVolumes call doesn't contain '" << volumePath << "' volume");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls_);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->FilePath_);
}

Y_UNIT_TEST(TestListVolumesReturnsError){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls_;
            FilePath_ = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(400000);

            return result;
        }

        size_t Calls_ = 0;
        TString FilePath_ = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName&) override {
            ++Calls_;
            VolumePath_ = path;

            return TPortoError{EPortoError::InvalidValue, "", "", "porto error"};
        }

        size_t Calls_ = 0;
        TString VolumePath_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TString volumePath = "volume_path";
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, volumePath, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "porto error");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(volumePath, ((TMyPortoClient*)myPorto.Get())->VolumePath_);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls_);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->FilePath_);
}

Y_UNIT_TEST(TestPosixWorkerReturnsError){
    struct TMyPosixClient : public TPosixWorker {
        TMyPosixClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixTimeResult> GetFileModificationTimeAsync(const TString& path) override {
            ++Calls_;
            FilePath_ = path;

            auto result = NThreading::NewPromise<TPosixTimeResult>();
            result.SetValue(TPosixError{-1, "posix error"});

            return result;
        }

        size_t Calls_ = 0;
        TString FilePath_ = "";
    };

    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName&) override {
            ++Calls_;
            VolumePath_ = path;

            TVector<TPortoVolume> result(1);
            result[0].set_path(path);
            result[0].set_build_time(500000);

            return result;
        }

        size_t Calls_ = 0;
        TString VolumePath_;
    };

    TPortoClientPtr myPorto = new TMyPortoClient();
    TAsyncPortoClientPtr porto = new TAsyncPortoClient(myPorto, new TFakeThreadPool());
    TString volumePath = "volume_path";
    TPosixWorkerPtr posixWorker = new TMyPosixClient();
    TString filePath = "file_path";

    auto node = CreateNode(porto, posixWorker, volumePath, filePath);
    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C(!result, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "posix error");
    UNIT_ASSERT_EQUAL(1, ((TMyPortoClient*)myPorto.Get())->Calls_);
    UNIT_ASSERT_EQUAL(volumePath, ((TMyPortoClient*)myPorto.Get())->VolumePath_);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixClient*)posixWorker.Get())->Calls_);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixClient*)posixWorker.Get())->FilePath_);
}

}

} // namespace NInfra::NPodAgent::NTestFileModifiedBeforePortoContainerStartNode
