#include "is_directory_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTestIsDirectoryNode {

static TLogger logger({});

void CreateFile(const TString& path) {
    TOFStream flag(path);
    flag.Flush();
    flag.Finish();
}

void CreateDirectory(const TString& path) {
    TOFStream flag(path);
    flag.Flush();
    flag.Finish();
}

TIsDirectoryNodePtr MakeIsDirectoryNode(const TString& path, TPosixWorkerPtr posixWorker) {
    return new TIsDirectoryNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , path
        , posixWorker
    );
}

Y_UNIT_TEST_SUITE(IsDirectoryNodeSuite) {

Y_UNIT_TEST(IsDirectorySuccess) {
    const TString dirPath = "dir";
    NFs::MakeDirectory(dirPath);
    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TPosixWorkerPtr posixWorker = new TPosixWorker(pool);

    TIsDirectoryNodePtr node = MakeIsDirectoryNode(dirPath, posixWorker);

    auto tickResult = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_C((bool)tickResult, tickResult.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, tickResult.Success().Status);
}

Y_UNIT_TEST(IsDirectoryFailure) {
    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TPosixWorkerPtr posixWorker = new TPosixWorker(pool);

    TIsDirectoryNodePtr node = MakeIsDirectoryNode("unknown", posixWorker);
    auto tickResult = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, tickResult.Success().Status);

    TString filePath = "file";
    CreateFile(filePath);

    node = MakeIsDirectoryNode(filePath, posixWorker);
    tickResult = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, tickResult.Success().Status);

    TString linkPath = "link";
    UNIT_ASSERT(NFs::SymLink(filePath, linkPath));

    node = MakeIsDirectoryNode(linkPath, posixWorker);
    tickResult = node->Tick(MockTickContext(logger));
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, tickResult.Success().Status);
}

Y_UNIT_TEST(IsDirectoryError) {
    struct TMyPosixFailingClient : public TPosixWorker {
        TMyPosixFailingClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixCheckResult> IsDirectoryAsync(const TString& path) override {
            ++Calls;
            LastPath = path;

            auto result = NThreading::NewPromise<TPosixCheckResult>();
            result.SetValue(TPosixError{-1, "IsDirectoryAsync"});

            return result;
        }

        size_t Calls = 0;
        TString LastPath = "";
    };

    TPosixWorkerPtr posixWorker = new TMyPosixFailingClient();

    TString path = "flag_other";
    TIsDirectoryNodePtr node = MakeIsDirectoryNode(path, posixWorker);

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "IsDirectoryAsync");
    UNIT_ASSERT_EQUAL(1, ((TMyPosixFailingClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(path, ((TMyPosixFailingClient*)posixWorker.Get())->LastPath);
}

}

} // namespace NInfra::NPodAgent::NTestFileExistNode
