#include "make_directory_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/layer_status_repository.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTestMakeDirectoryNode {

static TLogger logger({});

Y_UNIT_TEST_SUITE(MakeDirectoryNodeSuite) {

Y_UNIT_TEST(MakeDirectorySuccess) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    const TString directoryPath = "directory";
    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TPosixWorkerPtr posixWorker = new TPosixWorker(pool);

    TMakeDirectoryNodePtr node = new TMakeDirectoryNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , holder
        , hash
        , directoryPath
    );

    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(NFs::Exists(directoryPath));
}

Y_UNIT_TEST(MakeDirectoryFailure) {
    struct TMyPosixFailingClient : public TPosixWorker {
        TMyPosixFailingClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixResult> MakeDirectoryRecursiveAsync(const TString& path) override {
            ++Calls;
            LastPath = path;

            auto result = NThreading::NewPromise<TPosixResult>();
            result.SetValue(TPosixError{-1, "MakeDirectoryRecursiveAsync"});

            return result;
        }

        size_t Calls = 0;
        TString LastPath = "";
    };

    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    const TString directoryPath = "directory";
    TPosixWorkerPtr posixWorker = new TMyPosixFailingClient();

    TMakeDirectoryNodePtr node = new TMakeDirectoryNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , holder
        , hash
        , directoryPath
    );

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C((bool)result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "MakeDirectoryRecursiveAsync");
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).failed().message(), result.Success().Message);
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), 1);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixFailingClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(directoryPath, ((TMyPosixFailingClient*)posixWorker.Get())->LastPath);
}

}

} // namespace NInfra::NPodAgent::NTestMakeDirectoryNode
