#include "make_hard_link_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/layer_status_repository.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTestMakeHardLinkNode {

static TLogger logger({});

void CreateFile(const TString& path) {
    TOFStream flag(path);
    flag.Flush();
    flag.Finish();
}

Y_UNIT_TEST_SUITE(MakeHardLinkNodeSuite) {

Y_UNIT_TEST(MakeHardLinkSuccess) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    const TString filePath = "flag";
    const TString linkPath = "flag_link";
    CreateFile(filePath);

    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TPosixWorkerPtr posixWorker = new TPosixWorker(pool);

    TMakeHardLinkNodePtr node = new TMakeHardLinkNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , holder
        , hash
        , filePath
        , linkPath
    );

    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(NFs::Exists(linkPath));
}

Y_UNIT_TEST(MakeHardLinkFailure) {
    struct TMyPosixFailingClient : public TPosixWorker {
        TMyPosixFailingClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixResult> MakeHardLinkAsync(const TString &existingPath, const TString& linkPath) override {
            ++Calls;
            LastExistingPath = existingPath;
            LastLinkPath = linkPath;

            auto result = NThreading::NewPromise<TPosixResult>();
            result.SetValue(TPosixError{-1, "MakeHardLinkAsync"});

            return result;
        }

        size_t Calls = 0;
        TString LastExistingPath = "";
        TString LastLinkPath = "";
    };

    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    const TString filePath = "flag";
    const TString linkPath = "flag_link";
    TPosixWorkerPtr posixWorker = new TMyPosixFailingClient();

    TMakeHardLinkNodePtr node = new TMakeHardLinkNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , holder
        , hash
        , filePath
        , linkPath
    );

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C((bool)result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "MakeHardLinkAsync");
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).failed().message(), result.Success().Message);
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), 1);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixFailingClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixFailingClient*)posixWorker.Get())->LastExistingPath);
    UNIT_ASSERT_EQUAL(linkPath, ((TMyPosixFailingClient*)posixWorker.Get())->LastLinkPath);
}

}

} // namespace NInfra::NPodAgent::NTestMakeHardLinkNode
