#include "make_log_file_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTestMakeLogFileNode {

static TLogger logger({});

Y_UNIT_TEST_SUITE(MakeLogFileNodeSuite) {

Y_UNIT_TEST(MakeFileSuccess) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_worklaod";
    NStatusRepositoryTypes::TContainerDescription container(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
    );
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    const TString filePath = "file";
    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TPosixWorkerPtr posixWorker = new TPosixWorker(pool);

    TMakeLogFileNodePtr node = new TMakeLogFileNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , holder
        , container
        , filePath
    );

    auto tickResult = node->Tick(MockTickContext(logger));
    UNIT_ASSERT((bool)tickResult);
    UNIT_ASSERT(NFs::Exists(filePath));
}

Y_UNIT_TEST(MakeFileFailure) {
    struct TMyPosixFailingClient : public TPosixWorker {
        TMyPosixFailingClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixResult> MakeFileAsync(const TString& path) override {
            ++Calls;
            LastPath = path;

            auto result = NThreading::NewPromise<TPosixResult>();
            result.SetValue(TPosixError{-1, "MakeFileAsync"});

            return result;
        }

        size_t Calls = 0;
        TString LastPath = "";
    };

    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_worklaod";
    NStatusRepositoryTypes::TContainerDescription container(
        id
        , NStatusRepositoryTypes::EObjectType::WORKLOAD
        , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
    );
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    const TString path = "flag_other";
    TPosixWorkerPtr posixWorker = new TMyPosixFailingClient();

    TMakeLogFileNodePtr node = new TMakeLogFileNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , holder
        , container
        , path
    );

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C((bool)result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "MakeFileAsync");
    UNIT_ASSERT_EQUAL(
        holder->GetObjectStatus(id).readiness_status().container_status().current().fail_reason()
        , result.Success().Message
    );
    UNIT_ASSERT_EQUAL(
        holder->GetObjectStatus(id).readiness_status().container_status().system_failure_counter()
        , 1
    );
    UNIT_ASSERT_EQUAL(1, ((TMyPosixFailingClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(path, ((TMyPosixFailingClient*)posixWorker.Get())->LastPath);
}

}

} // namespace NInfra::NPodAgent::NTestMakeLogFileNode
