#include "make_sym_link_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/layer_status_repository.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTestMakeSymLinkNode {

static TLogger logger({});

Y_UNIT_TEST_SUITE(MakeSymLinkNodeSuite) {

Y_UNIT_TEST(MakeSymLinkSuccess) {
    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    const TString filePath = "flag";
    const TString linkPath = "flag_sym_link";

    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TPosixWorkerPtr posixWorker = new TPosixWorker(pool);

    TMakeSymLinkNodePtr node = new TMakeSymLinkNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , holder
        , hash
        , filePath
        , linkPath
    );

    auto result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT((bool)result);
    UNIT_ASSERT(NFs::Exists(linkPath));
}

Y_UNIT_TEST(MakeSymLinkFailure) {
    struct TMyPosixFailingClient : public TPosixWorker {
        TMyPosixFailingClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixResult> MakeSymLinkAsync(const TString &existingPath, const TString& linkPath) override {
            ++Calls;
            LastExistingPath = existingPath;
            LastLinkPath = linkPath;

            auto result = NThreading::NewPromise<TPosixResult>();
            result.SetValue(TPosixError{-1, "MakeSymLinkAsync"});

            return result;
        }

        size_t Calls = 0;
        TString LastExistingPath = "";
        TString LastLinkPath = "";
    };

    TLayerStatusRepositoryPtr holder = new TLayerStatusRepository();
    const TString id = "my_layer";
    const TString hash = "my_layer_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(id, hash));

    const TString filePath = "flag";
    const TString linkPath = "flag_link";
    TPosixWorkerPtr posixWorker = new TMyPosixFailingClient();

    TMakeSymLinkNodePtr node = new TMakeSymLinkNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , holder
        , hash
        , filePath
        , linkPath
    );

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C((bool)result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "MakeSymLinkAsync");
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).failed().message(), result.Success().Message);
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), 1);
    UNIT_ASSERT_EQUAL(1, ((TMyPosixFailingClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(filePath, ((TMyPosixFailingClient*)posixWorker.Get())->LastExistingPath);
    UNIT_ASSERT_EQUAL(linkPath, ((TMyPosixFailingClient*)posixWorker.Get())->LastLinkPath);
}

}

} // namespace NInfra::NPodAgent::NTestMakeSymLinkNode

