#include "set_file_access_mode_recursive_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/static_resource_status_repository.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTestSetFileAccessModeRecursive {

static TLogger logger({});

void CreateFile(const TString& path) {
    TOFStream flag(path);
    flag.Flush();
    flag.Finish();
}

Y_UNIT_TEST_SUITE(SetFileAccessModeRecursiveNodeSuite) {

Y_UNIT_TEST(SetModeSuccess) {
    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    const TString id = "my_resource";
    const TString hash = "my_resource_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple(id, hash));

    TString path = "directory";
    NFs::MakeDirectory(path);

    TFsPath dir(path);
    dir.Child("child_dir").MkDir();
    TString childFile = TStringBuilder() << path << "/child_dir/child_file";
    CreateFile(childFile);

    TAtomicSharedPtr<IThreadPool> pool = new TFakeThreadPool;
    TAtomicSharedPtr<TPosixWorker> posixWorker = new TPosixWorker(pool);

    TSetFileAccessModeRecursiveNodePtr node = new TSetFileAccessModeRecursiveNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , path
        , EFileAccessMode::Mode_660
        , holder
        , hash
    );

    auto tickResult = node->Tick(MockTickContext(logger));
    UNIT_ASSERT((bool)tickResult);

    int intMode = 0660;

    // Check that mode where applied only to file
    UNIT_ASSERT_EQUAL((int)TFileStat(childFile).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat(path).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat("directory/child_dir").Mode & 0x00000FFF, intMode);
}

Y_UNIT_TEST(SetModeFailure) {
    struct TMyPosixFailingClient : public TPosixWorker {
        TMyPosixFailingClient() : TPosixWorker(nullptr) {}

        NThreading::TFuture<TPosixResult> SetFileModeRecursiveAsync(const TString& path, EFileAccessMode) override {
            ++Calls;
            LastPath = path;

            auto result = NThreading::NewPromise<TPosixResult>();
            result.SetValue(TPosixError{-1, "SetFileModeRecursiveAsync"});

            return result;
        }

        size_t Calls = 0;
        TString LastPath = "";
    };

    TStaticResourceStatusRepositoryPtr holder = new TStaticResourceStatusRepository();
    const TString id = "my_resource";
    const TString hash = "my_resource_download_hash";
    holder->AddObject(NObjectMetaTestLib::CreateStaticResourceMetaSimple(id, hash));

    TPosixWorkerPtr posixWorker = new TMyPosixFailingClient();
    TString path = "directory_1";
    NFs::MakeDirectory(path);

    TFsPath dir(path);
    dir.Child("child_dir").MkDir();
    TString childFile = TStringBuilder() << path << "/child_dir/child_file";
    CreateFile(childFile);

    TSetFileAccessModeRecursiveNodePtr node = new TSetFileAccessModeRecursiveNode(
        TBasicTreeNodeDescriptor{1, "title"}
        , posixWorker
        , path
        , EFileAccessMode::Mode_660
        , holder
        , hash
    );

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "SetFileModeRecursiveAsync");
    UNIT_ASSERT_EQUAL(1, ((TMyPosixFailingClient*)posixWorker.Get())->Calls);
    UNIT_ASSERT_EQUAL(path, ((TMyPosixFailingClient*)posixWorker.Get())->LastPath);

    // Check that mode where not applied
    int intMode = 0660;
    UNIT_ASSERT_UNEQUAL((int)TFileStat(childFile).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat(path).Mode & 0x00000FFF, intMode);
    UNIT_ASSERT_UNEQUAL((int)TFileStat("directory_1/child_dir").Mode & 0x00000FFF, intMode);

    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).failed().message(), result.Success().Message);
    UNIT_ASSERT_EQUAL(holder->GetObjectStatus(id).fail_counter(), 1);
}

}

} // namespace NInfra::NPodAgent::NTestRemoveRecursiveNode
