#include "capture_unix_signal_status_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <infra/pod_agent/libs/system_logs_sender/mock_system_logs_session.h>
#include <infra/pod_agent/libs/system_logs_sender/sidecars_system_logs_filter_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_sender_impl.h>
#include <infra/pod_agent/libs/system_logs_sender/system_logs_session_collection_impl.h>

#include <infra/libs/logger/logger.h>

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTestCaptureUnixSignalStatusNode {

Y_UNIT_TEST_SUITE(CaptureUnixSignalStatusNodeSuite) {

static TLogger logger({});

class TTestSystemLogsSession: public TMockSystemLogsSession
{
public:
    virtual ~TTestSystemLogsSession() = default;

    void SendMessage(const TString& message) override {
        SendMessageCalls_++;
        SentMessage_ = message;
    }

    TString SentMessage_ = "";
    int SendMessageCalls_ = 0;
};

Y_UNIT_TEST(TestCaptureHookStatus) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";

    TSystemLogsSenderPtr systemLogsSender = new TSystemLogsSender(new TSystemLogsSessionCollection, new TFakeThreadPool, new TSystemLogsStatistics, new TSidecarsSystemLogsFilterImpl());
    
    TSystemLogsSessionPtr systemLogsSession = new TTestSystemLogsSession;
    systemLogsSender->Add(id, systemLogsSession);

    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));
    TCaptureUnixSignalStatusNodePtr node = new TCaptureUnixSignalStatusNode(TBasicTreeNodeDescriptor{1, "title"}, holder, systemLogsSender, id);

    holder->UpdateStopUnixSignalState(id, API::EUnixSignalState::EUnixSignalState_SUCCESS);
    holder->UpdateStopUnixSignalFailReason(id, "fail_reason");
    UNIT_ASSERT(!GetStopUnixSignalStatus(holder->GetObjectStatus(id)).has_last());

    auto result = node->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), result.Success());
    UNIT_ASSERT_EQUAL(((TTestSystemLogsSession*)systemLogsSession.Get())->SendMessageCalls_, 1);

    auto status = GetStopUnixSignalStatus(holder->GetObjectStatus(id)).last();
    UNIT_ASSERT_EQUAL(API::EUnixSignalState::EUnixSignalState_SUCCESS, status.state());
    UNIT_ASSERT_EQUAL("fail_reason", status.fail_reason());

    auto logEvent = node->CreateLogEvent(status);
    TString systemLogMessage = node->SystemLogsMessage(logEvent);
    UNIT_ASSERT_EQUAL(((TTestSystemLogsSession*)systemLogsSession.Get())->SentMessage_, systemLogMessage);

    status = GetStopUnixSignalStatus(holder->GetObjectStatus(id)).current();
    UNIT_ASSERT_EQUAL(API::EUnixSignalState::EUnixSignalState_UNKNOWN, status.state());
    UNIT_ASSERT_EQUAL("", status.fail_reason());

    systemLogsSender->RemoveAll();
    node = new TCaptureUnixSignalStatusNode(TBasicTreeNodeDescriptor{1, "title"}, holder, systemLogsSender, id);
    result = node->Tick(MockTickContext(logger));
    UNIT_ASSERT(!(bool)result);
}

Y_UNIT_TEST(TestWithUndefinedWorkloadStatusRepository) {
    TSystemLogsSenderPtr systemLogsSender = new TSystemLogsSender(new TSystemLogsSessionCollection, new TFakeThreadPool, new TSystemLogsStatistics, new TSidecarsSystemLogsFilterImpl());

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        TCaptureUnixSignalStatusNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , nullptr
            , systemLogsSender
            , "my_workload"
        )
        , yexception
        , "WorkloadStatusRepository not defined"
    );
}

Y_UNIT_TEST(TestWithUndefinedSystemLogsSender) {
    TWorkloadStatusRepositoryPtr statusRepository = new TWorkloadStatusRepository();

    UNIT_ASSERT_EXCEPTION_CONTAINS(
        TCaptureUnixSignalStatusNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , statusRepository
            , nullptr
            , "my_workload"
        )
        , yexception
        , "system logs sender not defined"
    );
}

}

} // namespace NInfra::NPodAgent::NTestCaptureUnixSignalStatusNode
