#include "porto_kill_with_unix_signal_feedback_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/workload_status_repository.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/logger/log_frame.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestPortoKillWithUnixSignalFeedbackNode {

Y_UNIT_TEST_SUITE(PortoKillWithUnixSignalFeedbackNodeSuite) {

static TLogger logger({});

TPortoKillWithUnixSignalFeedbackNodePtr CreateNode(
    TPortoClientPtr porto
    , const TWorkloadStatusRepositoryPtr workloadStatusRepository
    , const TString& workloadId
    , const TPortoContainerName& containerName
    , i32 signal
) {
    return TPortoKillWithUnixSignalFeedbackNodePtr(
        new TPortoKillWithUnixSignalFeedbackNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , new TAsyncPortoClient(porto, new TFakeThreadPool())
            , workloadStatusRepository
            , workloadId
            , containerName
            , signal
        )
    );
}

Y_UNIT_TEST(TestKillContainer) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int signal) override {
            ++Calls_;
            LastName_ = name;
            LastSignal_ = signal;

            return TExpected<void, TPortoError>::DefaultSuccess();
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        i32 LastSignal_ = 0;
    };

    const TString workloadId = "my_workload";
    const TPortoContainerName containerName = {"some_name"};
    const i32 signal = SIGTERM;

    TPortoClientPtr porto = new TMyPortoClient();
    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto creator = CreateNode(
        porto
        , workloadStatusRepository
        , workloadId
        , containerName
        , signal
    );
    auto result = creator->Tick(MockTickContext(logger));

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    auto status = GetStopUnixSignalStatus(workloadStatusRepository->GetObjectStatus(workloadId));
    UNIT_ASSERT(status.current().send_time().seconds() != 0 || status.current().send_time().nanos() != 0);
    UNIT_ASSERT_EQUAL_C(
        API::EUnixSignalState::EUnixSignalState_SUCCESS
        , status.current().state()
        , API::EUnixSignalState_Name(status.current().state())
    );
    UNIT_ASSERT_EQUAL_C(
        ""
        , status.current().fail_reason()
        , status.current().fail_reason()
    );
    UNIT_ASSERT_EQUAL_C(
        1
        , workloadStatusRepository->GetStopUnixSignalSuccessCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalSuccessCounter(workloadId)
    );
    UNIT_ASSERT_EQUAL_C(
        0
        , workloadStatusRepository->GetStopUnixSignalErrorCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalErrorCounter(workloadId)
    );
    UNIT_ASSERT_EQUAL_C(
        1
        , workloadStatusRepository->GetStopUnixSignalConsecutiveSuccessesCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalConsecutiveSuccessesCounter(workloadId)
    );
    UNIT_ASSERT_EQUAL_C(
        0
        , workloadStatusRepository->GetStopUnixSignalConsecutiveFailuresCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalConsecutiveFailuresCounter(workloadId)
    );

    UNIT_ASSERT_EQUAL_C(
        1
        , ((TMyPortoClient*)porto.Get())->Calls_
        , ((TMyPortoClient*)porto.Get())->Calls_
    );
    UNIT_ASSERT_EQUAL_C(
        containerName
        , ((TMyPortoClient*)porto.Get())->LastName_
        , ((TMyPortoClient*)porto.Get())->LastName_
    );
    UNIT_ASSERT_EQUAL_C(
        signal
        , ((TMyPortoClient*)porto.Get())->LastSignal_
        , ((TMyPortoClient*)porto.Get())->LastSignal_
    );
}

Y_UNIT_TEST(TestKillContainerWithFailingKill) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int signal) override {
            ++Calls_;
            LastName_ = name;
            LastSignal_ = signal;

            return TPortoError{EPortoError::InvalidState, "Kill", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        i32 LastSignal_ = 0;
    };

    const TString workloadId = "my_workload";
    const TPortoContainerName containerName = {"some_name"};
    const i32 signal = SIGTERM;

    TPortoClientPtr porto = new TMyPortoFailingClient();
    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto creator = CreateNode(
        porto
        , workloadStatusRepository
        , workloadId
        , containerName
        , signal
    );
    auto result = creator->Tick(MockTickContext(logger));


    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
    UNIT_ASSERT_STRING_CONTAINS(result.Success().Message, "Kill");

    auto status = GetStopUnixSignalStatus(workloadStatusRepository->GetObjectStatus(workloadId));
    UNIT_ASSERT(status.current().send_time().seconds() != 0 || status.current().send_time().nanos() != 0);
    UNIT_ASSERT_EQUAL_C(
        API::EUnixSignalState::EUnixSignalState_FAILURE
        , status.current().state()
        , API::EUnixSignalState_Name(status.current().state())
    );
    UNIT_ASSERT_STRING_CONTAINS(
        status.current().fail_reason()
        , "Kill"
    );
    UNIT_ASSERT_EQUAL_C(
        0
        , workloadStatusRepository->GetStopUnixSignalSuccessCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalSuccessCounter(workloadId)
    );
    UNIT_ASSERT_EQUAL_C(
        1
        , workloadStatusRepository->GetStopUnixSignalErrorCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalErrorCounter(workloadId)
    );
    UNIT_ASSERT_EQUAL_C(
        0
        , workloadStatusRepository->GetStopUnixSignalConsecutiveSuccessesCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalConsecutiveSuccessesCounter(workloadId)
    );
    UNIT_ASSERT_EQUAL_C(
        1
        , workloadStatusRepository->GetStopUnixSignalConsecutiveFailuresCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalConsecutiveFailuresCounter(workloadId)
    );

    UNIT_ASSERT_EQUAL_C(
        1
        , ((TMyPortoFailingClient*)porto.Get())->Calls_
        , ((TMyPortoFailingClient*)porto.Get())->Calls_
    );
    UNIT_ASSERT_EQUAL_C(
        containerName
        , ((TMyPortoFailingClient*)porto.Get())->LastName_
        , ((TMyPortoFailingClient*)porto.Get())->LastName_
    );
    UNIT_ASSERT_EQUAL_C(
        signal
        , ((TMyPortoFailingClient*)porto.Get())->LastSignal_
        , ((TMyPortoFailingClient*)porto.Get())->LastSignal_
    );
}


Y_UNIT_TEST(TestKillContainerWithFailingPorto) {
    struct TMyPortoFailingClient : public TMockPortoClient {
        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int signal) override {
            ++Calls_;
            LastName_ = name;
            LastSignal_ = signal;

            return TPortoError{EPortoError::Unknown, "Kill", "NO"};
        }

        size_t Calls_ = 0;
        TPortoContainerName LastName_ = {""};
        i32 LastSignal_ = 0;
    };

    const TString workloadId = "my_workload";
    const TPortoContainerName containerName = {"some_name"};
    const i32 signal = SIGTERM;

    TPortoClientPtr porto = new TMyPortoFailingClient();
    TWorkloadStatusRepositoryPtr workloadStatusRepository = new TWorkloadStatusRepository();
    workloadStatusRepository->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(workloadId));

    auto creator = CreateNode(
        porto
        , workloadStatusRepository
        , workloadId
        , containerName
        , signal
    );
    auto result = creator->Tick(MockTickContext(logger));


    UNIT_ASSERT_C(!result, result.Success().Message);
    UNIT_ASSERT_STRING_CONTAINS(result.Error().Message, "Kill");

    auto status = GetStopUnixSignalStatus(workloadStatusRepository->GetObjectStatus(workloadId));
    UNIT_ASSERT(status.current().send_time().seconds() != 0 || status.current().send_time().nanos() != 0);
    UNIT_ASSERT_EQUAL_C(
        API::EUnixSignalState::EUnixSignalState_FAILURE
        , status.current().state()
        , API::EUnixSignalState_Name(status.current().state())
    );
    UNIT_ASSERT_STRING_CONTAINS(
        status.current().fail_reason()
        , "Kill"
    );
    UNIT_ASSERT_EQUAL_C(
        0
        , workloadStatusRepository->GetStopUnixSignalSuccessCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalSuccessCounter(workloadId)
    );
    UNIT_ASSERT_EQUAL_C(
        1
        , workloadStatusRepository->GetStopUnixSignalErrorCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalErrorCounter(workloadId)
    );
    UNIT_ASSERT_EQUAL_C(
        0
        , workloadStatusRepository->GetStopUnixSignalConsecutiveSuccessesCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalConsecutiveSuccessesCounter(workloadId)
    );
    UNIT_ASSERT_EQUAL_C(
        1
        , workloadStatusRepository->GetStopUnixSignalConsecutiveFailuresCounter(workloadId)
        , workloadStatusRepository->GetStopUnixSignalConsecutiveFailuresCounter(workloadId)
    );

    UNIT_ASSERT_EQUAL_C(
        1
        , ((TMyPortoFailingClient*)porto.Get())->Calls_
        , ((TMyPortoFailingClient*)porto.Get())->Calls_
    );
    UNIT_ASSERT_EQUAL_C(
        containerName
        , ((TMyPortoFailingClient*)porto.Get())->LastName_
        , ((TMyPortoFailingClient*)porto.Get())->LastName_
    );
    UNIT_ASSERT_EQUAL_C(
        signal
        , ((TMyPortoFailingClient*)porto.Get())->LastSignal_
        , ((TMyPortoFailingClient*)porto.Get())->LastSignal_
    );
}

}

} // namespace NInfra::NPodAgent::NTestPortoKillWithUnixSignalFeedbackNode
