#include "unix_signal_send_time_expired_with_backoff_node.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/test/mock_tick_context.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/test_functions.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/generic/serialized_enum.h>

namespace NInfra::NPodAgent::NTestUnixSignalSendTimeExpiredWithBackoffNode {

TUnixSignalSendTimeExpiredWithBackoffNodePtr CreateNode(
    TWorkloadStatusRepositoryPtr statusRepository
    , const TString& id
    , TDuration restartPeriodScale
    , ui64 restartPeriodBackOff
    , TDuration maxRestartPeriod
    , TDuration minRestartPeriod
) {
    return TUnixSignalSendTimeExpiredWithBackoffNodePtr(
        new TUnixSignalSendTimeExpiredWithBackoffNode(
            TBasicTreeNodeDescriptor{1, "title"}
            , statusRepository
            , id
            , restartPeriodScale
            , restartPeriodBackOff
            , maxRestartPeriod
            , minRestartPeriod
        )
    );
}

static TLogger logger({});

Y_UNIT_TEST_SUITE(UnixSignalSendTimeExpiredWithBackoffSuite) {

Y_UNIT_TEST(TestUnixSignalSendTimeExpired) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateStopUnixSignalSendTime(id, TInstant::Seconds(10));
    holder->CaptureStopUnixSignalStatus(id);

    TDuration duration = TDuration::Seconds(60);
    auto node = CreateNode(holder, id, TDuration::FromValue(0), 1, duration, duration);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL_C(0, context->GetLongTickPeriodDuration(), context->GetLongTickPeriodDuration());
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
}

Y_UNIT_TEST(TestUnixSignalSendTimeNotExpired) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateStopUnixSignalSendTime(id, TInstant::Seconds(4000000000));
    holder->CaptureStopUnixSignalStatus(id);

    TDuration duration = TDuration::Seconds(60);
    auto node = CreateNode(holder, id, TDuration::FromValue(0), 1, duration, duration);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
}

Y_UNIT_TEST(TestUnixSignalSendTimeNowNotExpired) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateStopUnixSignalSendTime(id, TInstant::Now());
    holder->CaptureStopUnixSignalStatus(id);

    TDuration duration = TDuration::Seconds(60);
    auto node = CreateNode(holder, id, TDuration::FromValue(0), 1, duration, duration);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT(labs(60000 - (long) context->GetLongTickPeriodDuration()) < 1000);
    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
}

Y_UNIT_TEST(TestUnixSignalBackOff) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateStopUnixSignalSendTime(id, TInstant::Now() - TDuration::Seconds(59));
    holder->UpdateStopUnixSignalConsecutiveFailuresAndSuccessesCounter(id, true);
    holder->CaptureStopUnixSignalStatus(id);

    TDuration second = TDuration::Seconds(1);
    auto node = CreateNode(holder, id, second, 1000, second * 1000, second);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    holder->UpdateStopUnixSignalConsecutiveFailuresAndSuccessesCounter(id, true);

    context = MockTickContext(logger);
    result = node->Tick(context);

    UNIT_ASSERT(labs(1000000 - 59000 - (long) context->GetLongTickPeriodDuration()) < 1000);
    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::FAILURE, result.Success().Status);
}

Y_UNIT_TEST(TestUnixSignalNoBackOff) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateStopUnixSignalSendTime(id, TInstant::Now() - TDuration::Seconds(59));
    holder->UpdateStopUnixSignalConsecutiveFailuresAndSuccessesCounter(id, false);
    holder->CaptureStopUnixSignalStatus(id);

    TDuration second = TDuration::Seconds(1);
    auto node = CreateNode(holder, id, second, 1000, second * 1000, second);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    context = MockTickContext(logger);
    result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
}

Y_UNIT_TEST(TestUnixSignalDoubleCheck) {
    TWorkloadStatusRepositoryPtr holder = new TWorkloadStatusRepository();
    const TString id = "my_workload";
    holder->AddObject(NObjectMetaTestLib::CreateWorkloadMetaSimple(id));

    holder->UpdateStopUnixSignalSendTime(id, TInstant::Now() - TDuration::Seconds(59));
    holder->UpdateStopUnixSignalConsecutiveFailuresAndSuccessesCounter(id, true);
    holder->CaptureStopUnixSignalStatus(id);

    TDuration second = TDuration::Seconds(1);
    auto node = CreateNode(holder, id, second, 1000, second * 1000, second);

    auto context = MockTickContext(logger);
    auto result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);

    context = MockTickContext(logger);
    result = node->Tick(context);

    UNIT_ASSERT_C(result, result.Error().Message);
    UNIT_ASSERT_EQUAL(ENodeStatus::SUCCESS, result.Success().Status);
}

Y_UNIT_TEST(TestWithUndefinedWorkloadStatusRepository) {
    UNIT_ASSERT_EXCEPTION_CONTAINS(
        CreateNode(
            nullptr
            , "my_workload"
            , TDuration::Seconds(0)
            , 0
            , TDuration::Seconds(0)
            , TDuration::Seconds(0)
        )
        , yexception
        , "WorkloadStatusRepository not defined"
    );
}

}

} // namespace NInfra::NPodAgent::NTestUnixSignalSendTimeExpiredWithBackoffNode
