#include "console_renderer.h"

#include <util/generic/string.h>
#include <util/string/cast.h>

namespace NInfra::NPodAgent {

TString TConsoleRenderer::Render(TTreeConstPtr tree, bool all) {
    TStringBuilder result;
    result << RenderHeader();
    DFSRender(tree->GetRootNode(), 0, tree->GetTrace()->GetTrace(), all, result);
    return result;
}

TString TConsoleRenderer::RenderAll(TTreeConstPtr tree) {
    return Render(tree, /*all*/true);
}

void TConsoleRenderer::DFSRender(TBasicTreeNodePtr node, i32 depth, const THashSet<ui64>& traced, bool renderAll, TStringBuilder& result) {
    result << SINGLE_INDENT * depth;

    TString currentTitle = node->GetTitle();
    if (currentTitle.StartsWith("Root:")) {
        // Parse as "Root:<tree name>:<node title>
        ssize_t pos = currentTitle.find_last_of(':');
        TString realTitle = currentTitle.substr(pos + 1);
        if (realTitle.empty()) {
            realTitle = ToString(node->GetType());
        } else {
            realTitle = TStringBuilder()
                << ToString(node->GetType())
                << "(" << realTitle << ")"
            ;
        }
        result << "RR> " << BoldYellow(currentTitle.substr(0, pos + 1) + realTitle);
    } else {
        if (currentTitle.empty()) {
            currentTitle = ToString(node->GetType());
        } else {
            currentTitle = TStringBuilder()
                << ToString(node->GetType())
                << "(" << currentTitle << ")"
            ;
        }
        result << "--> " << BoldWhite(currentTitle);
    }

    if (traced.contains(node->GetId())) {
        result << " [" << NodeResult(node->GetTickResult()) << "]" << Endl;
    } else {
        result << " [IDLE]" << Endl;
    }
    for (auto&& child : node->GetChildren()) {
        if (renderAll || traced.contains(child->GetId())) {
            DFSRender(child, depth + 1, traced, renderAll, result);
        }
    }
}

TString TConsoleRenderer::RenderHeader() const {
    return TStringBuilder() << "-------------------------" << Endl;
}

TString TConsoleRenderer::NodeResult(const TTickResult& result) {
    TStringBuilder render;
    if (result) {
        ENodeStatus status = result.Success().Status;
        switch (status) {
            case ENodeStatus::SUCCESS:
                render << Green(ToString(status));
                break;
            case ENodeStatus::FAILURE:
                render << Red(ToString(status));
                break;
            case ENodeStatus::RUNNING:
                render << Blue(ToString(status));
                break;
            default:
                render << status;
        }
        TString message = result.Success().Message;
        if (message) {
            render << " : " << message;
        }
    } else {
        render << Purple("ERROR");
        TString message = result.Error().Message;
        if (message) {
            render << " : " << message;
        }
    }
    return render;
}

TString TConsoleRenderer::Red(TStringBuf text) {
    if (!UseColors_)
        return TString(text);
    return TStringBuilder() << Colors_.RedColor() << text << Colors_.OldColor();
}

TString TConsoleRenderer::Green(TStringBuf text) {
    if (!UseColors_)
        return TString(text);
    return TStringBuilder() << Colors_.GreenColor() << text << Colors_.OldColor();
}

TString TConsoleRenderer::Blue(TStringBuf text) {
    if (!UseColors_)
        return TString(text);
    return TStringBuilder() << Colors_.BlueColor() << text << Colors_.OldColor();
}

TString TConsoleRenderer::Purple(TStringBuf text) {
    if (!UseColors_)
        return TString(text);
    return TStringBuilder() << Colors_.PurpleColor() << text << Colors_.OldColor();
}

TString TConsoleRenderer::BoldWhite(TStringBuf text){
    if (!UseColors_)
        return TString(text);
    return TStringBuilder() << Colors_.WhiteColor() << Colors_.BoldColor() << text << Colors_.OldColor();
}

TString TConsoleRenderer::BoldYellow(TStringBuf text) {
    if (!UseColors_)
        return TString(text);
    return TStringBuilder() << Colors_.YellowColor() << Colors_.BoldColor() << text << Colors_.OldColor();
}


} // namespace NInfra::NPodAgent
