#include "proto_renderer.h"

#include <util/string/cast.h>

namespace NInfra::NPodAgent {

NLogEvent::TBehaviourTreeTickV2 TProtoRenderer::Render(TTreeConstPtr tree) {
    NLogEvent::TBehaviourTreeTickV2 result;
    result.SetTreeId(tree->GetTreeId());
    DFSRender(tree->GetRootNode(), tree->GetTrace()->GetTrace(), result.MutableRoot());
    return result;
}

void TProtoRenderer::DFSRender(
    TBasicTreeNodePtr node
    , const THashSet<ui64>& traced
    , NLogEvent::TBehaviourTreeTickV2::TNodeTick* result
) {
    result->SetTitle(node->GetTitle());
    result->SetNodeType(ConvertNodeType(node->GetType()));
    if (node->GetTickResult()) {
        auto success = node->GetTickResult().Success();
        auto ptr = result->MutableSuccess();
        ptr->SetStatus(ConvertNodeStatus(success.Status));
        ptr->SetMessage(success.Message);
    } else {
        auto error = node->GetTickResult().Error();
        result->MutableError()->SetMessage(error.Message);
    }
    for (auto&& child : node->GetChildren()) {
        if (traced.contains(child->GetId())) {
            auto ptr = result->add_children();
            DFSRender(child, traced, ptr);
        }
    }
}

NLogEvent::TBehaviourTreeTickV2::ENodeStatus TProtoRenderer::ConvertNodeStatus(ENodeStatus nodeStatus) {
    switch (nodeStatus) {
        case ENodeStatus::RUNNING:
            return NLogEvent::TBehaviourTreeTickV2::ENodeStatus_RUNNING;
        case ENodeStatus::SUCCESS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeStatus_SUCCESS;
        case ENodeStatus::FAILURE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeStatus_FAILURE;
        case ENodeStatus::IDLE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeStatus_IDLE;
    }

    ythrow yexception() << "Unknown node status '" << ToString(nodeStatus) << "'";
}

NLogEvent::TBehaviourTreeTickV2::ENodeType TProtoRenderer::ConvertNodeType(ENodeType nodeType) {
    switch (nodeType) {
        case ENodeType::UNKNOWN:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_UNKNOWN;
        case ENodeType::MEM_SEQUENCE_GENERATOR:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_MEM_SEQUENCE_GENERATOR;
        case ENodeType::SUCCEEDER:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_SUCCEEDER;
        case ENodeType::FAILER:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FAILER;
        case ENodeType::ERROR:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_ERROR;
        case ENodeType::PORTO_CONTAINER_EXISTS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_CONTAINER_EXISTS;
        case ENodeType::PORTO_CREATE_RECURSIVE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_CREATE_RECURSIVE;
        case ENodeType::PORTO_START:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_START;
        case ENodeType::PORTO_STOP:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_STOP;
        case ENodeType::PORTO_DESTROY:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_DESTROY;
        case ENodeType::PORTO_KILL:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_KILL;
        case ENodeType::PORTO_GET_AND_CHECK_PROPERTIES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_GET_AND_CHECK_PROPERTIES;
        case ENodeType::PORTO_GET_DOWNLOAD_PROGRESS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_GET_DOWNLOAD_PROGRESS;
        case ENodeType::PORTO_GET_AND_CHECK_VOLUME_PROPERTIES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_GET_AND_CHECK_VOLUME_PROPERTIES;
        case ENodeType::PORTO_SET_PROPERTY:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_SET_PROPERTY;
        case ENodeType::PORTO_CONTAINER_PROPERTY_MATCHES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_CONTAINER_PROPERTY_MATCHES;
        case ENodeType::PORTO_CONTAINER_TREE_HASH_MATCHES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_CONTAINER_TREE_HASH_MATCHES;
        case ENodeType::PORTO_GET_CONTAINER_PRIVATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_GET_CONTAINER_PRIVATE;
        case ENodeType::PORTO_CONTAINER_PRIVATE_MATCHES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_CONTAINER_PRIVATE_MATCHES;
        case ENodeType::PORTO_GET_LAYER_PRIVATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_GET_LAYER_PRIVATE;
        case ENodeType::PORTO_GET_CONTAINER_STATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_GET_CONTAINER_STATE;
        case ENodeType::PORTO_SET_CONTAINER_PRIVATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_SET_CONTAINER_PRIVATE;
        case ENodeType::CONTAINER_START_TIME_EXPIRED_WITH_BACK_OFF:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CONTAINER_START_TIME_EXPIRED_WITH_BACK_OFF;
        case ENodeType::PORTO_CONTAINER_TIME_EXPIRED:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_CONTAINER_TIME_EXPIRED;
        case ENodeType::PORTO_STATIC_RESOURCE_VERIFY_CONTAINER_TIME_EXPIRED:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_STATIC_RESOURCE_VERIFY_CONTAINER_TIME_EXPIRED;
        case ENodeType::FILE_MODIFIED_BEFORE_PORTO_CONTAINER_START:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FILE_MODIFIED_BEFORE_PORTO_CONTAINER_START;
        case ENodeType::FILE_MODIFIED_RECURSIVE_BEFORE_PORTO_CONTAINER_START:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FILE_MODIFIED_RECURSIVE_BEFORE_PORTO_CONTAINER_START;
        case ENodeType::PORTO_VOLUME_EXISTS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_VOLUME_EXISTS;
        case ENodeType::PORTO_CREATE_VOLUME:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_CREATE_VOLUME;
        case ENodeType::PORTO_LINK_VOLUME:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_LINK_VOLUME;
        case ENodeType::PORTO_UNLINK_VOLUME:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_UNLINK_VOLUME;
        case ENodeType::PORTO_VOLUME_IS_LINK_OF:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_VOLUME_IS_LINK_OF;
        case ENodeType::PORTO_LAYER_EXISTS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_LAYER_EXISTS;
        case ENodeType::PORTO_LAYERS_READY:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_LAYERS_READY;
        case ENodeType::PORTO_SET_LAYER_PRIVATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_SET_LAYER_PRIVATE;
        case ENodeType::PORTO_IMPORT_LAYER:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_IMPORT_LAYER;
        case ENodeType::PORTO_REMOVE_LAYER:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_REMOVE_LAYER;
        case ENodeType::PORTO_CLEAN_LINKED_VOLUMES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_CLEAN_LINKED_VOLUMES;
        case ENodeType::PORTO_CLEAN_OLD_CONTAINERS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_CLEAN_OLD_CONTAINERS;
        case ENodeType::FEEDBACK_OBJECT_STATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_OBJECT_STATE;
        case ENodeType::FEEDBACK_OBJECT_DOWNLOAD_COMPLETE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_OBJECT_DOWNLOAD_COMPLETE;
        case ENodeType::STATIC_RESOURCE_STATE_MATCHES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_STATIC_RESOURCE_STATE_MATCHES;
        case ENodeType::FEEDBACK_OBJECT_FAILED_MESSAGE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_OBJECT_FAILED_MESSAGE;
        case ENodeType::FEEDBACK_OBJECT_IP_ADDRESS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_OBJECT_IP_ADDRESS;
        case ENodeType::SET_WORKLOAD_WAS_DESTROYED_BY_HOOK:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_SET_WORKLOAD_WAS_DESTROYED_BY_HOOK;
        case ENodeType::WORKLOAD_SHOULD_BE_DESTROYED_BY_HOOK:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_WORKLOAD_SHOULD_BE_DESTROYED_BY_HOOK;
        case ENodeType::PORTO_ADD_PROPERTIES_TO_OBJECT_FAILED_MESSAGE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_ADD_PROPERTIES_TO_OBJECT_FAILED_MESSAGE;
        case ENodeType::FEEDBACK_OBJECT_VERIFY_ATTEMPT:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_OBJECT_VERIFY_ATTEMPT;
        case ENodeType::PORTO_STORAGE_PRIVATE_MATCHES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_STORAGE_PRIVATE_MATCHES;
        case ENodeType::PORTO_REMOVE_STORAGE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_REMOVE_STORAGE;
        case ENodeType::FEEDBACK_OBJECT_CHECK_FAILED_MESSAGE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_OBJECT_CHECK_FAILED_MESSAGE;
        case ENodeType::FEEDBACK_OBJECT_DOWNLOAD_ATTEMPT:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_OBJECT_DOWNLOAD_ATTEMPT;
        case ENodeType::CHECK_CONTAINER_RETRIES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CHECK_CONTAINER_RETRIES;
        case ENodeType::WORKLOAD_TARGET_ACTIVE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_WORKLOAD_TARGET_ACTIVE;
        case ENodeType::FEEDBACK_CONTAINER_FAIL_REASON:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_CONTAINER_FAIL_REASON;
        case ENodeType::FEEDBACK_CONTAINER_STATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_CONTAINER_STATE;
        case ENodeType::FEEDBACK_CONTAINER_STATUS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_CONTAINER_STATUS;
        case ENodeType::FEEDBACK_CONTAINER_START_TIME:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_CONTAINER_START_TIME;
        case ENodeType::FEEDBACK_CONTAINER_DEATH_TIME:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_CONTAINER_DEATH_TIME;
        case ENodeType::FEEDBACK_CONTAINER_STDERR:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_CONTAINER_STDERR;
        case ENodeType::FEEDBACK_CONTAINER_STDOUT:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_CONTAINER_STDOUT;
        case ENodeType::FEEDBACK_CONTAINER_OWNER_STATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_CONTAINER_OWNER_STATE;
        case ENodeType::VARIABLE_IS_SET:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_VARIABLE_IS_SET;
        case ENodeType::MAKE_DIRECTORY:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_MAKE_DIRECTORY;
        case ENodeType::MAKE_LOG_FILE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_MAKE_LOG_FILE;
        case ENodeType::FILE_EXISTS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FILE_EXISTS;
        case ENodeType::MAKE_HARD_LINK:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_MAKE_HARD_LINK;
        case ENodeType::MAKE_SYM_LINK:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_MAKE_SYM_LINK;
        case ENodeType::REMOVE_FILE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_REMOVE_FILE;
        case ENodeType::REMOVE_RECURSIVE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_REMOVE_RECURSIVE;
        case ENodeType::NETWORK_HOOK_START_TIME_EXPIRED_WITH_BACKOFF:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_NETWORK_HOOK_START_TIME_EXPIRED_WITH_BACKOFF;
        case ENodeType::CHECK_HTTP_HOOK_RETRIES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CHECK_HTTP_HOOK_RETRIES;
        case ENodeType::FEEDBACK_HTTP_HOOK_FAIL_REASON:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_HTTP_HOOK_FAIL_REASON;
        case ENodeType::CAPTURE_NETWORK_HOOK_STATUS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CAPTURE_NETWORK_HOOK_STATUS;
        case ENodeType::CAPTURE_CONTAINER_STATUS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CAPTURE_CONTAINER_STATUS;
        case ENodeType::ADD_IP_ADDRESS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_ADD_IP_ADDRESS;
        case ENodeType::MEM_SEQUENCE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_MEM_SEQUENCE;
        case ENodeType::MEM_SELECTOR:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_MEM_SELECTOR;
        case ENodeType::INVERTER:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_INVERTER;
        case ENodeType::CASE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CASE;
        case ENodeType::SWITCH:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_SWITCH;
        case ENodeType::STATIC_SWITCH:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_STATIC_SWITCH;
        case ENodeType::TIMED_CACHE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_TIMED_CACHE;
        case ENodeType::TRY_LOCK:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_TRY_LOCK;
        case ENodeType::MOCK:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_MOCK;
        case ENodeType::NETWORK_REMOVE_REQUEST:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_NETWORK_REMOVE_REQUEST;
        case ENodeType::CHECK_SAME_INODE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CHECK_SAME_INODE;
        case ENodeType::NETWORK_HOOK_CONSECUTIVE_TRIES_REACHED:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_NETWORK_HOOK_CONSECUTIVE_TRIES_REACHED;
        case ENodeType::NETWORK_GET_REQUEST_STATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_NETWORK_GET_REQUEST_STATE;
        case ENodeType::PORTO_ACTIVE_DOWNLOAD_CONTAINERS_LIMIT_REACHED:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_ACTIVE_DOWNLOAD_CONTAINERS_LIMIT_REACHED;
        case ENodeType::FEEDBACK_NETWORK_HOOK_STATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_NETWORK_HOOK_STATE;
        case ENodeType::NETWORK_CHECK_AND_ADD_REQUEST:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_NETWORK_CHECK_AND_ADD_REQUEST;
        case ENodeType::NETWORK_GET_AND_REMOVE_REQUEST_RESPONSE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_NETWORK_GET_AND_REMOVE_REQUEST_RESPONSE;
        case ENodeType::CONTAINER_CONSECUTIVE_TRIES_REACHED:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CONTAINER_CONSECUTIVE_TRIES_REACHED;
        case ENodeType::LAYER_KEEP_SOURCE_FILE_AFTER_IMPORT:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_LAYER_KEEP_SOURCE_FILE_AFTER_IMPORT;
        case ENodeType::CAPTURE_UNIX_SIGNAL_STATUS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CAPTURE_UNIX_SIGNAL_STATUS;
        case ENodeType::CHECK_UNIX_SIGNAL_RETRIES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_CHECK_UNIX_SIGNAL_RETRIES;
        case ENodeType::FEEDBACK_UNIX_SIGNAL_STATE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FEEDBACK_UNIX_SIGNAL_STATE;
        case ENodeType::UNIX_SIGNAL_SEND_TIME_EXPIRED_WITH_BACKOFF:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_UNIX_SIGNAL_SEND_TIME_EXPIRED_WITH_BACKOFF;
        case ENodeType::PORTO_KILL_WITH_UNIX_SIGNAL_FEEDBACK:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_KILL_WITH_UNIX_SIGNAL_FEEDBACK;
        case ENodeType::WORKLOAD_CLEAR_STOP_STATUS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_WORKLOAD_CLEAR_STOP_STATUS;
        case ENodeType::WORKLOAD_CLEAR_DESTROY_STATUS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_WORKLOAD_CLEAR_DESTROY_STATUS;
        case ENodeType::PORTO_ADD_PROPERTIES_TO_CONTAINER_FAIL_REASON:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_ADD_PROPERTIES_TO_CONTAINER_FAIL_REASON;
        case ENodeType::PORTO_SET_PROPERTIES:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_SET_PROPERTIES;
        case ENodeType::SET_FILE_ACCESS_MODE_RECURSIVE:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_SET_FILE_ACCESS_MODE_RECURSIVE;
        case ENodeType::PORTO_STORAGE_EXISTS:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_STORAGE_EXISTS;
        case ENodeType::FILE_MODIFIED_BEFORE_PORTO_VOLUME_BUILD:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_FILE_MODIFIED_BEFORE_PORTO_VOLUME_BUILD;
        case ENodeType::IS_DIRECTORY:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_IS_DIRECTORY;
        case ENodeType::PORTO_ACTIVE_VERIFY_CONTAINERS_LIMIT_REACHED:
            return NLogEvent::TBehaviourTreeTickV2::ENodeType_PORTO_ACTIVE_VERIFY_CONTAINERS_LIMIT_REACHED;
    }

    ythrow yexception() << "Unknown node type '" << ToString(nodeType) << "'";
}

} // namespace NInfra::NPodAgent
