#include <infra/pod_agent/libs/behaviour/bt/core/tree.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/behaviour/loaders/proto/behavior3.pb.h>

#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/mock_client.h>

#include <google/protobuf/util/json_util.h>

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/generic/algorithm.h>
#include <util/stream/file.h>
#include <util/string/join.h>

namespace NInfra::NPodAgent::NTestBehavior3EditorJsonReader {

static TLogger logger({});


Y_UNIT_TEST_SUITE(Behavior3TestSuite) {

Y_UNIT_TEST(LoadOneNode) {
    TBehavior3 protoTree;
    {
        protoTree.set_title("A behavior tree");
        protoTree.set_root("1");
        TBehavior3Node root;
        {
            root.set_id("1");
            root.set_name("FeedbackObjectState");
            root.set_title("ELayerState_DOWNLOADING");
            google::protobuf::Value val;
            val.set_string_value("my_layer_download_hash");
            (*root.mutable_properties())["object_id_or_hash"] = val;
            val.set_string_value("ELayerState_DOWNLOADING");
            (*root.mutable_properties())["state"] = val;
            val.set_string_value("layer");
            (*root.mutable_properties())["object_type"] = val;
        }
        (*protoTree.mutable_nodes())["1"] = root;
    }

    TAsyncPortoClientPtr porto = new TAsyncPortoClient(new TMockPortoClient(), new TFakeThreadPool());
    TPosixWorkerPtr posixWorker = new TPosixWorker(new TFakeThreadPool());
    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));

    TTreePtr tree = new TTree(
        logger
        , "tree"
        , TBehavior3EditorJsonReader(protoTree)
            .WithPosixWorker(posixWorker)
            .WithPorto(porto)
            .WithLayerStatusRepository(statusRepository)
            .BuildRootNode()
    );
    auto tick = tree->Tick();
    UNIT_ASSERT_C(tick, "Some error on tick: " << tick.Error().Message);
    UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), tick.Success());
    UNIT_ASSERT_EQUAL(API::ELayerState_DOWNLOADING, statusRepository->GetObjectStatus("my_layer").state());
}

Y_UNIT_TEST(LoadWithGeneratorNode) {
    TBehavior3 protoTree;
    {
        protoTree.set_title("A behavior tree");
        protoTree.set_root("1");
        {
            TBehavior3Node root;
            {
                root.set_id("1");
                root.set_name("MemSequenceGenerator");
                root.set_title("");
                google::protobuf::Value val;
                val.set_string_value("ELayerState_DOWNLOADING;ELayerState_READY;");
                (*root.mutable_properties())["STATE"] = val;
                root.set_child("2");
            }
            (*protoTree.mutable_nodes())["1"] = root;
        }
        {
            TBehavior3Node child;
            {
                child.set_id("2");
                child.set_name("FeedbackObjectState");
                child.set_title("");
                google::protobuf::Value val;
                val.set_string_value("my_layer_download_hash");
                (*child.mutable_properties())["object_id_or_hash"] = val;
                val.set_string_value("<GNRT:STATE>");
                (*child.mutable_properties())["state"] = val;
                val.set_string_value("layer");
                (*child.mutable_properties())["object_type"] = val;
            }
            (*protoTree.mutable_nodes())["2"] = child;
        }
    }

    TAsyncPortoClientPtr porto = new TAsyncPortoClient(new TMockPortoClient(), new TFakeThreadPool());
    TPosixWorkerPtr posixWorker = new TPosixWorker(new TFakeThreadPool());
    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("my_layer", "my_layer_download_hash"));

    TTreePtr tree = new TTree(
        logger
        , "tree"
        , TBehavior3EditorJsonReader(protoTree)
            .WithPosixWorker(posixWorker)
            .WithPorto(porto)
            .WithLayerStatusRepository(statusRepository)
            .BuildRootNode()
    );
    auto tick = tree->Tick();
    UNIT_ASSERT_C(tick, "Some error on tick: " << tick.Error().Message);
    UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), tick.Success());
    UNIT_ASSERT_EQUAL(API::ELayerState_READY, statusRepository->GetObjectStatus("my_layer").state());
}

Y_UNIT_TEST(LoadFromFile) {
    struct TMyPortoClient : public TMockPortoClient {
        TExpected<TString, TPortoError>
        GetProperty(const TPortoContainerName&, EPortoContainerProperty property, int) override {
            if (property == EPortoContainerProperty::State) {
                return TString("dead");
            }
            return TExpected<TString, TPortoError>::DefaultSuccess();
        }
    };

    using namespace google::protobuf::util;
    TBehavior3 protoTree;

    JsonParseOptions options;
    options.ignore_unknown_fields = true;
    auto status = JsonStringToMessage(
            TFileInput(ArcadiaSourceRoot() + "/infra/pod_agent/libs/behaviour/loaders/ut/tree.json").ReadAll(),
            &protoTree, options);
    UNIT_ASSERT_C(status.ok(), status.message().ToString());

    TAsyncPortoClientPtr porto = new TAsyncPortoClient(new TMyPortoClient(), new TFakeThreadPool());
    TPosixWorkerPtr posixWorker = new TPosixWorker(new TFakeThreadPool());
    TLayerStatusRepositoryPtr statusRepository = new TLayerStatusRepository();
    statusRepository->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple("LayerId", "layer_download_hash"));

    TTemplateBTStorageConfig templateBtConfig;
    templateBtConfig.AddTreePaths(ArcadiaSourceRoot() + "/infra/pod_agent/libs/behaviour/loaders/ut/tree_verify.json");
    templateBtConfig.AddTreePaths(ArcadiaSourceRoot() + "/infra/pod_agent/libs/behaviour/loaders/ut/tree_check_create.json");
    TTreePtr tree = new TTree(
        logger
        , "tree"
        , TBehavior3EditorJsonReader(protoTree)
            .WithPosixWorker(posixWorker)
            .WithPorto(porto)
            .WithLayerStatusRepository(statusRepository)
            .WithTemplateBTStorage(new TTemplateBTStorage(templateBtConfig))
            .BuildRootNode()
    );

    auto tick = tree->Tick();
    UNIT_ASSERT_C(tick, "Some error on tick: " << tick.Error().Message);
    UNIT_ASSERT_EQUAL(TNodeSuccess(ENodeStatus::SUCCESS), tick.Success());
}

Y_UNIT_TEST(EscapeCharacters) {
    const TVector<TString> testStringList = {
        ";test;;"
        , "a;b;c;d3;e"
        , "test"
        , "test_\\\\;env_escaping\\;"
    };

    const TVector<TString> testStringAnsList = {
        "\\;test\\;\\;;"
        , "a\\;b\\;c\\;d3\\;e;"
        , "test;"
        , "test_\\\\\\;env_escaping\\\\;;"
    };

    UNIT_ASSERT_EQUAL(testStringList.size(), testStringAnsList.size());
    UNIT_ASSERT_EQUAL(TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(testStringList), JoinSeq("", testStringAnsList));
}


Y_UNIT_TEST(RemoveEscapeCharacters) {
    const TVector<TString> testStringList = {
        ";test;"
        , "a;b;d;d3;e;"
        , "test;"
        , ""
        , ";;;;;;;;;"
        , "\\;;"
        , ";;;;\\;test\\;;;\\;some\\; mo\\re\\;test\\;==;;=;"
        , "test\\\\;;;\\;\\;\\\\;;"
    };

    const TVector<TVector<TString>> testStringAnsList = {
        {"", "test"}
        , {"a", "b", "d", "d3", "e"}
        , {"test"}
        , {}
        , {9, ""}
        , {";"}
        , {"", "", "", "", ";test;", "",  ";some; mo\\re;test;==", "", "="}
        , {"test\\;", "", ";;\\;"}
    };

    UNIT_ASSERT_EQUAL(testStringList.size(), testStringAnsList.size());
    for (size_t i = 0; i < testStringList.size(); ++i) {
        UNIT_ASSERT_EQUAL(TBehavior3EditorJsonReader::SplitAndUnescapeCharactersForMemSequenceGenerator(testStringList[i]), testStringAnsList[i]);
    }
}

}

} //namespace{
