#include "behavior3_template_resolver.h"

#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NBehavior3TemplateResolverTest {

Y_UNIT_TEST_SUITE(Behavior3TemplateResolverTestSuite) {

Y_UNIT_TEST(ResolveTemplateSimpleReplace) {
    TBehavior3 templateTree;
    {
        templateTree.set_title("A behavior tree");
        templateTree.set_root("77252927-093c-4478-98ff-8fc3f2863bd2");
        TBehavior3Node root;
        {
            root.set_id("77252927-093c-4478-98ff-8fc3f2863bd2");
            root.set_name("FeedbackLayerState");
            root.set_title("ELayerState_DOWNLOADING");
            google::protobuf::Value val;
            val.set_string_value("<RSLV:LAYER_ID>"); (*root.mutable_properties())["layer_id"] = val;
            val.set_string_value("ELayerState_DOWNLOADING"); (*root.mutable_properties())["state"] = val;
        }
        (*templateTree.mutable_nodes())["77252927-093c-4478-98ff-8fc3f2863bd2"] = root;
    }

    TMap<TString, TString> replace;
    replace["LAYER_ID"] = "my_layer";

    TBehavior3TemplateResolver().Resolve(templateTree, replace);

    UNIT_ASSERT_EQUAL("my_layer", templateTree.Getnodes().at("77252927-093c-4478-98ff-8fc3f2863bd2").Getproperties().at("layer_id").string_value());
    UNIT_ASSERT_EQUAL("ELayerState_DOWNLOADING", templateTree.Getnodes().at("77252927-093c-4478-98ff-8fc3f2863bd2").Getproperties().at("state").string_value());
}

Y_UNIT_TEST(ResolveTemplateSubstring) {
    TBehavior3 templateTree;
    {
        templateTree.set_title("A behavior tree");
        templateTree.set_root("77252927-093c-4478-98ff-8fc3f2863bd2");
        TBehavior3Node root;
        {
            root.set_id("77252927-093c-4478-98ff-8fc3f2863bd2");
            root.set_name("FeedbackLayerState");
            root.set_title("ELayerState_DOWNLOADING");
            google::protobuf::Value val;
            val.set_string_value("some_prefix/<RSLV:LAYER_ID>/some_suffix"); (*root.mutable_properties())["layer_id"] = val;
            val.set_string_value("ELayerState_DOWNLOADING"); (*root.mutable_properties())["state"] = val;
        }
        (*templateTree.mutable_nodes())["77252927-093c-4478-98ff-8fc3f2863bd2"] = root;
    }

    TMap<TString, TString> replace;
    replace["LAYER_ID"] = "my_layer";

    TBehavior3TemplateResolver().Resolve(templateTree, replace);

    UNIT_ASSERT_EQUAL("some_prefix/my_layer/some_suffix", templateTree.Getnodes().at("77252927-093c-4478-98ff-8fc3f2863bd2").Getproperties().at("layer_id").string_value());
    UNIT_ASSERT_EQUAL("ELayerState_DOWNLOADING", templateTree.Getnodes().at("77252927-093c-4478-98ff-8fc3f2863bd2").Getproperties().at("state").string_value());
}

Y_UNIT_TEST(ResolveTemplateTitleSubstring) {
    TBehavior3 templateTree;
    {
        templateTree.set_title("A behavior tree");
        templateTree.set_root("77252927-093c-4478-98ff-8fc3f2863bd2");
        TBehavior3Node root;
        {
            root.set_id("77252927-093c-4478-98ff-8fc3f2863bd2");
            root.set_name("FeedbackLayerState");
            root.set_title("<RSLV:LAYER_ID>, ELayerState_DOWNLOADING");
            google::protobuf::Value val;
            val.set_string_value("some_prefix/<RSLV:LAYER_ID>/some_suffix"); (*root.mutable_properties())["layer_id"] = val;
            val.set_string_value("ELayerState_DOWNLOADING"); (*root.mutable_properties())["state"] = val;
        }
        (*templateTree.mutable_nodes())["77252927-093c-4478-98ff-8fc3f2863bd2"] = root;
    }

    TMap<TString, TString> replace;
    replace["LAYER_ID"] = "my_layer";

    TBehavior3TemplateResolver().Resolve(templateTree, replace);

    UNIT_ASSERT_EQUAL("my_layer, ELayerState_DOWNLOADING", templateTree.Getnodes().at("77252927-093c-4478-98ff-8fc3f2863bd2").Gettitle());
    UNIT_ASSERT_EQUAL("some_prefix/my_layer/some_suffix", templateTree.Getnodes().at("77252927-093c-4478-98ff-8fc3f2863bd2").Getproperties().at("layer_id").string_value());
    UNIT_ASSERT_EQUAL("ELayerState_DOWNLOADING", templateTree.Getnodes().at("77252927-093c-4478-98ff-8fc3f2863bd2").Getproperties().at("state").string_value());
}

Y_UNIT_TEST(ResolveTemplateBadSyntax) {
    TBehavior3 templateTree;
    {
        templateTree.set_title("A behavior tree");
        templateTree.set_root("77252927-093c-4478-98ff-8fc3f2863bd2");
        TBehavior3Node root;
        {
            root.set_id("77252927-093c-4478-98ff-8fc3f2863bd2");
            root.set_name("FeedbackLayerState");
            root.set_title("ELayerState_DOWNLOADING");
            google::protobuf::Value val;
            val.set_string_value("<RSLV:LAYER_ID/something_else"); (*root.mutable_properties())["layer_id"] = val;
            val.set_string_value("ELayerState_DOWNLOADING"); (*root.mutable_properties())["state"] = val;
        }
        (*templateTree.mutable_nodes())["77252927-093c-4478-98ff-8fc3f2863bd2"] = root;
    }

    TMap<TString, TString> replace;
    replace["LAYER_ID"] = "my_layer";

    UNIT_ASSERT_EXCEPTION_CONTAINS(TBehavior3TemplateResolver().Resolve(templateTree, replace), yexception, "Unended template");
}

Y_UNIT_TEST(ResolveTemplateNotFound) {
    TBehavior3 templateTree;
    {
        templateTree.set_title("A behavior tree");
        templateTree.set_root("77252927-093c-4478-98ff-8fc3f2863bd2");
        TBehavior3Node root;
        {
            root.set_id("77252927-093c-4478-98ff-8fc3f2863bd2");
            root.set_name("FeedbackLayerState");
            root.set_title("ELayerState_DOWNLOADING");
            google::protobuf::Value val;
            val.set_string_value("<RSLV:LAYER_ID>/something_else"); (*root.mutable_properties())["layer_id"] = val;
            val.set_string_value("ELayerState_DOWNLOADING"); (*root.mutable_properties())["state"] = val;
        }
        (*templateTree.mutable_nodes())["77252927-093c-4478-98ff-8fc3f2863bd2"] = root;
    }

    TMap<TString, TString> replace;
    replace["LAYER_IDS"] = "my_layer";

    UNIT_ASSERT_EXCEPTION_CONTAINS(TBehavior3TemplateResolver().Resolve(templateTree, replace), yexception, "not found at replace map");
}

Y_UNIT_TEST(ResolveTemplateNotFoundAtTitle) {
    TBehavior3 templateTree;
    {
        templateTree.set_title("A behavior tree");
        templateTree.set_root("77252927-093c-4478-98ff-8fc3f2863bd2");
        TBehavior3Node root;
        {
            root.set_id("77252927-093c-4478-98ff-8fc3f2863bd2");
            root.set_name("FeedbackLayerState");
            root.set_title("<RSLV:LAYER_ID>, ELayerState_DOWNLOADING");
            google::protobuf::Value val;
            val.set_string_value("something_else"); (*root.mutable_properties())["layer_id"] = val;
            val.set_string_value("ELayerState_DOWNLOADING"); (*root.mutable_properties())["state"] = val;
        }
        (*templateTree.mutable_nodes())["77252927-093c-4478-98ff-8fc3f2863bd2"] = root;
    }

    TMap<TString, TString> replace;
    replace["LAYER_IDS"] = "my_layer";

    UNIT_ASSERT_EXCEPTION_CONTAINS(TBehavior3TemplateResolver().Resolve(templateTree, replace), yexception, "not found at replace map");
}

} // namespace NInfra::NPodAgent::NBehavior3TemplateResolverTest

}
