#include "mtp_period_ticker.h"

#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/bt/render/proto_renderer.h>

namespace NInfra::NPodAgent {

void TMtpPeriodTicker::Start() {
    TGuard<TMutex> gMutex(Mutex_);
    Y_ENSURE(AtomicGet(NeedQuit_), "TMtpPeriodTicker already started");
    AtomicSet(NeedQuit_, false);
    Queue_.Start(PoolSize_ + 1); // +1 thread for Ticker()
    NThreading::Async([this]{this->Ticker();}, Queue_);
}

void TMtpPeriodTicker::Shutdown() {
    TGuard<TMutex> gMutex(Mutex_);
    AtomicSet(NeedQuit_, true);
    QuitSignal_.Signal();
}

void TMtpPeriodTicker::Wait() {
    Queue_.Stop();
}

bool TMtpPeriodTicker::HasTree(const TString& id) const {
    TReadGuardBase<TLightRWLock> guard(Lock_);
    return Trees_.contains(id);
}

TVector<TString> TMtpPeriodTicker::GetTreeIds() const {
    TReadGuardBase<TLightRWLock> guard(Lock_);
    TVector<TString> result;
    for (auto& it : Trees_) {
        result.push_back(it.first);
    }
    return result;
}

void TMtpPeriodTicker::AddTree(TTreePtr tree, const TString& treeHash) {
    const TString& id = tree->GetTreeId();
    TWriteGuardBase<TLightRWLock> guard(Lock_);
    if (Trees_.contains(id)) {
        Y_ENSURE(Trees_.at(id).TreeHash_ == treeHash, "TMtpPeriodTicker already has tree with id: '" + id + "' but with a different hash");
        ++Trees_[id].NumberOfOccurrences_;
    } else {
        Trees_[id].Tree_ = tree;
        Trees_[id].TreeHash_ = treeHash;
        Trees_[id].NumberOfOccurrences_ = 1;
        IncCounter("TreeCnt");
    }
}

void TMtpPeriodTicker::RemoveTree(const TString& id) {
    {
        TReadGuardBase<TLightRWLock> guard(Lock_);
        auto currentTree = Trees_.FindPtr(id);
        Y_ENSURE(currentTree, "TMtpPeriodTicker doesn't have tree with id: '" + id + "'");

        bool isRunning = true;
        {
            TGuard<TMutex> gMutex(currentTree->Mutex_);

            Y_ENSURE(currentTree->NumberOfOccurrences_ > 0, "Tree with id: '" + id + "' should already be removed");
            --currentTree->NumberOfOccurrences_;

            if (currentTree->NumberOfOccurrences_ != 0) {
                // We don't need to remove a tree from a tiсker if ticker contains tree nonzero number of times
                return;
            }

            isRunning = currentTree->IsRunning_;
        }

        for (size_t i = 0; i < TMtpPeriodTicker::MAX_REMOVE_TREE_WAIT_TRIES && isRunning; ++i) {
            {
                TGuard<TMutex> gMutex(currentTree->Mutex_);
                isRunning = currentTree->IsRunning_;
            }
            if (isRunning) {
                Sleep(TMtpPeriodTicker::REMOVE_TREE_WAIT_PERIOD);
            }
        }
    }

    {
        TWriteGuardBase<TLightRWLock> guard(Lock_);
        auto currentTree = Trees_.FindPtr(id);
        if (!currentTree) {
            // Tree already removed
            return;
        }

        {
            TGuard<TMutex> gMutex(currentTree->Mutex_);
            // Lock was removed and taken back, so we need to check this condition again
            if (currentTree->NumberOfOccurrences_ != 0) {
                // We don't need to remove a tree from a tiсker if ticker contains tree nonzero number of times
                return;
            }

            if (currentTree->IsRunning_) {
                NLogEvent::TBehaviourTreeTickError ev;
                ev.SetType("Timeout");
                ev.SetTreeId(id);
                ev.SetTitle(TStringBuilder() << "Tree is still in RUNNING state");
                CommonLogFrame_->LogEvent(ELogPriority::TLOG_CRIT, ev);
            }
        }

        Trees_.erase(id);
        DecCounter("TreeCnt");
    }
}

void TMtpPeriodTicker::TTreeHolder::FlushError(const TLogFramePtr& commonLogFrame, const TLogFramePtr& treeTraceLogFrame) {
    if (ErrorCount_ > SAME_ERROR_COUNT_LIMIT) {
        NLogEvent::TBehaviourTreeTickError ev;
        ev.SetType(ErrorType_);
        ev.SetTitle(TStringBuilder()
            << ErrorTitle_ << ": total " << ErrorCount_ << " times"
        );
        ev.SetTreeId(Tree_->GetTreeId());
        ev.SetTick(TickId_);
        commonLogFrame->LogEvent(ELogPriority::TLOG_CRIT, ev);
        treeTraceLogFrame->LogEvent(ELogPriority::TLOG_CRIT, ev);
    }
    ErrorCount_ = 0;
}

void TMtpPeriodTicker::TTreeHolder::LogError(const TString& type, const TString& title, const TLogFramePtr& commonLogFrame, const TLogFramePtr& treeTraceLogFrame) {
    if (type == ErrorType_ && title == ErrorTitle_) {
        ErrorCount_++;
    } else {
        FlushError(commonLogFrame, treeTraceLogFrame);
        ErrorType_ = type;
        ErrorTitle_ = title;
        ErrorCount_ = 1;
    }
    if (ErrorCount_ <= SAME_ERROR_COUNT_LIMIT || ErrorCount_ % SAME_ERROR_COUNT_LOG_INTERVAL == 0) {
        NLogEvent::TBehaviourTreeTickV2 ev = TProtoRenderer().Render(Tree_);
        ev.SetTick(TickId_);
        commonLogFrame->LogEvent(ELogPriority::TLOG_CRIT, ev);
        treeTraceLogFrame->LogEvent(ELogPriority::TLOG_CRIT, ev);
    } else {
        NLogEvent::TBehaviourTreeTickError ev;
        ev.SetType(ErrorType_);
        ev.SetTitle(TStringBuilder()
            << ErrorTitle_ << ": skip full tree render because error appears " << ErrorCount_ << " times"
        );
        ev.SetTreeId(Tree_->GetTreeId());
        ev.SetTick(TickId_);
        commonLogFrame->LogEvent(ELogPriority::TLOG_CRIT, ev);
        treeTraceLogFrame->LogEvent(ELogPriority::TLOG_CRIT, ev);
    }
}

void TMtpPeriodTicker::TTreeHolder::LogSuccess(const TLogFramePtr& treeTraceLogFrame) {
    NLogEvent::TBehaviourTreeTickV2 ev = TProtoRenderer().Render(Tree_);
    ev.SetTick(TickId_);
    treeTraceLogFrame->LogEvent(ELogPriority::TLOG_DEBUG, ev);
}

void TMtpPeriodTicker::TickTree(const TString& treeId, TTreeHolder& treeHolder, TInstant now) {
    TGuard<TMutex> guard(treeHolder.Mutex_);
    try {
        ++treeHolder.TickId_;
        TTickResult result = treeHolder.Tree_->Tick();
        treeHolder.LastTickTime_ = now;
        if (!result) {
            treeHolder.LogError(
                "Error",
                result.Error().Message,
                CommonLogFrame_,
                TreeTraceLogFrame_
            );
            IncCounter("ERROR");
        } else if (result.Success().Status == ENodeStatus::SUCCESS || result.Success().Status == ENodeStatus::RUNNING) {
            treeHolder.FlushError(CommonLogFrame_, TreeTraceLogFrame_);
            treeHolder.LogSuccess(TreeTraceLogFrame_);
            if (result.Success().Status == ENodeStatus::SUCCESS) {
                IncCounter("SUCCESS");
            } else {
                IncCounter("RUNNING");
            }
        } else {
            treeHolder.LogError(
                "Bad status",
                TStringBuilder() << ToString(result.Success().Status) << ':' << result.Success().Message,
                CommonLogFrame_,
                TreeTraceLogFrame_
            );
            IncCounter("FAILURE");
        }
        if (TickOnReadline_) {
            Cout << TConsoleRenderer(::isatty(fileno(stdout))).Render(treeHolder.Tree_);
        }
        treeHolder.IsRunning_ = result && (result.Success().Status == ENodeStatus::RUNNING);
    } catch (...) {
        NLogEvent::TBehaviourTreeTickError ev;
        ev.SetType("Exception");
        ev.SetTitle(CurrentExceptionMessage());
        ev.SetTreeId(treeId);
        ev.SetTick(treeHolder.TickId_);
        CommonLogFrame_->LogEvent(ELogPriority::TLOG_CRIT, ev);
        treeHolder.IsRunning_ = false;
    }
}

void TMtpPeriodTicker::Ticker() {
    while (!AtomicGet(NeedQuit_)) {
        {
            TReadGuardBase<TLightRWLock> guard(Lock_);
            TInstant now = TInstant::Now();
            TVector<NThreading::TFuture<void>> futures;
            for (auto it = Trees_.begin(); it != Trees_.end() && !AtomicGet(NeedQuit_); ++it) {
                TString treeId = it->first;
                TTreeHolder& treeHolder = it->second;
                TInstant nextTickTime = treeHolder.LastTickTime_ + TDuration::MilliSeconds(treeHolder.Tree_->GetUseLongTickPeriod()
                    ? Max(PeriodMs_, Min(LongPeriodMs_, treeHolder.Tree_->GetLongTickPeriodDuration()))
                    : PeriodMs_
                );
                if ((nextTickTime < now && treeHolder.NumberOfOccurrences_ != 0) || treeHolder.IsRunning_) {
                    futures.push_back(
                        NThreading::Async(
                            [now, treeId, &treeHolder, this] {
                                TickTree(treeId, treeHolder, now);
                            }, Queue_
                        ));
                }
            }
            for (auto& f : futures) {
                f.GetValueSync();
            }
        }
        {
            TGuard<TMutex> gMutex(Mutex_);
            if (TickOnReadline_) {
                Cout << "Waiting for newline to tick trees..." << Endl;
                Cin.ReadLine();
            } else if (!AtomicGet(NeedQuit_)) {
                QuitSignal_.WaitT(Mutex_, TDuration::MilliSeconds(PeriodMs_));
            }
        }
    }
    // wait for running trees
    bool someTreeRunning = true;
    for (size_t i = 0; i < 200 && someTreeRunning; ++i) {
        someTreeRunning = false;
        TInstant now = TInstant::Now();
        {
            TReadGuardBase<TLightRWLock> guard(Lock_);
            for (auto& it : Trees_) {
                TGuard<TMutex> gMutex(it.second.Mutex_);
                if (it.second.IsRunning_) {
                    someTreeRunning = true;
                    TickTree(it.first, it.second, now);
                }
            }
        }
        if (someTreeRunning) {
            Sleep(TDuration::MilliSeconds(PeriodMs_));
        }
    }
    if (someTreeRunning) {
        NLogEvent::TBehaviourTreeTickError ev;
        ev.SetType("Timeout");
        TStringBuilder title;
        title << "Some tree are still in RUNNING state:" << Endl;
        {
            TReadGuardBase<TLightRWLock> guard(Lock_);
            for (auto& it : Trees_) {
                TGuard<TMutex> gMutex(it.second.Mutex_);
                if (it.second.IsRunning_) {
                    title << it.second.Tree_->GetTreeId() << Endl;
                }
            }
        }
        ev.SetTitle(title);
        CommonLogFrame_->LogEvent(ELogPriority::TLOG_CRIT, ev);
    }
}

const TString& TMtpPeriodTicker::GetTreeHash(const TString& id) const {
    TReadGuardBase<TLightRWLock> readGuard(Lock_);
    TGuard<TMutex> treeGuard(Trees_.at(id).Mutex_);

    return Trees_.at(id).TreeHash_;
}

void TMtpPeriodTicker::InitSignals() {
    const TVector<TString> signalNames = {
        "SUCCESS"
        , "RUNNING"
        , "FAILURE"
        , "ERROR"
    };

    TMultiUnistat::Instance().DrillFloatHole(
        TMultiUnistat::ESignalNamespace::INFRA
        , COUNTER_PREFIX + "TreeCnt"
        , "aeev"
        , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
        , NUnistat::TStartValue(0)
        , EAggregationType::Sum
    );
    for (const auto& name : signalNames) {
        TMultiUnistat::Instance().DrillFloatHole(
            TMultiUnistat::ESignalNamespace::INFRA
            , COUNTER_PREFIX + name
            , "deee"
            , NUnistat::TPriority(TMultiUnistat::ESignalPriority::INFRA_INFO)
            , NUnistat::TStartValue(0)
            , EAggregationType::Sum
        );
    }
}

void TMtpPeriodTicker::AddToCounter(const TString& name, double val) {
    TString fullName = COUNTER_PREFIX + name;

    if (!TMultiUnistat::Instance().PushSignalUnsafe(TMultiUnistat::ESignalNamespace::INFRA, fullName, val)) {
        CommonLogFrame_->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TMtpPeriodTickerSignalError(fullName, "PushSingalUnsafe(" + fullName + ", " + ToString(val) + ") returned false."));
    }
}

void TMtpPeriodTicker::IncCounter(const TString& name) {
    AddToCounter(name, 1);
}

void TMtpPeriodTicker::DecCounter(const TString& name) {
    AddToCounter(name, -1);
}

} // namespace NInfra::NPodAgent
