#pragma once

#include <infra/pod_agent/libs/behaviour/bt/core/tree.h>
#include <infra/pod_agent/libs/config/config.pb.h>
#include <infra/pod_agent/libs/multi_unistat/multi_unistat.h>

#include <library/cpp/threading/light_rw_lock/lightrwlock.h>

#include <util/generic/map.h>
#include <util/system/condvar.h>

namespace NInfra::NPodAgent {

/**
    Thread-safe ticker for multiple behavior trees
    Ticks every N milliseconds
    Uses its own MtpQueue
    Method Wait() waits for signal from Shutdown() and stops the MtpQueue
    Method Stop() calls {Shutdown(); Wait();}
*/
class TMtpPeriodTicker: public TAtomicRefCount<TMtpPeriodTicker> {
public:
    TMtpPeriodTicker(const TBehaviorTickerConfig& config, TLogFramePtr commonLogFrame, TLogFramePtr treeTraceLogFrame)
        : NeedQuit_(true)
        , PeriodMs_(config.GetTreeTickPeriodMs())
        , LongPeriodMs_(config.GetTreeLongTickPeriodMs())
        , PoolSize_(config.GetPoolSize())
        , TickOnReadline_(config.GetTickOnReadline())
        , CommonLogFrame_(commonLogFrame)
        , TreeTraceLogFrame_(treeTraceLogFrame)
    {
        InitSignals();
    }

    ~TMtpPeriodTicker() {
        try {
            Stop();
        } catch (...) {
        }
    }

    void Start();

    void Stop() {
        Shutdown();
        Wait();
    }

    void Shutdown();

    void Wait();

    bool HasTree(const TString& id) const;

    TVector<TString> GetTreeIds() const;

    void AddTree(TTreePtr tree, const TString& treeHash);

    void RemoveTree(const TString& id);

    const TString& GetTreeHash(const TString& id) const;

private:
    class TTreeHolder;

    void TickTree(const TString& treeId, TTreeHolder& treeHolder, TInstant now);

    void Ticker();

    void InitSignals();

    void AddToCounter(const TString& name, double val);

    void IncCounter(const TString& name);

    void DecCounter(const TString& name);

private:
    class TTreeHolder {
    public:
        void LogError(const TString& type, const TString& title, const TLogFramePtr& commonLogFrame, const TLogFramePtr& treeTraceLogFrame);
        void FlushError(const TLogFramePtr& commonLogFrame, const TLogFramePtr& treeTraceLogFrame);
        void LogSuccess(const TLogFramePtr& treeTraceLogFrame);
    public:
        TMutex Mutex_;
        TTickId TickId_ = 0;
        TTreePtr Tree_{nullptr};
        TString TreeHash_{};
        bool IsRunning_ = false;
        TInstant LastTickTime_{};
        ui32 NumberOfOccurrences_ = 0;

    private:
        static constexpr size_t SAME_ERROR_COUNT_LIMIT = 10;
        static constexpr size_t SAME_ERROR_COUNT_LOG_INTERVAL = 1000;
    
    private:
        TString ErrorType_{};
        TString ErrorTitle_{};
        size_t ErrorCount_{0};
    };

private:
    TAtomic NeedQuit_;
    TCondVar QuitSignal_;
    TMutex Mutex_;
    TLightRWLock Lock_;

    TThreadPool Queue_;
    TMap<TString, TTreeHolder> Trees_;
    size_t PeriodMs_;
    size_t LongPeriodMs_;
    size_t PoolSize_;
    bool TickOnReadline_;
    TLogFramePtr CommonLogFrame_;
    TLogFramePtr TreeTraceLogFrame_;
    

    static inline const TString COUNTER_PREFIX = "pod_agent_mtp_period_ticker_";

    // 1 second per wait
    static constexpr size_t MAX_REMOVE_TREE_WAIT_TRIES = 10;
    static inline const TDuration REMOVE_TREE_WAIT_PERIOD = TDuration::MilliSeconds(100);
};

using TMtpPeriodTickerPtr = TIntrusivePtr<TMtpPeriodTicker>;

} // namespace NInfra::NPodAgent
