#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_template_resolver.h>
#include <infra/pod_agent/libs/behaviour/trees/base/test/test_canon.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/nested_client.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/client_with_retries.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/folder/dirut.h>
#include <util/system/user.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestTreeCheckAndCreateContainerCanon: public ITestBehaviourTreeCanon {
public:
    ITestTreeCheckAndCreateContainerCanon(const TString &testName)
        : ITestBehaviourTreeCanon(testName, "TreeCheckAndCreateContainer", "TreeCheckAndCreateContainer")
    {
    }

    virtual ~ITestTreeCheckAndCreateContainerCanon() = default;

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["CONTAINER_NAME"] = CONTAINER_NAME;
        replace["CONTAINER_PLACE"] = CONTAINER_PLACE;
        replace["CONTAINER_CMD"] = CONTAINER_CMD;
        replace["CONTAINER_CWD"] = CONTAINER_CWD;
        replace["CONTAINER_ROOT"] = CONTAINER_ROOT;

        replace["CONTAINER_MEMORY_GUARANTEE"] = CONTAINER_MEMORY_GUARANTEE;
        replace["CONTAINER_MEMORY_LIMIT"] = CONTAINER_MEMORY_LIMIT;
        replace["CONTAINER_ANON_LIMIT"] = CONTAINER_ANON_LIMIT;
        replace["CONTAINER_RECHARGE_ON_PGFAULT"] = CONTAINER_RECHARGE_ON_PGFAULT;

        replace["CONTAINER_CPU_GUARANTEE"] = CONTAINER_CPU_GUARANTEE;
        replace["CONTAINER_CPU_LIMIT"] = CONTAINER_CPU_LIMIT;
        replace["CONTAINER_CPU_POLICY"] = CONTAINER_CPU_POLICY;
        replace["CONTAINER_CPU_WEIGHT"] = CONTAINER_CPU_WEIGHT;

        replace["CONTAINER_THREAD_LIMIT"] = CONTAINER_THREAD_LIMIT;

        replace["CONTAINER_ENVIRONMENT"] = CONTAINER_ENVIRONMENT;
        replace["CONTAINER_SECRET_ENVIRONMENT"] = CONTAINER_SECRET_ENVIRONMENT;
        replace["CONTAINER_HOSTNAME"] = CONTAINER_HOSTNAME;
        replace["CONTAINER_CAPABILITIES_AMBIENT"] = CONTAINER_CAPABILITIES_AMBIENT;
        replace["CONTAINER_ULIMIT"] = CONTAINER_ULIMIT;
        replace["CONTAINER_ISOLATE"] = CONTAINER_ISOLATE;
        replace["CONTAINER_RESOLV_CONF"] = CONTAINER_RESOLV_CONF;
        replace["CONTAINER_CORE_COMMAND"] = CONTAINER_CORE_COMMAND;
        replace["CONTAINER_AGING_TIME"] = CONTAINER_AGING_TIME;

        replace["CONTAINER_USER"] = CONTAINER_USER;
        replace["CONTAINER_GROUP"] = CONTAINER_GROUP;

        replace["CONTAINER_STDOUT_LOG_PATH"] = CONTAINER_STDOUT_LOG_PATH;
        replace["CONTAINER_STDERR_LOG_PATH"] = CONTAINER_STDERR_LOG_PATH;
        replace["CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE;
        replace["CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE;
        replace["CONTAINER_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = CONTAINER_STDOUT_AND_STDERR_FILE_SIZE_LIMIT;

        replace["CONTAINER_IO_LIMIT"] = CONTAINER_IO_LIMIT;
        replace["CONTAINER_IO_OPS_LIMIT"] = CONTAINER_IO_OPS_LIMIT;
        replace["CONTAINER_IO_POLICY"] = CONTAINER_IO_POLICY;
        replace["CONTAINER_IO_WEIGHT"] = CONTAINER_IO_WEIGHT;

        replace["CONTAINER_OBJECT_TYPE"] = CONTAINER_OBJECT_TYPE;
        replace["CONTAINER_OBJECT_ID_OR_HASH"] = CONTAINER_OBJECT_ID_OR_HASH;
        replace["CONTAINER_TYPE"] = CONTAINER_TYPE;
        replace["CONTAINER_INIT_NUM"] = "";
        replace["CONTAINER_SECRET_PROPERTIES_LIST"] = "env;";
        replace["CONTAINER_ENABLE_PORTO"] = "";
        replace["CONTAINER_GROUP_ID"] = "";
        replace["CONTAINER_CGROUP_FS_MOUNT_TYPE"] = CONTAINER_CGROUPFS;
        replace["TREE_HASH"] = "tree_hash";

        return replace;
    }

protected:
    inline static const TString CONTAINER_NAME = "ContainerName";

    inline static const TString CONTAINER_PLACE = "/place"; // Differs from the default value "/place;***"
    inline static const TString CONTAINER_CMD = "ls";
    inline static const TString CONTAINER_CWD = RealPath(GetWorkPath());
    inline static const TString CONTAINER_ROOT = "/";
    inline static const TString CONTAINER_MEMORY_GUARANTEE = ToString((1 << 20) + 1);
    inline static const TString CONTAINER_MEMORY_LIMIT = ToString((1 << 20) + 2);
    inline static const TString CONTAINER_ANON_LIMIT = ToString((1 << 20) + 3);
    inline static const TString CONTAINER_RECHARGE_ON_PGFAULT = "true";
    inline static const TString CONTAINER_CPU_GUARANTEE = "1c";
    inline static const TString CONTAINER_CPU_LIMIT = "2c";
    inline static const TString CONTAINER_CPU_POLICY = "high";
    inline static const TString CONTAINER_CPU_WEIGHT = "1.01";
    inline static const TString CONTAINER_THREAD_LIMIT = "1001";
    inline static const TString CONTAINER_ENVIRONMENT = "name=value";
    inline static const TString CONTAINER_SECRET_ENVIRONMENT = "secret_name=value_value";
    inline static const TString CONTAINER_SECRET_ENVIRONMENT_WITH_HIDDEN_VALUE = "secret_name=<secret";
    inline static const TString CONTAINER_ULIMIT = "core: unlimited unlimited; ";
    inline static const TString CONTAINER_ISOLATE = "false";
    inline static const TString CONTAINER_CORE_COMMAND = "echo core dumped";
    inline static const TString CONTAINER_USER = GetUsername();
    inline static const TString CONTAINER_GROUP = "porto";
    inline static const TString CONTAINER_OBJECT_TYPE = "layer";
    inline static const TString CONTAINER_OBJECT_ID = "LayerId";
    inline static const TString CONTAINER_OBJECT_ID_OR_HASH = "LayerDownloadHash";
    inline static const TString CONTAINER_TYPE = "download";
    inline static const TString CONTAINER_HOSTNAME = "hostname";
    inline static const TString CONTAINER_CAPABILITIES_AMBIENT = "NET_BIND_SERVICE";
    inline static const TString CONTAINER_STDOUT_LOG_PATH = "stdout_log_path";
    inline static const TString CONTAINER_STDERR_LOG_PATH = "stderr_log_path";
    inline static const TString CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE = "stdout_log_file_to_create";
    inline static const TString CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE = "stderr_log_file_to_create";
    inline static const TString CONTAINER_STDOUT_AND_STDERR_FILE_SIZE_LIMIT = ToString((1 << 20) + 4);
    inline static const TString CONTAINER_RESOLV_CONF = "keep";
    inline static const TString CONTAINER_AGING_TIME = ToString((1 << 16) + 1);
    inline static const TString CONTAINER_IO_LIMIT = "/tmp r: 20000";
    inline static const TString CONTAINER_IO_OPS_LIMIT = "/tmp r: 20001";
    inline static const TString CONTAINER_IO_POLICY = "high";
    inline static const TString CONTAINER_IO_WEIGHT = "1.02";
    inline static const TString CONTAINER_CGROUPFS = "ro";

    inline static const TMap<EPortoContainerProperty, TString> ALL_PROPERTIES = {
        {EPortoContainerProperty::Place, CONTAINER_PLACE}
        , {EPortoContainerProperty::Command, CONTAINER_CMD}
        , {EPortoContainerProperty::Cwd, CONTAINER_CWD}
        , {EPortoContainerProperty::Root, CONTAINER_ROOT}
        , {EPortoContainerProperty::MemoryGuarantee, CONTAINER_MEMORY_GUARANTEE}
        , {EPortoContainerProperty::MemoryLimit, CONTAINER_MEMORY_LIMIT}
        , {EPortoContainerProperty::AnonLimit, CONTAINER_ANON_LIMIT}
        , {EPortoContainerProperty::RechPgfault, CONTAINER_RECHARGE_ON_PGFAULT}
        , {EPortoContainerProperty::CpuGuarantee, CONTAINER_CPU_GUARANTEE}
        , {EPortoContainerProperty::CpuLimit, CONTAINER_CPU_LIMIT}
        , {EPortoContainerProperty::CpuPolicy, CONTAINER_CPU_POLICY}
        , {EPortoContainerProperty::CpuWeight, CONTAINER_CPU_WEIGHT}
        , {EPortoContainerProperty::ThreadLimit, CONTAINER_THREAD_LIMIT}
        , {EPortoContainerProperty::Env, CONTAINER_ENVIRONMENT}
        , {EPortoContainerProperty::EnvSecret, CONTAINER_SECRET_ENVIRONMENT_WITH_HIDDEN_VALUE}
        , {EPortoContainerProperty::Isolate, CONTAINER_ISOLATE}
        , {EPortoContainerProperty::CoreCommand, CONTAINER_CORE_COMMAND}
        , {EPortoContainerProperty::User, CONTAINER_USER}
        , {EPortoContainerProperty::Group, CONTAINER_GROUP}
        , {EPortoContainerProperty::Ulimit, CONTAINER_ULIMIT}
        , {EPortoContainerProperty::Hostname, CONTAINER_HOSTNAME}
        , {EPortoContainerProperty::CapabilitiesAmbient, CONTAINER_CAPABILITIES_AMBIENT}
        , {EPortoContainerProperty::StdOutPath, CONTAINER_STDOUT_LOG_PATH}
        , {EPortoContainerProperty::StdErrPath, CONTAINER_STDERR_LOG_PATH}
        , {EPortoContainerProperty::StdOutAndStdErrLimit, CONTAINER_STDOUT_AND_STDERR_FILE_SIZE_LIMIT}
        , {EPortoContainerProperty::ResolvConf, CONTAINER_RESOLV_CONF}
        , {EPortoContainerProperty::AgingTime, CONTAINER_AGING_TIME}
        , {EPortoContainerProperty::IoLimit, CONTAINER_IO_LIMIT}
        , {EPortoContainerProperty::IoOpsLimit, CONTAINER_IO_OPS_LIMIT}
        , {EPortoContainerProperty::IoPolicy, CONTAINER_IO_POLICY}
        , {EPortoContainerProperty::IoWeight, CONTAINER_IO_WEIGHT}
        , {EPortoContainerProperty::Cgroupfs, CONTAINER_CGROUPFS}
    };
};

Y_UNIT_TEST_SUITE(TreeCheckAndCreateTestSuite) {

Y_UNIT_TEST(TestCreateContainer) {
    class TTest : public ITestTreeCheckAndCreateContainerCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeCheckAndCreateContainerCanon(testName)
        {
        }

    protected:
        void Test() override {
            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(CONTAINER_OBJECT_ID, CONTAINER_OBJECT_ID_OR_HASH));

            TVector<EPortoContainerProperty> keys;
            for (const auto& it : ALL_PROPERTIES) {
                keys.push_back(it.first);
            }

            auto breakHook = [this, &keys]() {
                auto portoResult = SafePorto_->Get({CONTAINER_NAME}, keys);
                if (!portoResult) {
                    return false;
                }
                TMap<EPortoContainerProperty, TPortoGetResponse> properties = portoResult.Success()[CONTAINER_NAME];
                bool result = true;
                for (const auto& it : ALL_PROPERTIES) {
                    if (it.first == EPortoContainerProperty::EnvSecret) {
                        // secrets have salt and md5, let's check only prefix
                        result &= TString(properties[it.first].value()).StartsWith(it.second);
                    } else {
                        result &= (it.second == properties[it.first].value());
                    }
                }

                result &= NFs::Exists(CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE);
                result &= NFs::Exists(CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE);
                return result;
            };

            TickTree(Tree_, 4, breakHook);
            auto result = SafePorto_->Get({CONTAINER_NAME}, keys);
            UNIT_ASSERT_C(result, result.Error().Message);
            UNIT_ASSERT_C(result.Success().contains(CONTAINER_NAME), "no key, map size: " << result.Success().size());
            TMap<EPortoContainerProperty, TPortoGetResponse> properties = result.Success()[CONTAINER_NAME];
            for (const auto& it : ALL_PROPERTIES) {
                if (it.first == EPortoContainerProperty::EnvSecret) {
                    // secrets have salt and md5, let's check only substring
                    UNIT_ASSERT_STRING_CONTAINS(properties[it.first].value(), it.second);
                } else {
                    UNIT_ASSERT_EQUAL_C(it.second, properties[it.first].value(), it.second << " :: " << properties[it.first].value());
                }
            }

            UNIT_ASSERT_EQUAL_C(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message(), "", LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message());
            UNIT_ASSERT_EQUAL_C(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter(), 0, LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter());

            UNIT_ASSERT(NFs::Exists(CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE));
            UNIT_ASSERT(NFs::Exists(CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE));
        }
    };

    TTest test("TestCreateContainer");
    test.DoTest();
}

Y_UNIT_TEST(TestReCreateContainerIfLogFilesDeleted) {
    class TTest : public ITestTreeCheckAndCreateContainerCanon {
    public:
        TTest(const TString& testName)
                : ITestTreeCheckAndCreateContainerCanon(testName)
        {
        }

    protected:
        void Test() override {
            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(CONTAINER_OBJECT_ID, CONTAINER_OBJECT_ID_OR_HASH));

            TVector<EPortoContainerProperty> keys;
            for (const auto& it : ALL_PROPERTIES) {
                keys.push_back(it.first);
            }

            auto breakHook = [this, &keys]() {
                auto portoResult = SafePorto_->Get({CONTAINER_NAME}, keys);
                if (!portoResult) {
                    return false;
                }
                TMap<EPortoContainerProperty, TPortoGetResponse> properties = portoResult.Success()[CONTAINER_NAME];
                bool result = true;
                for (const auto& it : ALL_PROPERTIES) {
                    if (it.first == EPortoContainerProperty::EnvSecret) {
                        // secrets have salt and md5, let's check only prefix
                        result &= TString(properties[it.first].value()).StartsWith(it.second);
                    } else {
                        result &= (it.second == properties[it.first].value());
                    }
                }

                result &= NFs::Exists(CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE);
                result &= NFs::Exists(CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE);
                return result;
            };

            TickTree(Tree_, 4, breakHook);
            auto result = SafePorto_->Get({CONTAINER_NAME}, keys);
            UNIT_ASSERT_C(result, result.Error().Message);
            UNIT_ASSERT_C(result.Success().contains(CONTAINER_NAME), "no key, map size: " << result.Success().size());
            TMap<EPortoContainerProperty, TPortoGetResponse> properties = result.Success()[CONTAINER_NAME];
            for (const auto& it : ALL_PROPERTIES) {
                if (it.first == EPortoContainerProperty::EnvSecret) {
                    // secrets have salt and md5, let's check only substring
                    UNIT_ASSERT_STRING_CONTAINS(properties[it.first].value(), it.second);
                } else {
                    UNIT_ASSERT_EQUAL_C(it.second, properties[it.first].value(), it.second << " :: " << properties[it.first].value());
                }
            }

            UNIT_ASSERT_EQUAL_C(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message(), "", LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message());
            UNIT_ASSERT_EQUAL_C(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter(), 0, LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter());

            UNIT_ASSERT(NFs::Exists(CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE));
            UNIT_ASSERT(NFs::Exists(CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE));

            NFs::Remove(CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE);
            NFs::Remove(CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE);

            TickTree(Tree_, 4, breakHook);

            UNIT_ASSERT(NFs::Exists(CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE));
            UNIT_ASSERT(NFs::Exists(CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE));
        }
    };

    TTest test("TestReCreateContainerIfLogFilesDeleted");
    test.DoTest();
}

Y_UNIT_TEST(TestUpdateDeadContainerWithOneTick) {
    class TTest : public ITestTreeCheckAndCreateContainerCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeCheckAndCreateContainerCanon(testName)
        {
        }

    protected:
        void Test() override {
            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(CONTAINER_OBJECT_ID, CONTAINER_OBJECT_ID_OR_HASH));

            SafePorto_->Create(CONTAINER_NAME).Success();
            SafePorto_->SetProperty(CONTAINER_NAME, EPortoContainerProperty::Command, CONTAINER_CMD + " other").Success();
            SafePorto_->SetProperty(CONTAINER_NAME, EPortoContainerProperty::Private, ":tree_hash").Success();
            SafePorto_->Start(CONTAINER_NAME);

            auto breakHookTrue = []() {
                return true;
            };

            TickTree(Tree_, 1, breakHookTrue);

            auto result = SafePorto_->Get(
                {CONTAINER_NAME}
                , {
                    EPortoContainerProperty::Command
                }
            );
            UNIT_ASSERT_C(result, result.Error().Message);
            UNIT_ASSERT_C(result.Success().contains(CONTAINER_NAME), "no key, map size: " << result.Success().size());
            TMap<EPortoContainerProperty, TPortoGetResponse> properties = result.Success()[CONTAINER_NAME];
            UNIT_ASSERT_EQUAL_C(0, properties[EPortoContainerProperty::Command].error(), properties[EPortoContainerProperty::Command].errormsg());
            UNIT_ASSERT_EQUAL_C(CONTAINER_CMD, properties[EPortoContainerProperty::Command].value(), properties[EPortoContainerProperty::Command].value());

            UNIT_ASSERT_EQUAL_C(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message(), "", LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message());
            UNIT_ASSERT_EQUAL_C(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter(), 0, LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter());
        }
    };

    TTest test("TestUpdateDeadContainerWithOneTick");
    test.DoTest();
}

Y_UNIT_TEST(TestUpdateContainerWithWrongHash) {
    class TTest : public ITestTreeCheckAndCreateContainerCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeCheckAndCreateContainerCanon(testName)
        {
        }

    protected:
        void Test() override {
            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(CONTAINER_OBJECT_ID, CONTAINER_OBJECT_ID_OR_HASH));

            SafePorto_->Create(CONTAINER_NAME).Success();
            SafePorto_->SetProperty(CONTAINER_NAME, EPortoContainerProperty::Command, CONTAINER_CMD).Success();
            SafePorto_->SetProperty(CONTAINER_NAME, EPortoContainerProperty::Private, ":wrong_hash").Success();

            auto breakHookTrue = []() {
                return true;
            };

            TickTree(Tree_, 1, breakHookTrue);

            auto result = SafePorto_->Get(
                {CONTAINER_NAME}
                , {
                    EPortoContainerProperty::Command
                }
            );
            UNIT_ASSERT_C(result, result.Error().Message);
            UNIT_ASSERT_C(result.Success().contains(CONTAINER_NAME), "no key, map size: " << result.Success().size());
            TMap<EPortoContainerProperty, TPortoGetResponse> properties = result.Success()[CONTAINER_NAME];
            UNIT_ASSERT_EQUAL_C(0, properties[EPortoContainerProperty::Command].error(), properties[EPortoContainerProperty::Command].errormsg());
            UNIT_ASSERT_EQUAL_C(CONTAINER_CMD, properties[EPortoContainerProperty::Command].value(), properties[EPortoContainerProperty::Command].value());

            UNIT_ASSERT_EQUAL_C(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message(), "", LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message());
            UNIT_ASSERT_EQUAL_C(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter(), 0, LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter());
        }
    };

    TTest test("TestUpdateContainerWithWrongHash");
    test.DoTest();
}

Y_UNIT_TEST(TestPortoCreateRecursiveFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client)
            : TWrapperPortoClient(client)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)
        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestTreeCheckAndCreateContainerCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeCheckAndCreateContainerCanon(testName)
        {
        }

    protected:
        void Test() override {
            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(CONTAINER_OBJECT_ID, CONTAINER_OBJECT_ID_OR_HASH));

            TTickId tickId = 0;
            TTickResult tick = Tree_->Tick();
            while (tick && tick.Success().Status != ENodeStatus::FAILURE && tickId < 10) {
                Sleep(TDuration::MilliSeconds(500));
                tick = Tree_->Tick();
                ++tickId;
            }

            UNIT_ASSERT_C(tick, tick.Error().Message);
            UNIT_ASSERT_EQUAL(tick.Success().Status, ENodeStatus::FAILURE);

            UNIT_ASSERT_STRING_CONTAINS(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message(), CONTAINER_NAME);
            UNIT_ASSERT_STRING_CONTAINS(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).failed().message(), CONTAINER_TYPE);
            UNIT_ASSERT_STRING_CONTAINS(tick.Success().Message, TStringBuilder() << "Can't create " << CONTAINER_NAME << " container");
            UNIT_ASSERT_EQUAL_C(LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter(), 1, LayerStatusRepository_->GetObjectStatus(CONTAINER_OBJECT_ID).fail_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(CONTAINER_NAME), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_EQUAL_C(1, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls(), ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto);
        }
    };

    TTest test("TestPortoCreateRecursiveFail");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
