#include "box_test_canon.h"

#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>
#include <util/system/fstat.h>
#include <util/system/shellcommand.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestBoxTreeLinkedAndRbindVolumes: public ITestBoxCanon {
public:
    ITestBoxTreeLinkedAndRbindVolumes(const TString& testName)
        : ITestBoxCanon(testName, "BoxTreeLinkedAndRbindVolumes")
    {
    }
};

Y_UNIT_TEST_SUITE(BoxTreeLinkedAndRbindVolumesTestSuite) {

Y_UNIT_TEST(TestCleanVolumes) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            TString rootfsPath = PathHolder_->GetBoxRootfsPath(BoxId_);
            TVector<TString> volumePaths = {
                rootfsPath + "/new_folder"
                , rootfsPath + "/new_folder1/new_subfolder1"
                , rootfsPath + "/new_folder1/new_subfolder2"
            };

            for (auto& volumePath : volumePaths) {
                NFs::MakeDirectoryRecursive(volumePath);
                SafePorto_->CreateVolume(
                    volumePath
                    , ""
                    , ""
                    , {}
                    , 0
                    , ITestBoxCanon::CORRECT_VOLUME_HASH
                    , EPortoVolumeBackend::Auto
                    , {""}
                    , {}
                    , false
                ).Success();
            }

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 24, breakHook);

            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
            for (auto& volumePath : volumePaths) {
                UNIT_ASSERT_C(!(SafePorto_->IsVolumeExists(volumePath).Success()), volumePath << " exists");
            }
        }
    };

    TTest test("TestCleanVolumes");
    test.DoTest();
}

Y_UNIT_TEST(TestWaitingForLinkedVolumes) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_VOLUMES;
            };

            TickTree(Tree_, 60, breakHook);

            const auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EBoxState_WAITING_FOR_VOLUMES, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(status.failed().message(), "", Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWaitingForLinkedVolumes");
    test.DoTest();
}

Y_UNIT_TEST(TestWaitingForRbindVolumes) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareStaticResource();

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_VOLUMES
                    && BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message() != ""
                ;
            };

            TickTree(Tree_, 60, breakHook);

            const auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EBoxState_WAITING_FOR_VOLUMES, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "rbind volume storage", Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "does not exist", Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWaitingForRbindVolumes");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveLinkedVolumes) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            auto breakHookWaitingForVolumes = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_VOLUMES;
            };

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(MountVolumeId_)).Success();

            TickTree(Tree_, 20, breakHookWaitingForVolumes);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_VOLUMES, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareVolume();

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestRemoveLinkedVolumes");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveRbindVolumeStorage) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            auto breakHookWaitingForVolumes = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_VOLUMES
                    && BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message() != ""
                ;
            };

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            NFs::RemoveRecursive(PathHolder_->GetRbindVolumeStorage(RbindVolumeRef_));

            TickTree(Tree_, 20, breakHookWaitingForVolumes);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_VOLUMES, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message(), "rbind volume storage", Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareRbindVolumeStorage();

            UNIT_ASSERT_UNEQUAL_C(
                TFileStat(PathHolder_->GetRbindVolumeStorage(RbindVolumeRef_)).INode
                , TFileStat(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH).INode
                , "Rbind volume storage and porto volume must have different INode here"
            );

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_EQUAL_C(
                TFileStat(PathHolder_->GetRbindVolumeStorage(RbindVolumeRef_)).INode
                , TFileStat(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH).INode
                , "Rbind volume storage and porto volume must have same INode here"
            );
        }
    };

    TTest test("TestRemoveRbindVolumeStorage");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveLinkedAndRbindVolumesWithInit) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareRootfsLayer();
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            auto breakHookWaitingForVolumes = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_VOLUMES;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits_size(), 1, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits(0).last().state(), API::EContainerState_EXITED, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(MountVolumeId_)).Success();
            SafePorto_->UnlinkVolume(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH).Success();
            NFs::RemoveRecursive(PathHolder_->GetRbindVolumeStorage(RbindVolumeRef_));

            TickTree(Tree_, 32, breakHookWaitingForVolumes);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_VOLUMES, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareVolume();
            PrepareRbindVolumeStorage();

            TickTree(Tree_, 32, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits_size(), 1, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits(0).last().state(), API::EContainerState_EXITED, Endl << TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_C(NFs::Exists(PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT + "/init_mount.txt"), Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(NFs::Exists(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH + "/init_rbind.txt"), Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["INIT_CMD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(
                {
                    "bash -c 'echo init_mount > " + MOUNT_POINT + "/init_mount.txt; echo init_rbind > " + RBIND_VOLUME_MOUNT_PATH + "/init_rbind.txt'"
                }
            );
            replace["INIT_CWD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_NUM_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"0"});
            replace["INIT_CONTAINER_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetBoxInitContainer(BoxId_, 0)
            });
            replace["INIT_CONTAINER_PREVIOUS_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                ""
            });
            replace["INIT_ENVIRONMENT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_SECRET_ENVIRONMENT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});

            replace["INIT_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_STDERR_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_STDOUT_LOG_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_STDERR_LOG_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_STDOUT_AND_STDERR_FILE_SIZE_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});

            replace["INIT_CPU_GUARANTEE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1c"});
            replace["INIT_CPU_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1c"});
            replace["INIT_CPU_POLICY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "normal"});
            replace["INIT_CPU_WEIGHT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1.01"});
            replace["INIT_MEMORY_GUARANTEE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ToString(1 << 25)});
            replace["INIT_MEMORY_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ToString(1 << 25)});
            replace["INIT_ANON_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ToString(1 << 25)});
            replace["INIT_RECHARGE_ON_PGFAULT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "false"});
            replace["INIT_THREAD_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1001"});
            replace["INIT_ULIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ""});
            replace["INIT_CORE_COMMAND_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ""});
            replace["INIT_USER_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ""});
            replace["INIT_GROUP_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ""});
            replace["INIT_AGING_TIME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ToString(1 << 16)});
            replace["INIT_IO_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "/tmp r: 20000"});
            replace["INIT_IO_OPS_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "/tmp r: 20001"});
            replace["INIT_IO_POLICY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "normal"});
            replace["INIT_IO_WEIGHT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1.02"});

            replace["INIT_INITIAL_DELAY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "0"});
            replace["INIT_RESTART_PERIOD_SCALE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "0"});
            replace["INIT_RESTART_PERIOD_BACKOFF_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1"});
            replace["INIT_MAX_RESTART_PERIOD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1000"});
            replace["INIT_MIN_RESTART_PERIOD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1000"});
            replace["INIT_MAX_EXECUTION_TIME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "60000"});

            return replace;
        }

        ui32 GetBoxInitSize() const override {
            return 1;
        }
    };

    TTest test("TestRemoveLinkedAndRbindVolumesWithInit");
    test.DoTest();
}

Y_UNIT_TEST(TestUnlinkLinkedAndRbindVolume) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(MountVolumeId_), {""}, PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT).Success();
            SafePorto_->UnlinkVolume(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH, {""}).Success();
            BoxStatusRepository_->UpdateObjectState(BoxId_, API::EBoxState_UNKNOWN);

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestUnlinkLinkedAndRbindVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestLinkedAndRbindVolumeReadWriteWrite) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            {
                TStringBuilder commandBuilder;
                commandBuilder
                    << "bash -c '"
                    << "echo -n \"some_mounted_volume_data;\" > " << PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT  << "/some_mounted_volume_file;"
                    << "echo -n \"some_rbind_volume_data;\" > " << PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH  << "/some_rbind_volume_file;"
                    << "'"
                ;

                TShellCommand shellCommand(commandBuilder);
                shellCommand.Run();
                shellCommand.Wait();
                UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());
            }

            {
                TStringBuilder commandBuilder;
                commandBuilder
                    << "bash -c '"
                    << "cat " << PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT << "/some_mounted_volume_file;"
                    << "cat " << PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH << "/some_rbind_volume_file;"
                    << "'"
                ;

                TShellCommand shellCommand(commandBuilder);
                shellCommand.Run();
                shellCommand.Wait();
                UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());

                auto stdout = shellCommand.GetOutput();
                UNIT_ASSERT_EQUAL_C("some_mounted_volume_data;some_rbind_volume_data;", stdout, "unexpected stdout: '" << stdout << "'");
            }
        }
    };

    TTest test("TestLinkedAndRbindVolumeReadWriteWrite");
    test.DoTest();
}

Y_UNIT_TEST(TestLinkedAndRbindVolumeReadOnlyWrite) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            const TVector<TString> mountPoints = {
                PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT
                , PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH
            };

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            for (const TString& mountPoint : mountPoints) {
                TStringBuilder commandBuilder;
                commandBuilder << "touch " << mountPoint << "/some_file";

                TShellCommand shellCommand(commandBuilder);
                shellCommand.Run();
                shellCommand.Wait();
                UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_ERROR, shellCommand.GetStatus(), shellCommand.GetError());
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["MOUNT_VOLUME_RO_MODE_LIST"] = "true;";
            specificReplace["RBIND_VOLUME_RO_MODE_LIST"] = "true;";

            return specificReplace;
        }
    };

    TTest test("TestLinkedAndRbindVolumeReadOnlyWrite");
    test.DoTest();
}

Y_UNIT_TEST(TestLinkedAndRbindVolumeReadOnlyRead) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            {
                TStringBuilder commandBuilder;
                commandBuilder << "echo -n \"some_data\" > " << PathHolder_->GetVolumePath(MountVolumeId_) << "/some_file";

                TShellCommand shellCommand(commandBuilder);
                shellCommand.Run();
                shellCommand.Wait();
                UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());
            }

            {
                TStringBuilder commandBuilder;
                commandBuilder
                    << "bash -c '"
                    << "echo -n \"some_mounted_volume_data;\" > " << PathHolder_->GetVolumePath(MountVolumeId_) << "/some_mounted_volume_file;"
                    << "echo -n \"some_rbind_volume_data;\" > " << PathHolder_->GetRbindVolumeStorage(RbindVolumeRef_) << "/some_rbind_volume_file;"
                    << "'"
                ;

                TShellCommand shellCommand(commandBuilder);
                shellCommand.Run();
                shellCommand.Wait();
                UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());
            }

            {
                TStringBuilder commandBuilder;
                commandBuilder
                    << "bash -c '"
                    << "cat " << PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT << "/some_mounted_volume_file;"
                    << "cat " << PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH << "/some_rbind_volume_file;"
                    << "'"
                ;

                TShellCommand shellCommand(commandBuilder);
                shellCommand.Run();
                shellCommand.Wait();
                UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());

                auto stdout = shellCommand.GetOutput();
                UNIT_ASSERT_EQUAL_C("some_mounted_volume_data;some_rbind_volume_data;", stdout, "unexpected stdout: '" << stdout << "'");
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["MOUNT_VOLUME_RO_MODE_LIST"] = "true;";
            specificReplace["RBIND_VOLUME_RO_MODE_LIST"] = "true;";

            return specificReplace;
        }
    };

    TTest test("TestLinkedAndRbindVolumeReadOnlyRead");
    test.DoTest();
}

Y_UNIT_TEST(TestBadLinkedVolumeSourceHash) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume("", "wrong_hash");
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_VOLUMES;
            };

            TickTree(Tree_, 24, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_VOLUMES, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestBadLinkedVolumeSourceHash");
    test.DoTest();
}

Y_UNIT_TEST(TestBadLinkedVolume) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            NFs::MakeDirectoryRecursive(PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT);
            SafePorto_->CreateVolume(
                PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT
                , ""
                , ""
                , {}
                , 0
                , "bad_hash"
                , EPortoVolumeBackend::Auto
                , {""}
                , {}
                , false
            ).Success();

            { // check volume and mount point
                { // volume
                    auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetVolumePath(MountVolumeId_)).Success();
                    UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                    UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetVolumePath(MountVolumeId_), volumeList[0].path());

                    UNIT_ASSERT_EQUAL_C(volumeList[0].links_size(), 1, volumeList[0].links_size());
                    UNIT_ASSERT_UNEQUAL_C(volumeList[0].links(0).target(), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT, volumeList[0].links(0).target());
                }

                { // mount point
                    auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT).Success();
                    UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                    UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT, volumeList[0].path());
                    UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "bad_hash", volumeList[0].private_value());
                }
            }

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 24, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            { // check volume and mount point
                { // volume
                    auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetVolumePath(MountVolumeId_)).Success();
                    UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                    UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetVolumePath(MountVolumeId_), volumeList[0].path());

                    UNIT_ASSERT_EQUAL_C(volumeList[0].links_size(), 2, volumeList[0].links_size());
                    UNIT_ASSERT_C(
                        volumeList[0].links(0).target() == PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT || volumeList[0].links(1).target() == PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT
                        , "link " << PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT << " not found for " << PathHolder_->GetVolumePath(MountVolumeId_)
                    );
                }

                { // mount point
                    auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT).Success();
                    UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                    UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT, volumeList[0].path());
                    UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), CORRECT_VOLUME_HASH, volumeList[0].private_value());
                }
            }
        }
    };

    TTest test("TestBadLinkedVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestBadRbindVolume) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            NFs::MakeDirectoryRecursive(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH);
            SafePorto_->CreateVolume(
                PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH
                , ""
                , ""
                , {}
                , 0
                , "bad_hash"
                , EPortoVolumeBackend::Auto
                , {""}
                , {}
                , false
            ).Success();

            { // check rbind volume mount point
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH, volumeList[0].path());
                UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "bad_hash", volumeList[0].private_value());
            }

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 24, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            { // check rbind volume mount point
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH, volumeList[0].path());
                UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "tree_hash", volumeList[0].private_value());
            }
        }
    };

    TTest test("TestBadRbindVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestBadLinkedVolumeWithRootfsVolume) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(MountVolumeId_), TPortoContainerName(""), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT).Success();
            PrepareVolume("bad_volume", "bad_hash");
            SafePorto_->LinkVolume(PathHolder_->GetVolumePath(MountVolumeId_ + "bad_volume"), TPortoContainerName::NoEscape("."), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT);

            { // check volume and mount point
                { // volume
                    auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetVolumePath(MountVolumeId_)).Success();
                    UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                    UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetVolumePath(MountVolumeId_), volumeList[0].path());

                    UNIT_ASSERT_EQUAL_C(volumeList[0].links_size(), 1, volumeList[0].links_size());
                    UNIT_ASSERT_UNEQUAL_C(volumeList[0].links(0).target(), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT, volumeList[0].links(0).target());
                }

                { // mount point
                    auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT).Success();
                    UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                    UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT, volumeList[0].path());
                    UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "bad_hash", volumeList[0].private_value());
                }
            }

            BoxStatusRepository_->UpdateObjectState(BoxId_, API::EBoxState_UNKNOWN);
            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            { // check volume and mount point
                { // volume
                    auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetVolumePath(MountVolumeId_)).Success();
                    UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                    UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetVolumePath(MountVolumeId_), volumeList[0].path());

                    UNIT_ASSERT_EQUAL_C(volumeList[0].links_size(), 2, volumeList[0].links_size());
                    UNIT_ASSERT_C(
                        volumeList[0].links(0).target() == PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT || volumeList[0].links(1).target() == PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT
                        , "link " << PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT << " not found for " << PathHolder_->GetVolumePath(MountVolumeId_)
                    );
                }

                { // mount point
                    auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT).Success();
                    UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                    UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT, volumeList[0].path());
                    UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), CORRECT_VOLUME_HASH, volumeList[0].private_value());
                }
            }
        }
    };

    TTest test("TestBadLinkedVolumeWithRootfsVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestBadRbindVolumeWithRootfsVolume) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH, TPortoContainerName("")).Success();
            SafePorto_->CreateVolume(
                PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH
                , ""
                , ""
                , {}
                , 0
                , "bad_hash"
                , EPortoVolumeBackend::Auto
                , {""}
                , {}
                , false
            ).Success();

            { // check rbind volume mount point
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH, volumeList[0].path());
                UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "bad_hash", volumeList[0].private_value());
            }

            BoxStatusRepository_->UpdateObjectState(BoxId_, API::EBoxState_UNKNOWN);
            TickTree(Tree_, 24, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            { // check rbind volume mount point
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH, volumeList[0].path());
                UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "tree_hash", volumeList[0].private_value());
            }
        }
    };

    TTest test("TestBadRbindVolumeWithRootfsVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestMultiLinkedVolumeBox) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume("_A");
            PrepareVolume("_B", "wrong_hash");
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            auto breakHookWaitingForVolumes = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_VOLUMES;
            };

            TickTree(Tree_, 60, breakHookWaitingForVolumes);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_VOLUMES, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(MountVolumeId_ + "_B")).Success();
            VolumeStatusRepository_->RemoveObject(MountVolumeId_ + "_B");

            PrepareVolume("_B");

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["MOUNT_VOLUME_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetVolumePath(MountVolumeId_ + "_A"),
                PathHolder_->GetVolumePath(MountVolumeId_ + "_B")
            });
            replace["MOUNT_VOLUME_LINK_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT + "_A",
                PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT + "_B"
            });
            replace["MOUNT_VOLUME_RO_MODE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "false"});
            replace["MOUNT_VOLUME_TREE_HASH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, CORRECT_VOLUME_HASH});

            return replace;
        }

    };

    TTest test("TestMultiLinkedVolumeBox");
    test.DoTest();
}

Y_UNIT_TEST(TestMultiRbindVolumeBox) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage("_A");
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            auto breakHookWaitingForVolumes = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_VOLUMES;
            };

            TickTree(Tree_, 60, breakHookWaitingForVolumes);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_VOLUMES, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareRbindVolumeStorage("_B");

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["RBIND_VOLUME_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH + "_A",
                PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH + "_B"
            });
            replace["RBIND_VOLUME_STORAGE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetRbindVolumeStorage(RbindVolumeRef_) + "_A",
                PathHolder_->GetRbindVolumeStorage(RbindVolumeRef_) + "_B"
            });
            replace["RBIND_VOLUME_RO_MODE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "false"});

            return replace;
        }

    };

    TTest test("TestMultiRbindVolumeBox");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxLinkedVolumeNotReady) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TString& volumePathToBeNotReady)
            : TWrapperPortoClient(client)
            , VolumePathToBeNotReady_(volumePathToBeNotReady)
            , PatchVolumeState_(false)
        {
        }

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName& container) override {
            auto result = TWrapperPortoClient::ListVolumes(path, container);

            if (PatchVolumeState_ && result) {
                // Patch specific volume state
                for (auto& volume : result.Success()) {
                    if (volume.path() == VolumePathToBeNotReady_) {
                        volume.set_state("unready");
                    }
                }
            }

            return result;
        }

        void SetPatchVolumeState(bool patchVolumeState) {
            PatchVolumeState_ = patchVolumeState;
        }

    private:
        const TString VolumePathToBeNotReady_;
        bool PatchVolumeState_;
    };

    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource(100);

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };
            auto breakHookLinkingVolumes = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_LINKING_VOLUMES;
            };

            TickTree(Tree_, 60, breakHookReady);
            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(API::EBoxState_READY, status.state(), TConsoleRenderer(false).Render(Tree_));

            ((TTestPortoClient*)TreePorto_.Get())->SetPatchVolumeState(true);
            // One more tick to trigger state check
            TickTree(Tree_, 1, breakHookLinkingVolumes);
            UNIT_ASSERT_EQUAL_C(
                BoxStatusRepository_->GetObjectStatus(BoxId_).state()
                , API::EBoxState_LINKING_VOLUMES
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT);
        }
    };

    TTest test("TestCreateBoxLinkedVolumeNotReady");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxRbindVolumeNotReady) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TString& volumePathToBeNotReady)
            : TWrapperPortoClient(client)
            , VolumePathToBeNotReady_(volumePathToBeNotReady)
            , PatchVolumeState_(false)
        {
        }

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName& container) override {
            auto result = TWrapperPortoClient::ListVolumes(path, container);

            if (PatchVolumeState_ && result) {
                // Patch specific volume state
                for (auto& volume : result.Success()) {
                    if (volume.path() == VolumePathToBeNotReady_) {
                        volume.set_state("unready");
                    }
                }
            }

            return result;
        }

        void SetPatchVolumeState(bool patchVolumeState) {
            PatchVolumeState_ = patchVolumeState;
        }

    private:
        const TString VolumePathToBeNotReady_;
        bool PatchVolumeState_;
    };

    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource(100);

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };
            auto breakHookLinkingVolumes = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_LINKING_VOLUMES;
            };

            TickTree(Tree_, 60, breakHookReady);
            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(API::EBoxState_READY, status.state(), TConsoleRenderer(false).Render(Tree_));

            ((TTestPortoClient*)TreePorto_.Get())->SetPatchVolumeState(true);
            // One more tick to trigger state check
            TickTree(Tree_, 1, breakHookLinkingVolumes);
            UNIT_ASSERT_EQUAL_C(
                BoxStatusRepository_->GetObjectStatus(BoxId_).state()
                , API::EBoxState_LINKING_VOLUMES
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetBoxRootfsPath(BoxId_) + RBIND_VOLUME_MOUNT_PATH);
        }
    };

    TTest test("TestCreateBoxRbindVolumeNotReady");
    test.DoTest();
}

Y_UNIT_TEST(TestMountVolumesToExistingRoRootfsPaths) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareRootfsLayer();
            PrepareLayerWithFolders("_mount_points", {"mounted_data", "rbind_volume_data"});
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource(100);

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["ROOTFS_READ_ONLY"] = "true";
            replace["LAYER_NAME_LIST"] = PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_) + ';' + PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_mount_points") + ';';
            replace["LAYER_DOWNLOAD_HASH_LIST"] = LayerDownloadHashPrefix_ + ";" + LayerDownloadHashPrefix_ + "_mount_points" + ";";

            return replace;
        }
    };

    TTest test("TestMountVolumesToExistingRoRootfsPaths");
    test.DoTest();
}

Y_UNIT_TEST(TestMountLinkedVolumeToUnexistingRoRootfsPath) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareRootfsLayer();
            PrepareLayerWithFolders("_mount_points", {"rbind_volume_data"});
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource(100);

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_INVALID
                        && BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message() != "";
            };

            TickTree(Tree_, 60, breakHook);
            auto boxFailMessage = BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message();
            TString expectedBoxFailMessage = TStringBuilder() << "volume link path '"  << PathHolder_->GetBoxRootfsPath(BoxId_) << MOUNT_POINT << "' does not exist on RO rootfs";
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_INVALID, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(boxFailMessage, expectedBoxFailMessage, Endl << boxFailMessage);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["ROOTFS_READ_ONLY"] = "true";
            replace["LAYER_NAME_LIST"] = PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_) + ';' + PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_mount_points") + ';';
            replace["LAYER_DOWNLOAD_HASH_LIST"] = LayerDownloadHashPrefix_ + ";" + LayerDownloadHashPrefix_ + "_mount_points" + ";";

            return replace;
        }
    };

    TTest test("TestMountLinkedVolumeToUnexistingRoRootfsPath");
    test.DoTest();
}

Y_UNIT_TEST(TestMountRbindVolumeToUnexistingRoRootfsPath) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareRootfsLayer();
            PrepareLayerWithFolders("_mount_points", {"mounted_data"});
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource(100);

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_INVALID
                       && BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message() != "";
            };

            TickTree(Tree_, 60, breakHook);
            auto boxFailMessage = BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message();
            TString expectedBoxFailMessage = TStringBuilder() << "rbind volume mount path '"  << PathHolder_->GetBoxRootfsPath(BoxId_) << RBIND_VOLUME_MOUNT_PATH << "' does not exist on RO rootfs";
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_INVALID, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(boxFailMessage, expectedBoxFailMessage, Endl << boxFailMessage);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["ROOTFS_READ_ONLY"] = "true";
            replace["LAYER_NAME_LIST"] = PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_) + ';' + PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_mount_points") + ';';
            replace["LAYER_DOWNLOAD_HASH_LIST"] = LayerDownloadHashPrefix_ + ";" + LayerDownloadHashPrefix_ + "_mount_points" + ";";

            return replace;
        }
    };

    TTest test("TestMountRbindVolumeToUnexistingRoRootfsPath");
    test.DoTest();
}

Y_UNIT_TEST(TestMountYtAndBasesearchVolumesToExistingRoRootfsPaths) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareRootfsLayer();
            PrepareLayerWithFolders("_mount_points", {"yt", "basesearch"});
            NFs::MakeDirectoryRecursive(PathHolder_->GetRbindVolumeStorage("yt"));
            NFs::MakeDirectoryRecursive(PathHolder_->GetRbindVolumeStorage("basesearch"));
            PrepareStaticResource(100);

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["ROOTFS_READ_ONLY"] = "true";
            replace["LAYER_NAME_LIST"] = PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_) + ';' + PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_mount_points") + ';';
            replace["LAYER_DOWNLOAD_HASH_LIST"] = LayerDownloadHashPrefix_ + ";" + LayerDownloadHashPrefix_ + "_mount_points" + ";";

            replace["YT_VOLUME_STORAGE"] = PathHolder_->GetRbindVolumeStorage("yt");
            replace["YT_VOLUME_STORAGE"] = PathHolder_->GetRbindVolumeStorage("basesearch");
            replace["YT_ENABLED"] = "true";
            replace["BASESEARCH_ENABLED"] = "true";

            replace["MOUNT_VOLUME_PATH_LIST"] = "";
            replace["MOUNT_VOLUME_LINK_PATH_LIST"] = "";
            replace["MOUNT_VOLUME_RO_MODE_LIST"] = "";
            replace["MOUNT_VOLUME_TREE_HASH_LIST"] = "";

            replace["RBIND_VOLUME_PATH_LIST"] = "";
            replace["RBIND_VOLUME_STORAGE_LIST"] = "";
            replace["RBIND_VOLUME_RO_MODE_LIST"] = "";

            return replace;
        }
    };

    TTest test("TestMountYtAndBasesearchVolumesToExistingRoRootfsPaths");
    test.DoTest();
}

Y_UNIT_TEST(TestMountYtVolumeToUnexistingRoRootfsPaths) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareRootfsLayer();
            NFs::MakeDirectoryRecursive(PathHolder_->GetRbindVolumeStorage("yt"));
            NFs::MakeDirectoryRecursive(PathHolder_->GetRbindVolumeStorage("basesearch"));
            PrepareStaticResource(100);

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_INVALID
                       && BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message() != "";
            };

            TickTree(Tree_, 60, breakHook);
            auto failMessage = BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message();
            TString expectedFailMessage = TStringBuilder() << "yt volume path '" << PathHolder_->GetBoxRootfsPath(BoxId_) + "/yt'" << " does not exist on RO rootfs";
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_INVALID, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(failMessage, expectedFailMessage, Endl << failMessage);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["ROOTFS_READ_ONLY"] = "true";

            replace["YT_ENABLED"] = "true";

            replace["YT_VOLUME_STORAGE"] = PathHolder_->GetRbindVolumeStorage("yt");
            replace["YT_VOLUME_STORAGE"] = PathHolder_->GetRbindVolumeStorage("basesearch");
            replace["MOUNT_VOLUME_PATH_LIST"] = "";
            replace["MOUNT_VOLUME_LINK_PATH_LIST"] = "";
            replace["MOUNT_VOLUME_RO_MODE_LIST"] = "";
            replace["MOUNT_VOLUME_TREE_HASH_LIST"] = "";

            replace["RBIND_VOLUME_PATH_LIST"] = "";
            replace["RBIND_VOLUME_STORAGE_LIST"] = "";
            replace["RBIND_VOLUME_RO_MODE_LIST"] = "";

            return replace;
        }
    };

    TTest test("TestMountYtVolumeToUnexistingRoRootfsPaths");
    test.DoTest();
}

Y_UNIT_TEST(TestMountBasesearchVolumeToUnexistingRoRootfsPaths) {
    class TTest : public ITestBoxTreeLinkedAndRbindVolumes {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeLinkedAndRbindVolumes(testName)
        {
        }

    protected:
        void Test() override {
            PrepareRootfsLayer();
            NFs::MakeDirectoryRecursive(PathHolder_->GetRbindVolumeStorage("yt"));
            NFs::MakeDirectoryRecursive(PathHolder_->GetRbindVolumeStorage("basesearch"));
            PrepareStaticResource(100);

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_INVALID
                       && BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message() != "";
            };

            TickTree(Tree_, 60, breakHook);
            auto failMessage = BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message();
            TString expectedFailMessage = TStringBuilder() << "basesearch volume path '" << PathHolder_->GetBoxRootfsPath(BoxId_) + "/basesearch'" << " does not exist on RO rootfs";
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_INVALID, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(failMessage, expectedFailMessage, Endl << failMessage << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["YT_VOLUME_STORAGE"] = PathHolder_->GetRbindVolumeStorage("yt");
            replace["YT_VOLUME_STORAGE"] = PathHolder_->GetRbindVolumeStorage("basesearch");

            replace["ROOTFS_READ_ONLY"] = "true";

            replace["BASESEARCH_ENABLED"] = "true";

            replace["MOUNT_VOLUME_PATH_LIST"] = "";
            replace["MOUNT_VOLUME_LINK_PATH_LIST"] = "";
            replace["MOUNT_VOLUME_RO_MODE_LIST"] = "";
            replace["MOUNT_VOLUME_TREE_HASH_LIST"] = "";

            replace["RBIND_VOLUME_PATH_LIST"] = "";
            replace["RBIND_VOLUME_STORAGE_LIST"] = "";
            replace["RBIND_VOLUME_RO_MODE_LIST"] = "";

            return replace;
        }
    };

    TTest test("TestMountBasesearchVolumeToUnexistingRoRootfsPaths");
    test.DoTest();
}


}

} // namespace NInfra::NPodAgent::NTreeTest
