#include "box_test_canon.h"

#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestBoxTreeRootfsVolume: public ITestBoxCanon {
public:
    ITestBoxTreeRootfsVolume(const TString& testName)
        : ITestBoxCanon(testName, "BoxTreeRootfsVolume")
    {
    }
};

Y_UNIT_TEST_SUITE(BoxTreeRootfsVolumeTestSuite) {

Y_UNIT_TEST(TestBadRootfsLayerPrivateValue) {
    class TTest : public ITestBoxTreeRootfsVolume {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeRootfsVolume(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();
            TString layerName = PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_);
            SafePorto_->SetLayerPrivate(":wrong_hash", layerName);

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_ROOTFS_LAYERS;
            };

            TickTree(Tree_, 24, breakHook);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_ROOTFS_LAYERS, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestBadRootfsLayerPrivateValue");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveRootfsVolume) {
    class TTest : public ITestBoxTreeRootfsVolume {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeRootfsVolume(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetBoxRootfsPath(BoxId_)).Success();
            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(MountVolumeId_), TPortoContainerName(""), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT).Success();
            BoxStatusRepository_->UpdateObjectState(BoxId_, API::EBoxState_UNKNOWN);

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestRemoveRootfsVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveRootfsVolumeAndLayer) {
    class TTest : public ITestBoxTreeRootfsVolume {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeRootfsVolume(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            auto breakHookWaitingForRootfsLayers = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_ROOTFS_LAYERS;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetBoxRootfsPath(BoxId_)).Success();
            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(MountVolumeId_), TPortoContainerName(""), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT).Success();
            SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_)).Success();

            TickTree(Tree_, 60, breakHookWaitingForRootfsLayers);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_ROOTFS_LAYERS, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareLayer("", "some_data");

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestRemoveRootfsVolumeAndLayer");
    test.DoTest();
}

Y_UNIT_TEST(TestBadRootfsVolumePrivate) {
    class TTest : public ITestBoxTreeRootfsVolume {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeRootfsVolume(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            NFs::MakeDirectoryRecursive(PathHolder_->GetBoxRootfsPath(BoxId_));
            SafePorto_->CreateVolume(
                PathHolder_->GetBoxRootfsPath(BoxId_)
                , PathHolder_->GetBoxRootfsPersistentStorage(BoxId_)
                , ""
                , {}
                , 0
                , "bad_hash"
                , EPortoVolumeBackend::Auto
                , {""}
                , {}
                , false
            ).Success();

            {
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_)).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_), volumeList[0].path());
                UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "bad_hash", volumeList[0].private_value());
            }

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 32, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            {
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_)).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_), volumeList[0].path());
                UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "tree_hash", volumeList[0].private_value());
            }
        }
    };

    TTest test("TestBadRootfsVolumePrivate");
    test.DoTest();
}

Y_UNIT_TEST(TestUnlinkRootfsVolumeFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TString& badVolume)
            : TWrapperPortoClient(client)
            , BadVolume_(badVolume)
        {
        }

        TExpected<void, TPortoError> UnlinkVolume(
            const TString& path
            , const TPortoContainerName& container
            , const TString& target
            , bool strict
        ) override {
            if (path == BadVolume_) {
                return TPortoError{EPortoError::VolumeNotLinked, TStringBuilder() << "UnlinkVolume failed to unlink '" << path << "'"};
            } else {
                return TWrapperPortoClient::UnlinkVolume(path, container, target, strict);
            }
        }

    private:
        const TString BadVolume_;
    };

    class TTest : public ITestBoxTreeRootfsVolume {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeRootfsVolume(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            NFs::MakeDirectoryRecursive(PathHolder_->GetBoxRootfsPath(BoxId_));
            SafePorto_->CreateVolume(
                PathHolder_->GetBoxRootfsPath(BoxId_)
                , PathHolder_->GetBoxRootfsPersistentStorage(BoxId_)
                , ""
                , {}
                , 0
                , "bad_hash"
                , EPortoVolumeBackend::Auto
                , {""}
                , {}
                , false
            ).Success();

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_INVALID;
            };

            TickTree(Tree_, 12, breakHook);

            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EBoxState_INVALID
                , API::EBoxState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_STRING_CONTAINS(
                status.failed().message()
                , "UnlinkVolume failed"
            );
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetBoxRootfsPath(BoxId_));
        }
    };

    TTest test("TestUnlinkRootfsVolumeFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWrongRootfsStorage) {
    class TTest : public ITestBoxTreeRootfsVolume {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeRootfsVolume(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            NFs::MakeDirectoryRecursive(PathHolder_->GetBoxRootfsPath(BoxId_));
            SafePorto_->CreateVolume(PathHolder_->GetBoxRootfsPath(BoxId_), PersistentStoragePrefix_ + "wrong_storage", "", {}, 0, "tree_hash", EPortoVolumeBackend::Auto, {""}, {}, false).Success();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL(SafePorto_->GetStoragePrivate("", PathHolder_->GetBoxRootfsPersistentStorage(BoxId_)).Success(), "tree_hash");
        }
    };

    TTest test("TestWrongRootfsStorage");
    test.DoTest();
}

Y_UNIT_TEST(TestWrongRootfsStorageHash) {
    class TTest : public ITestBoxTreeRootfsVolume {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeRootfsVolume(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            NFs::MakeDirectoryRecursive(PathHolder_->GetBoxRootfsPath(BoxId_));
            TString storage = PathHolder_->GetBoxRootfsPersistentStorage(BoxId_);
            SafePorto_->CreateVolume(PathHolder_->GetBoxRootfsPath(BoxId_), storage, "", {}, 0, "wrong_hash", EPortoVolumeBackend::Auto, {""}, {}, false).Success();
            SafePorto_->UnlinkVolume(PathHolder_->GetBoxRootfsPath(BoxId_));

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 20, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL(SafePorto_->GetStoragePrivate("", PathHolder_->GetBoxRootfsPersistentStorage(BoxId_)).Success(), "tree_hash");
        }
    };

    TTest test("TestWrongRootfsStorageHash");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxRootfsCreateVolumeFail) {
    class TTest : public ITestBoxTreeRootfsVolume {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeRootfsVolume(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).fail_counter() > 1;
            };

            TickTree(Tree_, 60, breakHook);
            UNIT_ASSERT_C(BoxStatusRepository_->GetObjectStatus(BoxId_).fail_counter() > 1, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message(), "CreateVolume", TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(BoxStatusRepository_->GetObjectStatus(BoxId_).failed().message(), "Private value too log, max 4096 bytes", TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["TREE_HASH"] = TString(4096 + 1, 'a'); // too big for volume private
            return specificReplace;
        }
    };

    TTest test("TestCreateBoxRootfsCreateVolumeFail");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
