#include "box_test_canon.h"

#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/util/string_utils.h>
#include <infra/pod_agent/libs/ip_client/mock_client.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>
#include <util/system/shellcommand.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestBoxTreeTemplate: public ITestBoxCanon {
public:
    ITestBoxTreeTemplate(const TString& testName)
        : ITestBoxCanon(testName, "BoxTreeTemplate")
    {
    }
};

Y_UNIT_TEST_SUITE(BoxTreeTemplateTestSuite) {

Y_UNIT_TEST(TestCreateBox) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(API::EBoxState_READY, status.state(), TConsoleRenderer(false).Render(Tree_));

            const EPortoContainerState boxMetaContainerState = FromString(SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::State).Success());
            UNIT_ASSERT_EQUAL_C(
                EPortoContainerState::Meta
                , boxMetaContainerState
                , ToString(boxMetaContainerState) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            { // test compute resource
                const TString cpuGuarantee = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::CpuGuarantee).Success();
                const TString cpuLimit = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::CpuLimit).Success();
                const TString memoryGuarantee = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::MemoryGuarantee).Success();
                const TString memoryLimit = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::MemoryLimit).Success();
                const TString anonLimit = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::AnonLimit).Success();
                const TString rechPgfault = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::RechPgfault).Success();

                UNIT_ASSERT_EQUAL_C(cpuGuarantee, "2c", cpuGuarantee);
                UNIT_ASSERT_EQUAL_C(cpuLimit, "1c", cpuLimit);
                UNIT_ASSERT_EQUAL_C(memoryGuarantee, ToString(1 << 25), memoryGuarantee);
                UNIT_ASSERT_EQUAL_C(memoryLimit, ToString(1 << 26), memoryLimit);
                UNIT_ASSERT_EQUAL_C(anonLimit, ToString(1 << 27), anonLimit);
                UNIT_ASSERT_EQUAL_C(rechPgfault, "false", rechPgfault);

            }

            { // test no porto isolation
                const TString enablePorto = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::EnablePorto).Success();

                UNIT_ASSERT_EQUAL_C(enablePorto, "true", enablePorto);
            }

            { // test quota
                { // 1K
                    TShellCommand shellCommand(
                        TStringBuilder() << "dd if=/dev/zero bs=1K count=1 of=" << PathHolder_->GetBoxRootfsPath(BoxId_) << "/sample.txt"
                    );
                    shellCommand.Run();
                    shellCommand.Wait();
                    UNIT_ASSERT_EQUAL_C(0, *(shellCommand.GetExitCode()), *(shellCommand.GetExitCode()) << shellCommand.GetError());
                }
                { // 2M
                    TShellCommand shellCommand(
                        TStringBuilder() << "dd if=/dev/zero bs=2M count=1 of=" << PathHolder_->GetBoxRootfsPath(BoxId_) << "/sample.txt"
                    );
                    shellCommand.Run();
                    shellCommand.Wait();
                    UNIT_ASSERT_EQUAL_C(1, *(shellCommand.GetExitCode()), *(shellCommand.GetExitCode()) << shellCommand.GetError());
                }
            }
        }
    };

    TTest test("TestCreateBox");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxWithChildOnlyIsolation) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(API::EBoxState_READY, status.state(), TConsoleRenderer(false).Render(Tree_));

            const EPortoContainerState boxMetaContainerState = FromString(SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::State).Success());
            UNIT_ASSERT_EQUAL_C(
                EPortoContainerState::Meta
                , boxMetaContainerState
                , ToString(boxMetaContainerState) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            { // test porto isolation
                const TString enablePorto = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::EnablePorto).Success();

                UNIT_ASSERT_EQUAL_C(enablePorto, "child-only", enablePorto);
            }
        }
        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["BOX_ENABLE_PORTO"] = "child-only";

            return specificReplace;
        }
    };

    TTest test("TestCreateBoxWithChildOnlyIsolationt");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxWithSpecificPlace) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareSpecificPlace();
            PrepareLayer("", "some_data", SpecificPlace_);
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource(100, SpecificPlace_);

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(API::EBoxState_READY, status.state(), TConsoleRenderer(false).Render(Tree_));

            {
                // Check rootfs volume
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetBoxRootfsPath(BoxId_)).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetBoxRootfsPath(BoxId_), volumeList[0].path());
                if (NPortoTestLib::IsInsideSandboxPortoIsolation()) {
                    UNIT_ASSERT_C(TString(volumeList[0].place()).EndsWith(SpecificPlacePath_), volumeList[0].place());
                } else {
                    UNIT_ASSERT_EQUAL_C(volumeList[0].place(), SpecificPlacePath_, volumeList[0].place());
                }
            }

            {
                // Check box meta container
                auto boxMetaContainerPlace = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::Place).Success();
                UNIT_ASSERT_EQUAL_C(boxMetaContainerPlace, SpecificPlacePath_, boxMetaContainerPlace);
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["BOX_CONTAINER_PLACE"] = SpecificPlacePath_;
            specificReplace["ROOTFS_VOLUME_PLACE"] = SpecificPlace_;
            specificReplace["STATIC_RESOURCE_ORIGINAL_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHashPrefix_, SpecificPlace_)});

            return specificReplace;
        }
    };

    TTest test("TestCreateBoxWithSpecificPlace");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxWithIpAddress) {
    class TTestIpClient: public TMockIpClient {
    public:
        TTestIpClient()
            : LastDevice_("")
            , LastIp_(TIpDescription("", 0))
            , AddAddressCalls_(0)
        {
        }

        TExpected<void, TIpClientError> AddAddress(const TString& device, const TIpDescription& ip) override {
            ++AddAddressCalls_;
            LastDevice_ = device;
            LastIp_ = ip;

            return TExpected<void, TIpClientError>::DefaultSuccess();
        }

        TString GetLastDevice() const {
            return LastDevice_;
        }

        TIpDescription GetLastIp() const {
            return LastIp_;
        }

        ui32 GetAddAddressCalls() {
            return AddAddressCalls_;
        }

    private:
        TString LastDevice_;
        TIpDescription LastIp_;
        ui32 AddAddressCalls_;
    };

    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource(100);

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(API::EBoxState_READY, status.state(), TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(BoxIpAddress_, status.ip6_address(), status.ip6_address() << Endl << TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_EQUAL_C(((TTestIpClient*)TreeIpClient_.Get())->GetLastDevice(), NetworkDevice_, ((TTestIpClient*)TreeIpClient_.Get())->GetLastDevice());
            UNIT_ASSERT_EQUAL_C(((TTestIpClient*)TreeIpClient_.Get())->GetLastIp().Ip6, BoxIpAddress_, ((TTestIpClient*)TreeIpClient_.Get())->GetLastIp().Ip6);
            UNIT_ASSERT_EQUAL_C(((TTestIpClient*)TreeIpClient_.Get())->GetLastIp().Subnet, 128, ((TTestIpClient*)TreeIpClient_.Get())->GetLastIp().Subnet);
            UNIT_ASSERT_EQUAL_C(((TTestIpClient*)TreeIpClient_.Get())->GetAddAddressCalls(), 1, ((TTestIpClient*)TreeIpClient_.Get())->GetAddAddressCalls());
        }

        TIpClientPtr GetSpecificIpClient(TIpClientPtr /* ipClient */) const override {
            return new TTestIpClient();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["NETWORK_DEVICE"] = NetworkDevice_;
            specificReplace["BOX_IP6_ADDRESS"] = BoxIpAddress_;

            return specificReplace;
        }

    private:
        const TString NetworkDevice_ = "my_device";
        const TString BoxIpAddress_ = "2a02:6b8:c08:68a3:0:696:6937:1";
    };

    TTest test("TestCreateBoxWithIpAddress");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxWithIpAddressFail) {
    class TTestIpClient: public TMockIpClient {
    public:
        TTestIpClient()
            : LastDevice_("")
            , LastIp_(TIpDescription("", 0))
            , AddAddressCalls_(0)
        {
        }

        TExpected<void, TIpClientError> AddAddress(const TString& device, const TIpDescription& ip) override {
            ++AddAddressCalls_;
            LastDevice_ = device;
            LastIp_ = ip;

            return TIpClientError(EIpClientError::Unspecified, "AddAddressError");
        }

        TString GetLastDevice() const {
            return LastDevice_;
        }

        TIpDescription GetLastIp() const {
            return LastIp_;
        }

        ui32 GetAddAddressCalls() {
            return AddAddressCalls_;
        }

    private:
        TString LastDevice_;
        TIpDescription LastIp_;
        ui32 AddAddressCalls_;
    };

    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource(100);

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            BoxStatusRepository_->UpdateObjectIpAddress(BoxId_, BoxIpAddress_);

            TickTree(Tree_, 60, breakHookReady);
            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(API::EBoxState_READY, status.state(), TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_EQUAL_C("", status.ip6_address(), status.ip6_address() << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "cannot add ip address", status.failed().message() << Endl << TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_EQUAL_C(((TTestIpClient*)TreeIpClient_.Get())->GetLastDevice(), NetworkDevice_, ((TTestIpClient*)TreeIpClient_.Get())->GetLastDevice());
            UNIT_ASSERT_EQUAL_C(((TTestIpClient*)TreeIpClient_.Get())->GetLastIp().Ip6, BoxIpAddress_, ((TTestIpClient*)TreeIpClient_.Get())->GetLastIp().Ip6);
            UNIT_ASSERT_EQUAL_C(((TTestIpClient*)TreeIpClient_.Get())->GetLastIp().Subnet, 128, ((TTestIpClient*)TreeIpClient_.Get())->GetLastIp().Subnet);
            UNIT_ASSERT_EQUAL_C(((TTestIpClient*)TreeIpClient_.Get())->GetAddAddressCalls(), 1, ((TTestIpClient*)TreeIpClient_.Get())->GetAddAddressCalls());
        }

        TIpClientPtr GetSpecificIpClient(TIpClientPtr /* ipClient */) const override {
            return new TTestIpClient();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["NETWORK_DEVICE"] = NetworkDevice_;
            specificReplace["BOX_IP6_ADDRESS"] = BoxIpAddress_;

            return specificReplace;
        }

    private:
        const TString NetworkDevice_ = "my_device";
        const TString BoxIpAddress_ = "2a02:6b8:c08:68a3:0:696:6937:1";
    };

    TTest test("TestCreateBoxWithIpAddressFail");
    test.DoTest();
}


Y_UNIT_TEST(TestWaitingForStaticResources) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareRbindVolumeStorage();
            PrepareVolume();

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_STATIC_RESOURCES;
            };

            TickTree(Tree_, 60, breakHook);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_STATIC_RESOURCES, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWaitingForStaticResources");
    test.DoTest();
}

Y_UNIT_TEST(TestWaitingForRootfsLayer) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareStaticResource();
            LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(LayerIdPrefix_, LayerDownloadHashPrefix_));
            SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_));

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_ROOTFS_LAYERS;
            };

            TickTree(Tree_, 60, breakHook);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_ROOTFS_LAYERS, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWaitingForRootfsLayer");
    test.DoTest();
}

Y_UNIT_TEST(TestMultiLayerRootfsVolume) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("_A", "some_data");
            PrepareLayer("_B", "other_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            auto breakHookWaitingForRootfsLayers = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_ROOTFS_LAYERS;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetBoxRootfsPath(BoxId_)).Success();
            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(MountVolumeId_), TPortoContainerName(""), PathHolder_->GetBoxRootfsPath(BoxId_) + MOUNT_POINT).Success();
            SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_A")).Success();

            TickTree(Tree_, 60, breakHookWaitingForRootfsLayers);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_ROOTFS_LAYERS, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareLayer("_A", "some_data");

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT(NFs::Exists(PathHolder_->GetBoxRootfsPath(BoxId_) + "/" + "my_static_resource"));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["LAYER_NAME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_A"),
                PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_B")
            });
            replace["LAYER_DOWNLOAD_HASH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                LayerDownloadHashPrefix_ + "_A"
                , LayerDownloadHashPrefix_ + "_B"
            });

            replace["STATIC_RESOURCE_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"my_static_resource"});
            replace["STATIC_RESOURCE_ORIGINAL_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHashPrefix_, "")});

            return replace;
        }
    };

    TTest test("TestMultiLayerRootfsVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestMultiLayerRootfsVolumeWithInit) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("_A", "some_data");
            PrepareLayer("_B", "other_data");
            PrepareRootfsLayer();
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits_size(), 1, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits(0).last().state(), API::EContainerState_EXITED, Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["LAYER_NAME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_),
                PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_A"),
                PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_B")
            });
            replace["LAYER_DOWNLOAD_HASH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                LayerDownloadHashPrefix_
                , LayerDownloadHashPrefix_ + "_A"
                , LayerDownloadHashPrefix_ + "_B"
            });

            replace["STATIC_RESOURCE_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"my_static_resource"});
            replace["STATIC_RESOURCE_ORIGINAL_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHashPrefix_, "")});

            replace["INIT_CMD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"bash -c \"echo some_data | diff - tmp.txt \""});
            replace["INIT_CWD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_NUM_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"0"});
            replace["INIT_CONTAINER_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetBoxInitContainer(BoxId_, 0)
            });
            replace["INIT_CONTAINER_PREVIOUS_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_ENVIRONMENT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_SECRET_ENVIRONMENT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});

            replace["INIT_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_STDERR_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_STDOUT_LOG_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_STDERR_LOG_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});
            replace["INIT_STDOUT_AND_STDERR_FILE_SIZE_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({""});

            replace["INIT_CPU_GUARANTEE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1c"});
            replace["INIT_CPU_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1c"});
            replace["INIT_CPU_POLICY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "normal"});
            replace["INIT_CPU_WEIGHT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1.01"});
            replace["INIT_MEMORY_GUARANTEE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ToString(1 << 25)});
            replace["INIT_MEMORY_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ToString(1 << 25)});
            replace["INIT_ANON_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ToString(1 << 25)});
            replace["INIT_RECHARGE_ON_PGFAULT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "false"});
            replace["INIT_THREAD_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1001"});
            replace["INIT_ULIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ""});
            replace["INIT_CORE_COMMAND_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ""});
            replace["INIT_USER_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ""});
            replace["INIT_GROUP_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ""});
            replace["INIT_AGING_TIME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, ToString(1 << 16)});
            replace["INIT_IO_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "/tmp r: 20000"});
            replace["INIT_IO_OPS_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "/tmp r: 20001"});
            replace["INIT_IO_POLICY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "normal"});
            replace["INIT_IO_WEIGHT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1.02"});

            replace["INIT_INITIAL_DELAY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "0"});
            replace["INIT_RESTART_PERIOD_SCALE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "0"});
            replace["INIT_RESTART_PERIOD_BACKOFF_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "1"});
            replace["INIT_MAX_RESTART_PERIOD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "60000"});
            replace["INIT_MIN_RESTART_PERIOD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "60000"});
            replace["INIT_MAX_EXECUTION_TIME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({1, "60000"});

            return replace;
        }

        ui32 GetBoxInitSize() const override {
            return 1;
        }
    };

    TTest test("TestMultiLayerRootfsVolumeWithInit");
    test.DoTest();
}

Y_UNIT_TEST(TestModifyStaticResourceAfterBoxBuild) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
                : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            auto breakHookCreating = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_CREATING_ROOTFS_VOLUME;
            };

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            // We compare time in seconds
            // So we need at least one second diff between container start and resource modification
            Sleep(TDuration::MilliSeconds(1200));
            NFs::Remove(PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHashPrefix_, ""));
            PrepareStaticResource();

            TickTree(Tree_, 60, breakHookCreating);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_CREATING_ROOTFS_VOLUME, Endl << TConsoleRenderer(false).Render(Tree_));

            TickTree(Tree_, 60, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestModifyStaticResourceAfterBoxBuild");
    test.DoTest();
}

Y_UNIT_TEST(TestRestartInit) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareRootfsLayer();
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookExited = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).inits_size() > 1
                    && BoxStatusRepository_->GetObjectStatus(BoxId_).inits(1).positive_return_code_counter() > 1
                ;
            };

            TickTree(Tree_, 60, breakHookExited);

            UNIT_ASSERT_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits_size() == 2, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits(1).positive_return_code_counter() > 1, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits(1).zero_return_code_counter() == 0, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits(1).last().return_code() == 1, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(BoxStatusRepository_->GetObjectStatus(BoxId_).inits(0).last().stdout_tail() == "init\n", Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["INIT_CMD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"echo init", "false"});
            replace["INIT_CWD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_NUM_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"0", "1"});
            replace["INIT_CONTAINER_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetBoxInitContainer(BoxId_, 0),
                PathHolder_->GetBoxInitContainer(BoxId_, 1)
            });
            replace["INIT_CONTAINER_PREVIOUS_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                "",
                PathHolder_->GetBoxInitContainer(BoxId_, 0),
            });
            replace["INIT_ENVIRONMENT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_SECRET_ENVIRONMENT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});

            replace["INIT_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_STDERR_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_STDOUT_LOG_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_STDERR_LOG_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_STDOUT_AND_STDERR_FILE_SIZE_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});

            replace["INIT_CPU_GUARANTEE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1c"});
            replace["INIT_CPU_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1c"});
            replace["INIT_CPU_POLICY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "normal"});
            replace["INIT_CPU_WEIGHT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1.01"});
            replace["INIT_MEMORY_GUARANTEE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ToString(1 << 25)});
            replace["INIT_MEMORY_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ToString(1 << 25)});
            replace["INIT_ANON_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ToString(1 << 25)});
            replace["INIT_RECHARGE_ON_PGFAULT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "false"});
            replace["INIT_THREAD_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1001"});
            replace["INIT_ULIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ""});
            replace["INIT_CORE_COMMAND_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ""});
            replace["INIT_USER_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ""});
            replace["INIT_GROUP_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ""});
            replace["INIT_AGING_TIME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ToString(1 << 16)});
            replace["INIT_IO_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "/tmp r: 20000"});
            replace["INIT_IO_OPS_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "/tmp r: 20001"});
            replace["INIT_IO_POLICY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "normal"});
            replace["INIT_IO_WEIGHT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1.02"});

            replace["INIT_INITIAL_DELAY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "0"});
            replace["INIT_RESTART_PERIOD_SCALE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "0"});
            replace["INIT_RESTART_PERIOD_BACKOFF_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1"});
            replace["INIT_MAX_RESTART_PERIOD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1000"});
            replace["INIT_MIN_RESTART_PERIOD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1000"});
            replace["INIT_MAX_EXECUTION_TIME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "60000"});

            return replace;
        }

        ui32 GetBoxInitSize() const override {
            return 2;
        }
    };

    TTest test("TestRestartInit");
    test.DoTest();
}

Y_UNIT_TEST(TestBoxMetaAndInitEnvironment) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareRootfsLayer();
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);

            UNIT_ASSERT_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            {
                const TString boxMetaEnvironment = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::Env).Success();
                UNIT_ASSERT_EQUAL_C(boxMetaEnvironment, BoxSpecificEnvironment_, "Bad box meta container environment: " << Quote(boxMetaEnvironment));
            }

            for (size_t i = 0; i < 2; ++i) {
                TPortoContainerName boxInitContainer = PathHolder_->GetBoxInitContainer(BoxId_, i);
                const TString initEnvironment = SafePorto_->GetProperty(boxInitContainer, EPortoContainerProperty::Env).Success();
                UNIT_ASSERT_EQUAL_C(initEnvironment, InitSpecificEnvironment_[i], "Bad init " << i << " container environment: " << Quote(initEnvironment));

                const TString initSecretEnvironment = SafePorto_->GetProperty(boxInitContainer, EPortoContainerProperty::EnvSecret).Success();
                // secrets have different salt and md5, let's check only substring
                UNIT_ASSERT_STRING_CONTAINS_C(initSecretEnvironment, InitSpecificSecretEnvironmentHidden_[i], "Bad init " << i << " container secret environment: " << Quote(initSecretEnvironment));

                const TString initStdout = SafePorto_->GetProperty(boxInitContainer, EPortoContainerProperty::StdOut).Success();
                UNIT_ASSERT_EQUAL_C(
                    initStdout
                    , InitStdout_[i]
                    , "Bad init " << i << " stdout. Probably missed environment inheritance from box or init specific environment: " << Quote(initStdout)
                );
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> replace;

            replace["INIT_CMD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"echo -n init0 $first_name $container_name", "echo -n init1 $second_name $container_name"});
            replace["INIT_CWD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_NUM_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"0", "1"});
            replace["INIT_CONTAINER_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                PathHolder_->GetBoxInitContainer(BoxId_, 0),
                PathHolder_->GetBoxInitContainer(BoxId_, 1)
            });
            replace["INIT_CONTAINER_PREVIOUS_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({
                "",
                PathHolder_->GetBoxInitContainer(BoxId_, 0),
            });
            replace["INIT_ENVIRONMENT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(InitSpecificEnvironment_);
            replace["INIT_SECRET_ENVIRONMENT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(InitSpecificSecretEnvironment_);

            replace["INIT_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_STDERR_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_STDOUT_LOG_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_STDERR_LOG_PATH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});
            replace["INIT_STDOUT_AND_STDERR_FILE_SIZE_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"", ""});

            replace["INIT_CPU_GUARANTEE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1c"});
            replace["INIT_CPU_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1c"});
            replace["INIT_CPU_POLICY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "normal"});
            replace["INIT_CPU_WEIGHT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1.01"});
            replace["INIT_MEMORY_GUARANTEE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ToString(1 << 25)});
            replace["INIT_MEMORY_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ToString(1 << 25)});
            replace["INIT_ANON_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ToString(1 << 25)});
            replace["INIT_RECHARGE_ON_PGFAULT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "false"});
            replace["INIT_THREAD_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1001"});
            replace["INIT_ULIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ""});
            replace["INIT_CORE_COMMAND_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ""});
            replace["INIT_USER_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ""});
            replace["INIT_GROUP_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ""});
            replace["INIT_AGING_TIME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, ToString(1 << 16)});
            replace["INIT_IO_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "/tmp r: 20000"});
            replace["INIT_IO_OPS_LIMIT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "/tmp r: 20001"});
            replace["INIT_IO_POLICY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "normal"});
            replace["INIT_IO_WEIGHT_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1.02"});

            replace["INIT_INITIAL_DELAY_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "0"});
            replace["INIT_RESTART_PERIOD_SCALE_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "0"});
            replace["INIT_RESTART_PERIOD_BACKOFF_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1"});
            replace["INIT_MAX_RESTART_PERIOD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1000"});
            replace["INIT_MIN_RESTART_PERIOD_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "1000"});
            replace["INIT_MAX_EXECUTION_TIME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({2, "60000"});

            replace["BOX_ENVIRONMENT"] = BoxSpecificEnvironment_;

            return replace;
        }

        ui32 GetBoxInitSize() const override {
            return 2;
        }

    private:
        const TString BoxSpecificEnvironment_ = "first_name=first_value;second_name=second_value";
        const TVector<TString> InitSpecificEnvironment_ = {
            "container_name=init0_container"
            , "container_name=init1_container"
        };
        const TVector<TString> InitSpecificSecretEnvironment_ = {
            "secret_key=value1",
            "secret_key=value2"
        };
        const TVector<TString> InitSpecificSecretEnvironmentHidden_ = {
            "secret_key=<secret",
            "secret_key=<secret"
        };
        const TVector<TString> InitStdout_ = {
            "init0 first_value init0_container"
            , "init1 second_value init1_container"
        };
    };

    TTest test("TestInitEnvironment");
    test.DoTest();
}

Y_UNIT_TEST(TestStaticResourceWasDeleted) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
                : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();

            auto breakHookWaitingForStaticResources = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_WAITING_FOR_STATIC_RESOURCES;
            };

            TickTree(Tree_, 32, breakHookWaitingForStaticResources);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_WAITING_FOR_STATIC_RESOURCES, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 32, breakHookReady);
            UNIT_ASSERT_EQUAL_C(BoxStatusRepository_->GetObjectStatus(BoxId_).state(), API::EBoxState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestStaticResourceWasDeleted");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxMetaContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)

        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            if (name == ContainerToFail_) {
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHook = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_INVALID;
            };

            TickTree(Tree_, 60, breakHook);

            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EBoxState_INVALID, API::EBoxState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString containerName = PathHolder_->GetBoxContainer(BoxId_);
            const TString containerType = ToString(NStatusRepositoryTypes::TContainerDescription::EContainerType::META);

            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), containerType);
            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_C(status.fail_counter() > 0, status.fail_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(containerName), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls() > 0, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetBoxContainer(BoxId_));
        }
    };

    TTest test("TestCreateBoxMetaContainerFail");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxRootfsVolumeNotReady) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TString& volumePathToBeNotReady)
            : TWrapperPortoClient(client)
            , VolumePathToBeNotReady_(volumePathToBeNotReady)
            , PatchVolumeState_(false)
        {
        }

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName& container) override {
            auto result = TWrapperPortoClient::ListVolumes(path, container);

            if (PatchVolumeState_ && result) {
                // Patch specific volume state
                for (auto& volume : result.Success()) {
                    if (volume.path() == VolumePathToBeNotReady_) {
                        volume.set_state("unready");
                    }
                }
            }

            return result;
        }

        void SetPatchVolumeState(bool patchVolumeState) {
            PatchVolumeState_ = patchVolumeState;
        }

    private:
        const TString VolumePathToBeNotReady_;
        bool PatchVolumeState_;
    };

    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource(100);

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };
            auto breakHookCreating = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_CREATING_ROOTFS_VOLUME;
            };

            TickTree(Tree_, 60, breakHookReady);
            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(API::EBoxState_READY, status.state(), TConsoleRenderer(false).Render(Tree_));

            ((TTestPortoClient*)TreePorto_.Get())->SetPatchVolumeState(true);
            // One more tick to trigger state check
            TickTree(Tree_, 1, breakHookCreating);
            UNIT_ASSERT_EQUAL_C(
                BoxStatusRepository_->GetObjectStatus(BoxId_).state()
                , API::EBoxState_CREATING_ROOTFS_VOLUME
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetBoxRootfsPath(BoxId_));
        }
    };

    TTest test("TestCreateBoxRootfsVolumeNotReady");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateBoxWithCgroupFsMount) {
    class TTest : public ITestBoxTreeTemplate {
    public:
        TTest(const TString& testName)
            : ITestBoxTreeTemplate(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareVolume();
            PrepareRbindVolumeStorage();
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return BoxStatusRepository_->GetObjectStatus(BoxId_).state() == API::EBoxState_READY;
            };

            TickTree(Tree_, 60, breakHookReady);
            auto status = BoxStatusRepository_->GetObjectStatus(BoxId_);
            UNIT_ASSERT_EQUAL_C(API::EBoxState_READY, status.state(), TConsoleRenderer(false).Render(Tree_));

            const EPortoContainerState boxMetaContainerState = FromString(SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::State).Success());
            UNIT_ASSERT_EQUAL_C(
                EPortoContainerState::Meta
                , boxMetaContainerState
                , ToString(boxMetaContainerState) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            { // test cgroupfs
                const TString cgroupfs = SafePorto_->GetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::Cgroupfs).Success();

                UNIT_ASSERT_EQUAL_C(cgroupfs, "ro", cgroupfs);
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["BOX_CGROUP_FS_MOUNT_TYPE"] = "ro";

            return specificReplace;
        }
    };

    TTest test("TestCreateBoxWithCgroupFsMount");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
