#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/trees/base/test/test_canon.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>
#include <util/system/shellcommand.h>
#include <util/system/user.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestLayerCanon: public ITestBehaviourTreeCanon {
public:
    ITestLayerCanon(const TString &testName)
        : ITestBehaviourTreeCanon(testName, "LayerTree", "LayerTree")
        , LayerId_("MyLayerId")
        , LayerDownloadHash_("MyLayerDownloadHash")
    {
    }

    virtual ~ITestLayerCanon() = default;

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["RESOURCE_DOWNLOAD_QUEUE_LOCK"] = TestName_ + "_" + TestSuiteName_ + "_download_queue_lock";
        replace["RESOURCE_VERIFY_QUEUE_LOCK"] = TestName_ + "_" + TestSuiteName_ + "_verify_queue_lock";

        replace["RESOURCE_GANG_CONTAINER"] = PathHolder_->GetResourceGangMetaContainer();
        replace["RESOURCE_GANG_CONTAINER_LOCK"] = TestName_ + "_" + TestSuiteName_ + "_resource_gang_meta_container_lock";

        replace["RESOURCE_GANG_CPU_GUARANTEE"] = RESOURCE_GANG_CPU_GUARANTEE;
        replace["RESOURCE_GANG_CPU_LIMIT"] = RESOURCE_GANG_CPU_LIMIT;
        replace["RESOURCE_GANG_CPU_POLICY"] = RESOURCE_GANG_CPU_POLICY;
        replace["RESOURCE_GANG_CPU_WEIGHT"] = RESOURCE_GANG_CPU_WEIGHT;

        replace["RESOURCE_GANG_MEMORY_GUARANTEE"] = RESOURCE_GANG_MEMORY_GUARANTEE;
        replace["RESOURCE_GANG_MEMORY_LIMIT"] = RESOURCE_GANG_MEMORY_LIMIT;
        replace["RESOURCE_GANG_ANON_LIMIT"] = RESOURCE_GANG_ANON_LIMIT;
        replace["RESOURCE_GANG_RECHARGE_ON_PGFAULT"] = RESOURCE_GANG_RECHARGE_ON_PGFAULT;

        replace["RESOURCE_GANG_THREAD_LIMIT"] = RESOURCE_GANG_THREAD_LIMIT;

        replace["RESOURCE_GANG_IO_LIMIT"] = RESOURCE_GANG_IO_LIMIT;
        replace["RESOURCE_GANG_IO_OPS_LIMIT"] = RESOURCE_GANG_IO_OPS_LIMIT;
        replace["RESOURCE_GANG_IO_POLICY"] = RESOURCE_GANG_IO_POLICY;
        replace["RESOURCE_GANG_IO_WEIGHT"] = RESOURCE_GANG_IO_WEIGHT;

        replace["LAYER_DOWNLOAD_HASH"] = LayerDownloadHash_;

        replace["LAYER_CONTAINER_USER"] = GetUsername();
        replace["LAYER_CONTAINER_GROUP"] = "porto";
        replace["LAYER_DOWNLOAD_CONTAINER"] = PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download");
        replace["LAYER_VERIFY_CONTAINER"] = PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify");

        replace["LAYER_DOWNLOAD_AGING_TIME"] = ToString(1 << 16);
        replace["LAYER_VERIFY_AGING_TIME"] = ToString(1 << 16);

        replace["LAYER_NAME"] = PathHolder_->GetLayerNameFromHash(LayerDownloadHash_);
        replace["LAYER_PLACE"] = "";

        replace["LAYER_VERIFICATION_CHECKSUM"] = "10240";
        replace["LAYER_VERIFICATION_CMD"] = "bash -c 'export LC_COLLATE=C;find downloaded -type f -print0 | sort -z | xargs -0 du -b | cut -d\"\t\" -f 1 | awk \"{printf \\$0}\"'";
        replace["LAYER_VERIFICATION_TYPE"] = "container";

        replace["LAYER_DOWNLOAD_CMD"] = TStringBuilder()
            << "bash -c 'rm -rf downloaded; rm -rf downloaded_result;"
            << "mkdir downloaded; echo some_data > downloaded/data;"
            << "tar -cf downloaded/tmp_file downloaded/data; rm downloaded/data;"
            << "mkdir downloaded_result; ln -s ../downloaded/tmp_file downloaded_result/layer_link'"
        ;
        replace["NEED_CHECK_DOWNLOAD_PROGRESS"] = "false";

        replace["LAYER_DOWNLOAD_CWD"] = PathHolder_->GetLayerDirectoryFromHash(LayerDownloadHash_, "");
        replace["LAYER_DOWNLOAD_DIRECTORY"] = PathHolder_->GetLayerDownloadDirectoryFromHash(LayerDownloadHash_, "");
        replace["LAYER_FINAL_PATH"] = PathHolder_->GetFinalLayerPathFromHash(LayerDownloadHash_, "");
        replace["LAYER_REMOVE_SOURCE_FILE_AFTER_IMPORT"] = "";
 
        return replace;
    }

    void SetupTest() override {
        ClearPortoLayer();
        PrepareLayer(GetRemoveSourceFileAfterImport());
    }

    virtual bool GetRemoveSourceFileAfterImport() const {
        return false;
    }

private:
    void ClearPortoLayer() {
        SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHash_));
    }

    void PrepareLayer(bool removeSourceFileAfterImport) {
        LayerStatusRepository_->AddObject(
            NObjectMetaTestLib::CreateLayerMetaSimple(
                LayerId_
                , LayerDownloadHash_
                , 1 /* seed */
                , removeSourceFileAfterImport
            )
        );
        LayerStatusRepository_->UpdateActiveDownloadContainersLimit(1);
    }

protected:
    inline static const TString RESOURCE_GANG_CPU_GUARANTEE = "2c";
    inline static const TString RESOURCE_GANG_CPU_LIMIT = "1c";
    inline static const TString RESOURCE_GANG_CPU_POLICY = "normal";
    inline static const TString RESOURCE_GANG_CPU_WEIGHT = "1.01";

    inline static const TString RESOURCE_GANG_MEMORY_GUARANTEE = ToString(1 << 25);
    inline static const TString RESOURCE_GANG_MEMORY_LIMIT = ToString(1 << 26);
    inline static const TString RESOURCE_GANG_ANON_LIMIT = ToString(1 << 27);
    inline static const TString RESOURCE_GANG_RECHARGE_ON_PGFAULT = "false";

    inline static const TString RESOURCE_GANG_THREAD_LIMIT = "1001";

    inline static const TString RESOURCE_GANG_IO_LIMIT = "/tmp r: 20000";
    inline static const TString RESOURCE_GANG_IO_OPS_LIMIT = "/tmp r: 20001";
    inline static const TString RESOURCE_GANG_IO_POLICY = "normal";
    inline static const TString RESOURCE_GANG_IO_WEIGHT = "1.02";

    inline static const TMap<EPortoContainerProperty, TString> RESOURCE_GANG_PROPERTIES = {
        {EPortoContainerProperty::Command, ""}
        , {EPortoContainerProperty::CpuGuarantee, RESOURCE_GANG_CPU_GUARANTEE}
        , {EPortoContainerProperty::CpuLimit, RESOURCE_GANG_CPU_LIMIT}
        , {EPortoContainerProperty::CpuPolicy, RESOURCE_GANG_CPU_POLICY}
        , {EPortoContainerProperty::CpuWeight, RESOURCE_GANG_CPU_WEIGHT}
        , {EPortoContainerProperty::MemoryGuarantee, RESOURCE_GANG_MEMORY_GUARANTEE}
        , {EPortoContainerProperty::MemoryLimit, RESOURCE_GANG_MEMORY_LIMIT}
        , {EPortoContainerProperty::AnonLimit, RESOURCE_GANG_ANON_LIMIT}
        , {EPortoContainerProperty::RechPgfault, RESOURCE_GANG_RECHARGE_ON_PGFAULT}
        , {EPortoContainerProperty::ThreadLimit, RESOURCE_GANG_THREAD_LIMIT}
        , {EPortoContainerProperty::IoLimit, RESOURCE_GANG_IO_LIMIT}
        , {EPortoContainerProperty::IoOpsLimit, RESOURCE_GANG_IO_OPS_LIMIT}
        , {EPortoContainerProperty::IoPolicy, RESOURCE_GANG_IO_POLICY}
        , {EPortoContainerProperty::IoWeight, RESOURCE_GANG_IO_WEIGHT}
    };

    const TString LayerId_;
    const TString LayerDownloadHash_;
};

Y_UNIT_TEST_SUITE(LayerTreeTestSuite) {

Y_UNIT_TEST(TestModifyResourceAfterVerificationStart) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
                : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookVerifiedTwice = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).verification_attempts_counter() > 1;
            };

            TickTree(Tree_, 24, breakHookVerifiedTwice);
            UNIT_ASSERT_C(
                LayerStatusRepository_->GetObjectStatus(LayerId_).verification_attempts_counter() > 1
                , LayerStatusRepository_->GetObjectStatus(LayerId_).verification_attempts_counter() << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            // Emulates the situation when the verification takes more than one second,
            // otherwise this test may fail, since the error of taking the modification time from the resource is 1 second
            specificReplace["LAYER_VERIFICATION_CMD"] = "bash -c 'sleep 2; echo new_data > downloaded/tmp_file; echo 10240'";
            return specificReplace;
        }
    };

    TTest test("TestRemoveObject");
    test.DoTest();
}


Y_UNIT_TEST(TestRemoveLayer) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_READY;
            };

            TickTree(Tree_, 24, breakHook);
            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(API::ELayerState_READY, status.state(), ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_));

            auto result = SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHash_));
            UNIT_ASSERT_C((bool)result, result.Error().Message << Endl << TConsoleRenderer(false).Render(Tree_));
            LayerStatusRepository_->UpdateObjectState(LayerDownloadHash_, API::ELayerState_UNKNOWN);

            TickTree(Tree_, 24, breakHook);
            status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(API::ELayerState_READY, status.state(), ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestRemoveObject");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveFiles) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            const TVector<TString> files = {
                PathHolder_->GetLayersDirectory("") + "/"
                , PathHolder_->GetLayersDirectory("") + "/" + LayerDownloadHash_
                , PathHolder_->GetLayersDirectory("") + "/" + LayerDownloadHash_ + "/downloaded"
                , PathHolder_->GetLayersDirectory("") + "/" + LayerDownloadHash_ + "/downloaded/tmp_file"
                , PathHolder_->GetLayersDirectory("") + "/" + LayerDownloadHash_ + "/downloaded_result"
                , PathHolder_->GetLayersDirectory("") + "/" + LayerDownloadHash_ + "/downloaded_result/layer_link"
            };

            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_READY;
            };

            TickTree(Tree_, 24, breakHook);
            for (auto file : files) {
                NFs::RemoveRecursive(file);
                LayerStatusRepository_->UpdateObjectState(LayerDownloadHash_, API::ELayerState_UNKNOWN);

                TickTree(Tree_, 24, breakHook);

                auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
                UNIT_ASSERT_EQUAL_C(
                    API::ELayerState_READY, status.state()
                    , ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_)
                );
            }
        }
    };

    TTest test("TestRemoveFiles");
    test.DoTest();
}

Y_UNIT_TEST(TestDownloadProgress) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).download_progress().percent() == 57;
            };

            TickTree(Tree_, 40, breakHook);

            ui32 percent = LayerStatusRepository_->GetObjectStatus(LayerId_).download_progress().percent();

            UNIT_ASSERT_EQUAL_C(percent, 57, percent);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_DOWNLOAD_CMD"] = R"(bash -c 'echo "57%" 1>&2; sleep 100')";
            specificReplace["NEED_CHECK_DOWNLOAD_PROGRESS"] = "true";

            return specificReplace;
        }
    };

    TTest test("TestDownloadProgress");
    test.DoTest();
}

Y_UNIT_TEST(TestTmpFileRemoving) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            const TString file = PathHolder_->GetLayersDirectory("") + "/" + LayerDownloadHash_ + "/downloaded/my_file";

            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_READY;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(
                API::ELayerState_READY, status.state()
                , ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_C(!NFs::Exists(file), "file " << file << " not removed by LayerTree");
            UNIT_ASSERT_C(!SafePorto_->ListLayers("", PathHolder_->GetLayerNameFromHash(LayerDownloadHash_)).Success().empty(), "layer " << LayerDownloadHash_ << " does not exist");
        }
    };

    TTest test("TestTmpFileRemoving");
    test.DoTest();
}

Y_UNIT_TEST(TestInvalidUrl) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).download_attempts_counter() > 0
                    && LayerStatusRepository_->GetObjectStatus(LayerId_).failed().message() != ""
                ;
            };

            TickTree(Tree_, 60, breakHook);

            UNIT_ASSERT_C(LayerStatusRepository_->GetObjectStatus(LayerId_).download_attempts_counter() > 0, TConsoleRenderer(false).Render(Tree_));

            TString correctFailedMessage = "exit_code = '4'\nstderr[:1048576] = '--2018-09-07 12:46:58--  http://htts//proxy.sandbox.yandex-team.ru/586812773\nResolving htts (htts)... failed: Name or service not known.\nwget: unable to resolve host address 'htts'\n'\nstdout[:1048576] = 'to_stdout\n'";
            TString failedMessage = LayerStatusRepository_->GetObjectStatus(LayerId_).failed().message();

            UNIT_ASSERT_EQUAL_C(failedMessage.size(), correctFailedMessage.size(), failedMessage);
            UNIT_ASSERT_EQUAL_C(failedMessage.find("--"), correctFailedMessage.find("--"), failedMessage);
            failedMessage.replace(failedMessage.find("--"), 23, "--2018-09-07 12:46:58--");
            for (auto& symbol : failedMessage) {
                if (symbol == '`') {
                    symbol = '\'';
                }
            }
            UNIT_ASSERT_EQUAL_C(failedMessage, correctFailedMessage, failedMessage);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            const TString invalidUrl = "htts//proxy.sandbox.yandex-team.ru/586812773";

            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_DOWNLOAD_CMD"] = R"(bash -c "echo to_stdout; wget --no-check-certificate )" + invalidUrl + '"';

            return specificReplace;
        }
    };

    TTest test("TestInvalidUrl");
    test.DoTest();
}

Y_UNIT_TEST(TestMismatchedCheckSum) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).failed().message() != ""
                    && LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_INVALID
                ;
            };

            TickTree(Tree_, 24, breakHook);

            const auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(
                status.failed().message()
                , "verify bad hash, expected \'" + MismatchedCheckSum_ + "\', got \'10240\'"
                , status.failed().message()
            );
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::ELayerState_INVALID
                , API::ELayerState_Name(status.state())
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_VERIFICATION_CHECKSUM"] = MismatchedCheckSum_;

            return specificReplace;
        }

    private:
        const TString MismatchedCheckSum_ = "3ea15169a95365cbf9b4ff4688c00000";
    };

    TTest test("TestMismatchedCheckSum");
    test.DoTest();
}

Y_UNIT_TEST(TestFileDoesNotExist) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).failed().message() != ""
                    && LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_INVALID
                ;
            };

            TickTree(Tree_, 24, breakHook);

            const auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(
                status.failed().message()
                , TStringBuilder() << "file '" <<  PathHolder_->GetLayersDirectory("") << "/" << LayerDownloadHash_ << "/downloaded_result/layer_link" << "' does not exist."
                , status.failed().message()
            );
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::ELayerState_INVALID
                , API::ELayerState_Name(status.state())
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_DOWNLOAD_CMD"] = "echo none";

            return specificReplace;
        }
    };

    TTest test("TestFileDoesNotExist");
    test.DoTest();
}

Y_UNIT_TEST(TestREADYState) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_READY
                    && !SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success()
                    && !SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success()
                ;
            };
            auto breakHookFalse = []() {
                return false;
            };

            LayerStatusRepository_->UpdateObjectFailedMessage(LayerDownloadHash_, "Fail message must not be clear");
            TickTree(Tree_, 24, breakHook);

            // Layer becomes ready immediately after import to the porto
            // However, source file will be deleted only on the next tick
            // To make sure that the file will not be deleted, we will make two more tree ticks
            TickTree(Tree_, 2, breakHookFalse);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 1, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.state(), API::ELayerState_READY, API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(!SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success(), "Layer download container still exists");
            UNIT_ASSERT_C(!SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success(), "Layer verify container still exists");
            UNIT_ASSERT_C(NFs::Exists(PathHolder_->GetLayerDownloadDirectoryFromHash(LayerDownloadHash_, "")), "Layer download directory not exists");
            UNIT_ASSERT_C(NFs::Exists(PathHolder_->GetFinalLayerPathFromHash(LayerDownloadHash_, "")), "Layer symbolic link not exists");
            UNIT_ASSERT_EQUAL_C(status.failed().message(), "Fail message must not be clear", status.failed().message() << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            {
                // Check resource gang container properties
                TVector<EPortoContainerProperty> keys;
                for (const auto& it : RESOURCE_GANG_PROPERTIES) {
                    keys.push_back(it.first);
                }

                auto result = SafePorto_->Get({PathHolder_->GetResourceGangMetaContainer()}, keys);
                UNIT_ASSERT_C(result, result.Error().Message);
                UNIT_ASSERT_C(result.Success().contains(PathHolder_->GetResourceGangMetaContainer()), "no key, map size: " << result.Success().size());

                TMap<EPortoContainerProperty, TPortoGetResponse> properties = result.Success().at(PathHolder_->GetResourceGangMetaContainer());
                for (const auto& it : RESOURCE_GANG_PROPERTIES) {
                    UNIT_ASSERT_EQUAL_C(it.second, properties[it.first].value(), it.second << " :: " << properties[it.first].value());
                }
            }
        }
    };

    TTest test("TestREADYState");
    test.DoTest();
}

Y_UNIT_TEST(TestREADYStateWithRemoveSourceFileAfterImport) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_READY
                    && !SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success()
                    && !SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success()
                    && !NFs::Exists(PathHolder_->GetLayerDownloadDirectoryFromHash(LayerDownloadHash_, ""))
                    && !NFs::Exists(PathHolder_->GetFinalLayerPathFromHash(LayerDownloadHash_, ""))
                ;
            };

            LayerStatusRepository_->UpdateObjectFailedMessage(LayerDownloadHash_, "Fail message must not be clear");
            TickTree(Tree_, 24, breakHook);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 1, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.state(), API::ELayerState_READY, API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(!SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success(), "Layer download container still exists");
            UNIT_ASSERT_C(!SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success(), "Layer verify container still exists");
            UNIT_ASSERT_C(!NFs::Exists(PathHolder_->GetLayerDownloadDirectoryFromHash(LayerDownloadHash_, "")), "Layer download directory still exists");
            UNIT_ASSERT_C(!NFs::Exists(PathHolder_->GetFinalLayerPathFromHash(LayerDownloadHash_, "")), "Layer symbolic link still exists");
            UNIT_ASSERT_EQUAL_C(status.failed().message(), "Fail message must not be clear", status.failed().message() << Endl << TConsoleRenderer(false).RenderAll(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_REMOVE_SOURCE_FILE_AFTER_IMPORT"] = "true";

            return specificReplace;
        }
    };

    TTest test("TestREADYStateWithRemoveSourceFileAfterImport");
    test.DoTest();
}

Y_UNIT_TEST(TestREADYStateWithSpecificPlace) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareSpecificPlace();

            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_READY;
            };

            LayerStatusRepository_->UpdateObjectFailedMessage(LayerDownloadHash_, "Fail message must not be clear");
            TickTree(Tree_, 24, breakHook);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 1, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.state(), API::ELayerState_READY, API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.failed().message(), "Fail message must not be clear", status.failed().message() << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            {
                const auto layerList = SafePorto_->ListLayers(SpecificPlace_, PathHolder_->GetLayerNameFromHash(LayerDownloadHash_)).Success();
                UNIT_ASSERT_EQUAL_C(layerList.size(), 1, layerList.size());
                UNIT_ASSERT_EQUAL_C(layerList[0].name(), PathHolder_->GetLayerNameFromHash(LayerDownloadHash_), layerList[0].name());
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["LAYER_PLACE"] = SpecificPlace_;
            specificReplace["LAYER_DOWNLOAD_CWD"] = PathHolder_->GetLayerDirectoryFromHash(LayerDownloadHash_, SpecificPlace_);
            specificReplace["LAYER_DOWNLOAD_DIRECTORY"] = PathHolder_->GetLayerDownloadDirectoryFromHash(LayerDownloadHash_, SpecificPlace_);
            specificReplace["LAYER_FINAL_PATH"] = PathHolder_->GetFinalLayerPathFromHash(LayerDownloadHash_, SpecificPlace_);

            return specificReplace;
        }
    };

    TTest test("TestREADYStateWithSpecificPlace");
    test.DoTest();
}

Y_UNIT_TEST(TestInvalidDownloadCmd) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookDownloadAttemptsCounter = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).download_attempts_counter() > 3;
            };

            TickTree(Tree_, 24, breakHookDownloadAttemptsCounter);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::ELayerState_INVALID, API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(status.download_attempts_counter() > 3, status.download_attempts_counter());
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "InvalidCommand:(No such file or directory: cannot exec bad_download_cmd)", status.failed().message());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_DOWNLOAD_CMD"] = "bad_download_cmd";

            return specificReplace;
        }
    };

    TTest test("TestInvalidDownloadCmd");
    test.DoTest();
}

Y_UNIT_TEST(TestInvalidVerificationCmd) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookVerificationAttemptsCounter = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).verification_attempts_counter() > 3;
            };

            TickTree(Tree_, 24, breakHookVerificationAttemptsCounter);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::ELayerState_INVALID, API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(status.verification_attempts_counter() > 3, status.verification_attempts_counter());
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "InvalidCommand:(No such file or directory: cannot exec bad_verification_cmd)", status.failed().message());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_VERIFICATION_CMD"] = "bad_verification_cmd";

            return specificReplace;
        }
    };

    TTest test("TestInvalidVerificationCmd");
    test.DoTest();
}

Y_UNIT_TEST(TestImportLayerFailWithContainerVerificationType) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).fail_counter() > 1;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() > 0, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(status.fail_counter() > 0, status.fail_counter() << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "compression by magic", status.failed().message() << Endl << TConsoleRenderer(false).RenderAll(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_DOWNLOAD_CMD"] = TStringBuilder()
                << "bash -c 'rm -rf downloaded; rm -rf downloaded_result;"
                << "mkdir downloaded; echo bad_data > downloaded/tmp_file;"
                << "mkdir downloaded_result; ln -s ../downloaded/tmp_file downloaded_result/layer_link'"
            ;
            specificReplace["LAYER_VERIFICATION_CHECKSUM"] = "9";

            return specificReplace;
        }
    };

    TTest test("TestImportLayerFailWithContainerVerificationType");
    test.DoTest();
}

Y_UNIT_TEST(TestLayerWithEmptyVerificationType) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_READY;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 0, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT(SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success());
            UNIT_ASSERT(!SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success());
            UNIT_ASSERT_EQUAL(LayerStatusRepository_->GetObjectStatus(LayerId_).state(), API::ELayerState_READY);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_VERIFICATION_TYPE"] = "empty";

            return specificReplace;
        }
    };

    TTest test("TestLayerWithEmptyVerificationType");
    test.DoTest();
}

Y_UNIT_TEST(TestBadLayerFinalPathWithEmptyVerificationType) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_INVALID;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 0, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT(!SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success());
            UNIT_ASSERT(!SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success());
            UNIT_ASSERT_EQUAL(LayerStatusRepository_->GetObjectStatus(LayerId_).state(), API::ELayerState_INVALID);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_VERIFICATION_TYPE"] = "empty";
            specificReplace["LAYER_FINAL_PATH"] = "bad_path";

            return specificReplace;
        }
    };

    TTest test("TestBadLayerFinalPathWithEmptyVerificationType");
    test.DoTest();
}

Y_UNIT_TEST(TestImportLayerFailWithEmptyLayerVerificationType) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_INVALID;
            };

            TickTree(Tree_, 24, breakHook);
            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 0, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT(SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success());
            UNIT_ASSERT(!SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success());
            UNIT_ASSERT_EQUAL(LayerStatusRepository_->GetObjectStatus(LayerId_).state(), API::ELayerState_INVALID);
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "compression by magic", status.failed().message() << Endl << TConsoleRenderer(false).RenderAll(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_VERIFICATION_TYPE"] = "empty";
            specificReplace["LAYER_DOWNLOAD_CMD"] = TStringBuilder()
                << "bash -c 'rm -rf downloaded; rm -rf downloaded_result;"
                << "mkdir downloaded; echo bad_data > downloaded/tmp_file;"
                << "mkdir downloaded_result; ln -s ../downloaded/tmp_file downloaded_result/layer_link'"
            ;

            return specificReplace;
        }
    };

    TTest test("TestImportLayerFailWithEmptyLayerVerificationType");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveObjectFail) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).fail_counter() > 1;
            };

            const TString layerPath = PathHolder_->GetLayersDirectory("") + "/" + LayerDownloadHash_;
            NFs::MakeDirectoryRecursive(layerPath);
            TShellCommand shellCommand(TStringBuilder() << "bash -c \"mkdir "
                                       << layerPath << "/downloaded; echo some_data > " << layerPath << "/downloaded/data; tar -cf "
                                       << layerPath << "/downloaded/tmp_file " << layerPath << "/downloaded/data; rm " << layerPath << "/downloaded/data\"");
            shellCommand.Run();
            shellCommand.Wait();
            UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());

            SafePorto_->ImportLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHash_), TStringBuilder() << PathHolder_->GetLayersDirectory("") << "/" << LayerDownloadHash_ << "/downloaded/tmp_file", false, "", "bad_hash").Success();

            const TString mountVolumeId = "MyMountVolumeId";
            const TString volumePath = PathHolder_->GetVolumePath(mountVolumeId);
            NFs::MakeDirectoryRecursive(volumePath);
            SafePorto_->CreateVolume(volumePath, "", "", {PathHolder_->GetLayerNameFromHash(LayerDownloadHash_)}, 0, "", EPortoVolumeBackend::Auto, {""}, {}, false).Success();

            TickTree(Tree_, 24, breakHook);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_C(status.fail_counter() > 0, status.fail_counter() << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "Busy", status.failed().message() << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            SafePorto_->UnlinkVolume(volumePath);
        }
    };

    TTest test("TestRemoveObjectFail");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateDownloadContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)

        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            if (name == ContainerToFail_) {
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_INVALID;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::ELayerState_INVALID, API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(status.download_attempts_counter() > 0, status.download_attempts_counter() << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString containerName = PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download");
            const TString containerType = ToString(NStatusRepositoryTypes::TContainerDescription::EContainerType::DOWNLOAD);

            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), containerType);
            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_C(status.fail_counter() > 0, status.fail_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(containerName), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls() > 0, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download"));
        }
    };

    TTest test("TestCreateDownloadContainerFail");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateVerifyContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)

        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            if (name == ContainerToFail_) {
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_INVALID;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::ELayerState_INVALID, API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(status.verification_attempts_counter() > 0, status.verification_attempts_counter() << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString containerName = PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify");
            const TString containerType = ToString(NStatusRepositoryTypes::TContainerDescription::EContainerType::VERIFY);

            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), containerType);
            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_C(status.fail_counter() > 0, status.fail_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(containerName), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls() > 0, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify"));
        }
    };

    TTest test("TestCreateVerifyContainerFail");
    test.DoTest();
}

Y_UNIT_TEST(TestDownloadQueue) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookInQueue = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_IN_QUEUE;
            };
            auto breakHookReady = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_READY;
            };

            {
                // Create correct resource gang meta container
                TickTree(Tree_, 24, breakHookReady);
                auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
                UNIT_ASSERT_EQUAL_C(
                    status.state()
                    , API::ELayerState_READY
                    , API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
                );

                // Completely remove layer
                SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHash_)).Success();
                // We can't guarantee that download and verify containers will exist
                // but if they do, we must destroy them
                if (SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success()) {
                    SafePorto_->Destroy(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success();
                }
                if (SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success()) {
                    SafePorto_->Destroy(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success();
                }
                NFs::RemoveRecursive(PathHolder_->GetLayersDirectory(""));
            }

            // Create fake active download containers
            const ui32 queueSize = 3;
            LayerStatusRepository_->UpdateActiveDownloadContainersLimit(queueSize);
            for (ui32 i = 0; i < queueSize; ++i) {
                const auto containerName = PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_ + ToString(i), "download");
                SafePorto_->Create(containerName).Success();
                SafePorto_->SetProperty(containerName, EPortoContainerProperty::Command, "sleep 1000").Success();
                SafePorto_->Start(containerName).Success();
            }

            TickTree(Tree_, 24, breakHookInQueue);
            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::ELayerState_IN_QUEUE
                , API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
            );

            {
                // Kill one of fake download containers
                // After that there should be a free space to start new container
                const auto containerName = PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_ + ToString(0), "download");
                SafePorto_->Kill(containerName, SIGKILL).Success();
            }

            TickTree(Tree_, 24, breakHookReady);
            status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::ELayerState_READY
                , API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
            );
        }
    };

    TTest test("TestDownloadQueue");
    test.DoTest();
}

Y_UNIT_TEST(TestVerifyQueue) {
    class TTest : public ITestLayerCanon {
    public:
        TTest(const TString& testName)
            : ITestLayerCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookInQueue = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_VERIFICATION_IN_QUEUE;
            };
            auto breakHookReady = [this]() {
                return LayerStatusRepository_->GetObjectStatus(LayerId_).state() == API::ELayerState_READY;
            };

            {
                // Create correct resource gang meta container
                TickTree(Tree_, 100, breakHookReady);
                auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
                UNIT_ASSERT_EQUAL_C(
                    status.state()
                    , API::ELayerState_READY
                    , API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
                );

                // Completely remove layer
                SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHash_)).Success();
                // We can't guarantee that download and verify containers will exist
                // but if they do, we must destroy them
                if (SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success()) {
                    SafePorto_->Destroy(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "download")).Success();
                }
                if (SafePorto_->IsContainerExists(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success()) {
                    SafePorto_->Destroy(PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_, "verify")).Success();
                }
                NFs::RemoveRecursive(PathHolder_->GetLayersDirectory(""));
            }

            // Create fake active download containers
            const ui32 queueSize = 3;
            LayerStatusRepository_->UpdateActiveVerifyContainersLimit(queueSize);
            for (ui32 i = 0; i < queueSize; ++i) {
                const auto containerName = PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_ + ToString(i), "verify");
                SafePorto_->Create(containerName).Success();
                SafePorto_->SetProperty(containerName, EPortoContainerProperty::Command, "sleep 1000").Success();
                SafePorto_->Start(containerName).Success();
            }

            TickTree(Tree_, 100, breakHookInQueue);
            auto status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::ELayerState_VERIFICATION_IN_QUEUE
                , API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
            );

            {
                // Kill one of fake download containers
                // After that there should be a free space to start new container
                const auto containerName = PathHolder_->GetLayerContainerWithNameFromHash(LayerDownloadHash_ + ToString(0), "verify");
                SafePorto_->Kill(containerName, SIGKILL).Success();
            }

            TickTree(Tree_, 100, breakHookReady);
            status = LayerStatusRepository_->GetObjectStatus(LayerId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::ELayerState_READY
                , API::ELayerState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
            );
            UNIT_ASSERT_C(NFs::Exists(PathHolder_->GetLayerDownloadDirectoryFromHash(LayerDownloadHash_, "")), "Layer download directory not exists");
            UNIT_ASSERT_C(NFs::Exists(PathHolder_->GetFinalLayerPathFromHash(LayerDownloadHash_, "")), "Layer symbolic link not exists");
        }
    };

    TTest test("TestVerifyQueue");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
