#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/trees/base/test/test_canon.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>
#include <infra/pod_agent/libs/posix_worker/posix_types.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>
#include <util/system/fstat.h>
#include <util/system/user.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestStaticResourceCanon: public ITestBehaviourTreeCanon {
public:
    ITestStaticResourceCanon(const TString &testName)
        : ITestBehaviourTreeCanon(testName, "StaticResourceTree", "StaticResourceTree")
        , StaticResourceId_("MyStaticResourceId")
        , StaticResourceDownloadHash_("MyStaticResourceDownloadHash")
    {
    }

    virtual ~ITestStaticResourceCanon() = default;

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["RESOURCE_DOWNLOAD_QUEUE_LOCK"] = TestName_ + "_" + TestSuiteName_ + "_download_queue_lock";
        replace["RESOURCE_VERIFY_QUEUE_LOCK"] = TestName_ + "_" + TestSuiteName_ + "_verify_queue_lock";

        replace["RESOURCE_GANG_CONTAINER"] = PathHolder_->GetResourceGangMetaContainer();
        replace["RESOURCE_GANG_CONTAINER_LOCK"] = TestName_ + "_" + TestSuiteName_ + "_resource_gang_meta_container_lock";

        replace["RESOURCE_GANG_CPU_GUARANTEE"] = RESOURCE_GANG_CPU_GUARANTEE;
        replace["RESOURCE_GANG_CPU_LIMIT"] = RESOURCE_GANG_CPU_LIMIT;
        replace["RESOURCE_GANG_CPU_POLICY"] = RESOURCE_GANG_CPU_POLICY;
        replace["RESOURCE_GANG_CPU_WEIGHT"] = RESOURCE_GANG_CPU_WEIGHT;

        replace["RESOURCE_GANG_MEMORY_GUARANTEE"] = RESOURCE_GANG_MEMORY_GUARANTEE;
        replace["RESOURCE_GANG_MEMORY_LIMIT"] = RESOURCE_GANG_MEMORY_LIMIT;
        replace["RESOURCE_GANG_ANON_LIMIT"] = RESOURCE_GANG_ANON_LIMIT;
        replace["RESOURCE_GANG_RECHARGE_ON_PGFAULT"] = RESOURCE_GANG_RECHARGE_ON_PGFAULT;

        replace["RESOURCE_GANG_THREAD_LIMIT"] = RESOURCE_GANG_THREAD_LIMIT;

        replace["RESOURCE_GANG_IO_LIMIT"] = RESOURCE_GANG_IO_LIMIT;
        replace["RESOURCE_GANG_IO_OPS_LIMIT"] = RESOURCE_GANG_IO_OPS_LIMIT;
        replace["RESOURCE_GANG_IO_POLICY"] = RESOURCE_GANG_IO_POLICY;
        replace["RESOURCE_GANG_IO_WEIGHT"] = RESOURCE_GANG_IO_WEIGHT;

        replace["STATIC_RESOURCE_DOWNLOAD_HASH"] = StaticResourceDownloadHash_;

        replace["STATIC_RESOURCE_CONTAINER_USER"] = GetUsername();
        replace["STATIC_RESOURCE_CONTAINER_GROUP"] = "porto";
        replace["STATIC_RESOURCE_DOWNLOAD_CONTAINER"] = PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download");
        replace["STATIC_RESOURCE_VERIFY_CONTAINER"] = PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify");

        replace["STATIC_RESOURCE_DOWNLOAD_AGING_TIME"] = ToString(1 << 16);
        replace["STATIC_RESOURCE_VERIFY_AGING_TIME"] = ToString(1 << 16);

        replace["STATIC_RESOURCE_VERIFICATION_CHECKSUM"] = "10240";
        replace["STATIC_RESOURCE_VERIFICATION_CMD"] = "bash -c 'export LC_COLLATE=C;find downloaded -type f -print0 | sort -z | xargs -0 du -b | cut -d\"\t\" -f 1 | awk \"{printf \\$0}\"'";
        replace["STATIC_RESOURCE_VERIFICATION_TYPE"] = "container";

        replace["STATIC_RESOURCE_DOWNLOAD_CMD"] = TStringBuilder()
            << "bash -c 'rm -rf downloaded; rm -rf downloaded_result;"
            << "mkdir downloaded; echo some_data > downloaded/data;"
            << "tar -cf downloaded/tmp_file downloaded/data; rm downloaded/data;'"
        ;
        replace["NEED_CHECK_DOWNLOAD_PROGRESS"] = "false";

        replace["STATIC_RESOURCE_DOWNLOAD_CWD"] = PathHolder_->GetStaticResourceDirectoryFromHash(StaticResourceDownloadHash_, "");
        replace["STATIC_RESOURCE_DOWNLOAD_DIRECTORY"] =  PathHolder_->GetStaticResourceDownloadDirectoryFromHash(StaticResourceDownloadHash_, "");
        replace["STATIC_RESOURCE_FINAL_PATH"] =  PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHash_, "");
        replace["STATIC_RESOURCE_ACCESS_MODE"] = "";
        replace["STATIC_RESOURCE_GROUP_ID"] = "";
        replace["STATIC_RESOURCE_SECRET_ENVIRONMENT"]="";
        return replace;
    }

    void SetupTest() override {
        ClearStaticResourcePath("");
        ClearStaticResourcePath(SpecificPlace_);
        PrepareStaticResource(GetCheckPeriodMs());
    }

    virtual ui64 GetCheckPeriodMs() const {
        return 100;
    }

private:
    void ClearStaticResourcePath(const TString& place = "") {
        NFs::Remove(PathHolder_->GetStaticResourceDownloadDirectoryFromHash(StaticResourceDownloadHash_, place));
        NFs::Remove(PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHash_, place));
    }

    void PrepareStaticResource(ui64 checkPeriodMs) {
        StaticResourceStatusRepository_->AddObject(
            TStaticResourceMeta(
                StaticResourceId_
                , 0
                , 0
                , StaticResourceDownloadHash_
                , checkPeriodMs
            )
        );
        StaticResourceStatusRepository_->UpdateActiveDownloadContainersLimit(1);
    }

protected:
    inline static const TString RESOURCE_GANG_CPU_GUARANTEE = "2c";
    inline static const TString RESOURCE_GANG_CPU_LIMIT = "1c";
    inline static const TString RESOURCE_GANG_CPU_POLICY = "normal";
    inline static const TString RESOURCE_GANG_CPU_WEIGHT = "1.01";

    inline static const TString RESOURCE_GANG_MEMORY_GUARANTEE = ToString(1 << 25);
    inline static const TString RESOURCE_GANG_MEMORY_LIMIT = ToString(1 << 26);
    inline static const TString RESOURCE_GANG_ANON_LIMIT = ToString(1 << 27);
    inline static const TString RESOURCE_GANG_RECHARGE_ON_PGFAULT = "false";

    inline static const TString RESOURCE_GANG_THREAD_LIMIT = "1001";

    inline static const TString RESOURCE_GANG_IO_LIMIT = "/tmp r: 20000";
    inline static const TString RESOURCE_GANG_IO_OPS_LIMIT = "/tmp r: 20001";
    inline static const TString RESOURCE_GANG_IO_POLICY = "normal";
    inline static const TString RESOURCE_GANG_IO_WEIGHT = "1.02";

    inline static const TMap<EPortoContainerProperty, TString> RESOURCE_GANG_PROPERTIES = {
        {EPortoContainerProperty::Command, ""}
        , {EPortoContainerProperty::CpuGuarantee, RESOURCE_GANG_CPU_GUARANTEE}
        , {EPortoContainerProperty::CpuLimit, RESOURCE_GANG_CPU_LIMIT}
        , {EPortoContainerProperty::CpuPolicy, RESOURCE_GANG_CPU_POLICY}
        , {EPortoContainerProperty::CpuWeight, RESOURCE_GANG_CPU_WEIGHT}
        , {EPortoContainerProperty::MemoryGuarantee, RESOURCE_GANG_MEMORY_GUARANTEE}
        , {EPortoContainerProperty::MemoryLimit, RESOURCE_GANG_MEMORY_LIMIT}
        , {EPortoContainerProperty::AnonLimit, RESOURCE_GANG_ANON_LIMIT}
        , {EPortoContainerProperty::RechPgfault, RESOURCE_GANG_RECHARGE_ON_PGFAULT}
        , {EPortoContainerProperty::ThreadLimit, RESOURCE_GANG_THREAD_LIMIT}
        , {EPortoContainerProperty::IoLimit, RESOURCE_GANG_IO_LIMIT}
        , {EPortoContainerProperty::IoOpsLimit, RESOURCE_GANG_IO_OPS_LIMIT}
        , {EPortoContainerProperty::IoPolicy, RESOURCE_GANG_IO_POLICY}
        , {EPortoContainerProperty::IoWeight, RESOURCE_GANG_IO_WEIGHT}
    };

    const TString StaticResourceId_;
    const TString StaticResourceDownloadHash_;
};

Y_UNIT_TEST_SUITE(StaticResourceTreeTestSuite) {

Y_UNIT_TEST(TestModifyExistingResourceFileAfterVerification) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            TPortoContainerName verifyContainerName = TPortoContainerName(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify"));

            auto breakHookReady = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY;
            };

            TickTree(Tree_, 24, breakHookReady);
            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(
                API::EStaticResourceState_READY
                , status.state()
                , EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.verification_attempts_counter()
                , 1
                , status.verification_attempts_counter() << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            const TString staticResourceFilePath = PathHolder_->GetStaticResourceDownloadDirectoryFromHash(StaticResourceDownloadHash_, "") + "/tmp_file";

            UNIT_ASSERT_EQUAL((int)TFileStat(staticResourceFilePath).Mode & 0x00000FFF, 0600);

            TUnbufferedFileOutput(staticResourceFilePath).Write("new_data");

            auto breakHookVerifiedTwice = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).verification_attempts_counter() > 1
                    && StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY
                ;
            };

            TickTree(Tree_, 40, breakHookVerifiedTwice);
            status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(
                API::EStaticResourceState_READY
                , status.state()
                , EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_C(
                status.verification_attempts_counter() > 1
                , status.verification_attempts_counter() << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            UNIT_ASSERT_EQUAL((int)TFileStat(staticResourceFilePath).Mode & 0x00000FFF, 0600);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            // Emulates the situation when the verification takes more than one second,
            // otherwise this test may fail, since the error of taking the modification time from the resource is 1 second
            specificReplace["STATIC_RESOURCE_VERIFICATION_CMD"] = "bash -c 'export LC_COLLATE=C;sleep 2;find downloaded -type f -print0 | sort -z | xargs -0 du -b | cut -d\"\t\" -f 1 | awk \"{printf \\$0}\"'";
            specificReplace["STATIC_RESOURCE_ACCESS_MODE"] = ToString(EFileAccessMode::Mode_600);
            return specificReplace;
        }
    };

    TTest test("TestModifyExistingResourceFileAfterVerification");
    test.DoTest();
}

Y_UNIT_TEST(TestAddNewResourceResourceFileAfterVerification) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            TPortoContainerName verifyContainerName = TPortoContainerName(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify"));

            auto breakHookReady = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY;
            };

            TickTree(Tree_, 24, breakHookReady);
            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(
                API::EStaticResourceState_READY
                , status.state()
                , EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.verification_attempts_counter()
                , 1
                , status.verification_attempts_counter() << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TUnbufferedFileOutput(PathHolder_->GetStaticResourceDownloadDirectoryFromHash(StaticResourceDownloadHash_, "") + "/new_file").Write("new_data");

            auto breakHookVerifiedTwice = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).verification_attempts_counter() > 1
                    && StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY
                ;
            };

            TickTree(Tree_, 40, breakHookVerifiedTwice);
            status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(
                API::EStaticResourceState_READY
                , status.state()
                , EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_C(
                status.verification_attempts_counter() > 1
                , status.verification_attempts_counter() << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            // Emulates the situation when the verification takes more than one second,
            // otherwise this test may fail, since the error of taking the modification time from the resource is 1 second
            specificReplace["STATIC_RESOURCE_VERIFICATION_CMD"] = "bash -c 'export LC_COLLATE=C;sleep 2;find downloaded -type f -print0 | sort -z | xargs -0 du -b | cut -d\"\t\" -f 1 | awk \"{printf \\$0}\"'";

            return specificReplace;
        }
    };

    TTest test("TestAddNewResourceResourceFileAfterVerification");
    test.DoTest();
}

Y_UNIT_TEST(TestREADYState) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY
                    && !SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download")).Success()
                ;
            };

            StaticResourceStatusRepository_->UpdateObjectFailedMessage(StaticResourceDownloadHash_, "Fail message must not be clear");

            TickTree(Tree_, 24, breakHook);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 1, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_READY, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT(!SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download")).Success());
            UNIT_ASSERT_EQUAL_C(status.failed().message(), "Fail message must not be clear", status.failed().message() << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            {
                // Check resource gang container properties
                TVector<EPortoContainerProperty> keys;
                for (const auto& it : RESOURCE_GANG_PROPERTIES) {
                    keys.push_back(it.first);
                }

                auto result = SafePorto_->Get({PathHolder_->GetResourceGangMetaContainer()}, keys);
                UNIT_ASSERT_C(result, result.Error().Message);
                UNIT_ASSERT_C(result.Success().contains(PathHolder_->GetResourceGangMetaContainer()), "no key, map size: " << result.Success().size());

                TMap<EPortoContainerProperty, TPortoGetResponse> properties = result.Success().at(PathHolder_->GetResourceGangMetaContainer());
                for (const auto& it : RESOURCE_GANG_PROPERTIES) {
                    UNIT_ASSERT_EQUAL_C(it.second, properties[it.first].value(), it.second << " :: " << properties[it.first].value());
                }
            }
        }

        ui64 GetCheckPeriodMs() const override {
            return 600 * 1000;
        }
    };

    TTest test("TestREADYState");
    test.DoTest();
}

Y_UNIT_TEST(TestREADYStateWithSpecificPlace) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareSpecificPlace();

            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY;
            };

            StaticResourceStatusRepository_->UpdateObjectFailedMessage(StaticResourceDownloadHash_,"Fail message must not be clear");
            TickTree(Tree_, 24, breakHook);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 1, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_READY, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.failed().message(), "Fail message must not be clear", status.failed().message() << Endl << TConsoleRenderer(false).RenderAll(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_DOWNLOAD_CWD"] = PathHolder_->GetStaticResourceDirectoryFromHash(StaticResourceDownloadHash_, SpecificPlace_);
            specificReplace["STATIC_RESOURCE_DOWNLOAD_DIRECTORY"] =  PathHolder_->GetStaticResourceDownloadDirectoryFromHash(StaticResourceDownloadHash_, SpecificPlace_);
            specificReplace["STATIC_RESOURCE_FINAL_PATH"] =  PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHash_, SpecificPlace_);

            return specificReplace;
        }
    };

    TTest test("TestREADYStateWithSpecificPlace");
    test.DoTest();
}

Y_UNIT_TEST(TestUseMinVerificationPeriod) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookReady = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY;
            };

            auto breakHookVerifiedTwice = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).verification_attempts_counter() > 1;
            };

            TickTree(Tree_, 24, breakHookReady);
            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 1, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_READY, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            // Verification period is 10 minutes, should be 1 verification attempt
            TickTree(Tree_, 6, breakHookVerifiedTwice);
            status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() == 1, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_READY, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            StaticResourceStatusRepository_->AddCacheObject(
                TStaticResourceMeta(
                    StaticResourceId_ + "_other"
                    , 0
                    , 0
                    , StaticResourceDownloadHash_
                    , 0
                )
             );

            // Verification period is 0, should be more than 1 verification attempt
            TickTree(Tree_, 6, breakHookVerifiedTwice);
            status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_C(status.verification_attempts_counter() > 1, Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_READY, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
        }

        ui64 GetCheckPeriodMs() const override {
            return 600 * 1000;
        }
    };

    TTest test("TestUseMinVerificationPeriod");
    test.DoTest();
}

Y_UNIT_TEST(TestModifyStaticResourceWhilePeriodicVerification) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {

            auto breakHookResourceReady = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY;
            };

            TickTree(Tree_, 24, breakHookResourceReady);

            const TString staticResourceFinalPath = PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHash_, "");

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_READY, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT(NFs::Exists(staticResourceFinalPath));

            TUnbufferedFileOutput(PathHolder_->GetStaticResourceDownloadDirectoryFromHash(StaticResourceDownloadHash_, "") + "/tmp_file").Write("new_data");

            auto breakHookResourceInvalid = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_INVALID;
            };

            TickTree(Tree_, 24, breakHookResourceInvalid);
            status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_INVALID, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT(!NFs::Exists(staticResourceFinalPath));

            TickTree(Tree_, 24, breakHookResourceReady);
            status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_READY, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT(NFs::Exists(staticResourceFinalPath));
        }
    };

    TTest test("TestModifyStaticResourceWhilePeriodicVerification");
    test.DoTest();
}

Y_UNIT_TEST(TestDownloadProgress) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).download_progress().percent() == 57;
            };

            TickTree(Tree_, 40, breakHook);

            ui32 percent = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).download_progress().percent();

            UNIT_ASSERT_EQUAL_C(percent, 57, percent);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_DOWNLOAD_CMD"] = R"(bash -c 'echo "57%" 1>&2; sleep 100')";
            specificReplace["NEED_CHECK_DOWNLOAD_PROGRESS"] = "true";

            return specificReplace;
        }
    };

    TTest test("TestDownloadProgress");
    test.DoTest();
}

Y_UNIT_TEST(TestInvalidUrl) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).failed().message() != "";
            };

            TickTree(Tree_, 60, breakHook);

            TString correctFailedMessage = "exit_code = '4'\nstderr[:1048576] = '--2018-09-07 12:46:58--  http://htts//proxy.sandbox.yandex-team.ru/586812773\nResolving htts (htts)... failed: Name or service not known.\nwget: unable to resolve host address 'htts'\n'\nstdout[:1048576] = 'to_stdout\n'";
            TString failedMessage = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).failed().message();

            UNIT_ASSERT_EQUAL_C(failedMessage.size(), correctFailedMessage.size(), failedMessage << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(failedMessage.find("--"), correctFailedMessage.find("--"), failedMessage << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            failedMessage.replace(failedMessage.find("--"), 23, "--2018-09-07 12:46:58--");
            for (auto& symbol : failedMessage) {
                if (symbol == '`') {
                    symbol = '\'';
                }
            }
            UNIT_ASSERT_EQUAL_C(failedMessage, correctFailedMessage, failedMessage << Endl << TConsoleRenderer(false).RenderAll(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            const TString invalidUrl = "htts//proxy.sandbox.yandex-team.ru/586812773";

            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_DOWNLOAD_CMD"] = R"(bash -c "echo to_stdout; wget --no-check-certificate )" + invalidUrl + '"';
            specificReplace["NEED_CHECK_DOWNLOAD_PROGRESS"] = "true";

            return specificReplace;
        }
    };

    TTest test("TestInvalidUrl");
    test.DoTest();
}

Y_UNIT_TEST(TestMismatchedCheckSum) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).failed().message() != ""
                    && StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_INVALID
                ;
            };

            TickTree(Tree_, 24, breakHook);

            const auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(
                status.failed().message()
                , "verify bad hash, expected \'" + MismatchedCheckSum_ + "\', got \'10240\'"
                , status.failed().message()
            );
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EStaticResourceState_INVALID
                , API::EStaticResourceState_Name(status.state())
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_VERIFICATION_CHECKSUM"] = MismatchedCheckSum_;

            return specificReplace;
        }

    private:
        const TString MismatchedCheckSum_ = "3ea15169a95365cbf9b4ff4688c00000";
    };

    TTest test("TestMismatchedCheckSum");
    test.DoTest();
}

Y_UNIT_TEST(TestInvalidDownloadCmd) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookDownloadAttemptsCounter = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).download_attempts_counter() > 3;
            };

            TickTree(Tree_, 24, breakHookDownloadAttemptsCounter);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_INVALID, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(status.download_attempts_counter() > 3, status.download_attempts_counter());
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "InvalidCommand:(No such file or directory: cannot exec bad_download_cmd)", status.failed().message());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_DOWNLOAD_CMD"] = "bad_download_cmd";

            return specificReplace;
        }
    };

    TTest test("TestInvalidDownloadCmd");
    test.DoTest();
}

Y_UNIT_TEST(TestInvalidVerificationCmd) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookVerificationAttemptsCounter = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).verification_attempts_counter() > 3;
            };

            TickTree(Tree_, 24, breakHookVerificationAttemptsCounter);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_INVALID, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(status.verification_attempts_counter() > 3, status.verification_attempts_counter());
            UNIT_ASSERT_STRING_CONTAINS_C(status.failed().message(), "InvalidCommand:(No such file or directory: cannot exec bad_verification_cmd)", status.failed().message());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_VERIFICATION_CMD"] = "bad_verification_cmd";

            return specificReplace;
        }
    };

    TTest test("TestInvalidVerificationCmd");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateDownloadContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)

        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            if (name == ContainerToFail_) {
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_INVALID;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_INVALID, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(status.download_attempts_counter() > 0, status.download_attempts_counter() << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString containerName = PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download");
            const TString containerType = ToString(NStatusRepositoryTypes::TContainerDescription::EContainerType::DOWNLOAD);

            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), containerType);
            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_C(status.fail_counter() > 0, status.fail_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(containerName), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls() > 0, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download"));
        }
    };

    TTest test("TestCreateDownloadContainerFail");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateVerifyContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)

        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            if (name == ContainerToFail_) {
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_INVALID;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_INVALID, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_C(status.verification_attempts_counter() > 0, status.verification_attempts_counter() << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString containerName = PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify");
            const TString containerType = ToString(NStatusRepositoryTypes::TContainerDescription::EContainerType::VERIFY);

            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), containerType);
            UNIT_ASSERT_STRING_CONTAINS(status.failed().message(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_C(status.fail_counter() > 0, status.fail_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(containerName), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls() > 0, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify"));
        }
    };

    TTest test("TestCreateVerifyContainerFail");
    test.DoTest();
}

Y_UNIT_TEST(TestDownloadQueue) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookInQueue = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_IN_QUEUE;
            };
            auto breakHookReady = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY;
            };

            {
                // Create correct resource gang meta container
                TickTree(Tree_, 24, breakHookReady);
                auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
                UNIT_ASSERT_EQUAL_C(
                    status.state()
                    , API::EStaticResourceState_READY
                    , API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
                );

                // Completely remove static resource

                // We can't guarantee that download container will exist
                // but if it does, we must destroy it
                if (SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download")).Success()) {
                    SafePorto_->Destroy(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download")).Success();
                }

                SafePorto_->Destroy(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify")).Success();
                NFs::RemoveRecursive(PathHolder_->GetStaticResourcesDirectory(""));
            }

            // Create fake active download containers
            const ui32 queueSize = 3;
            StaticResourceStatusRepository_->UpdateActiveDownloadContainersLimit(queueSize);
            for (ui32 i = 0; i < queueSize; ++i) {
                const auto containerName = PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_ + ToString(i), "download");
                SafePorto_->Create(containerName).Success();
                SafePorto_->SetProperty(containerName, EPortoContainerProperty::Command, "sleep 1000").Success();
                SafePorto_->Start(containerName).Success();
            }

            TickTree(Tree_, 24, breakHookInQueue);
            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EStaticResourceState_IN_QUEUE
                , API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
            );

            {
                // Kill one of fake download containers
                // After that there should be a free space to start new container
                const auto containerName = PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_ + ToString(0), "download");
                SafePorto_->Kill(containerName, SIGKILL).Success();
            }

            TickTree(Tree_, 24, breakHookReady);
            status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EStaticResourceState_READY
                , API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
            );
        }

        // To make sure that verify container exists after verification
        ui64 GetCheckPeriodMs() const override {
            return 600 * 1000;
        }
    };

    TTest test("TestDownloadQueue");
    test.DoTest();
}

Y_UNIT_TEST(TestVerifyQueue) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookInQueue = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_VERIFICATION_IN_QUEUE;
            };
            auto breakHookReady = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY;
            };

            {
                // Create correct resource gang meta container
                TickTree(Tree_, 100, breakHookReady);
                auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
                UNIT_ASSERT_EQUAL_C(
                    status.state()
                    , API::EStaticResourceState_READY
                    , API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
                );

                // Completely remove static resource

                // We can't guarantee that download container will exist
                // but if it does, we must destroy it
                if (SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download")).Success()) {
                    SafePorto_->Destroy(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download")).Success();
                }
                if (SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify")).Success()) {
                    SafePorto_->Destroy(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify")).Success();
                }

                NFs::RemoveRecursive(PathHolder_->GetStaticResourcesDirectory(""));
            }

            // Create fake active verify containers
            const ui32 queueSize = 3;
            StaticResourceStatusRepository_->UpdateActiveVerifyContainersLimit(queueSize);
            for (ui32 i = 0; i < queueSize; ++i) {
                const auto containerName = PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_ + ToString(i), "verify");
                SafePorto_->Create(containerName).Success();
                SafePorto_->SetProperty(containerName, EPortoContainerProperty::Command, "sleep 1000").Success();
                SafePorto_->Start(containerName).Success();
            }

            TickTree(Tree_, 100, breakHookInQueue);
            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EStaticResourceState_VERIFICATION_IN_QUEUE
                , API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
            );

            {
                // Kill one of fake verify containers
                // After that there should be a free space to start new container
                const auto containerName = PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_ + ToString(0), "verify");
                SafePorto_->Kill(containerName, SIGKILL).Success();
            }

            TickTree(Tree_, 100, breakHookReady);
            status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EStaticResourceState_READY
                , API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
            );
            UNIT_ASSERT(NFs::Exists(PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHash_, "")));
        }

        // To make sure that verify container exists after verification
        ui64 GetCheckPeriodMs() const override {
            return 600 * 1000;
        }
    };

    TTest test("TestVerifyQueue");
    test.DoTest();
}

Y_UNIT_TEST(TestStaticResourceWithEmptyVerificationType) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_READY, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT_EQUAL_C(status.verification_attempts_counter(), 0, status.verification_attempts_counter());
            UNIT_ASSERT(!SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify")).Success());
            UNIT_ASSERT(NFs::Exists(PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHash_, "")));

            const TString staticResourceFilePath = PathHolder_->GetStaticResourceDownloadDirectoryFromHash(StaticResourceDownloadHash_, "") + "/tmp_file";
            UNIT_ASSERT_EQUAL((int)TFileStat(staticResourceFilePath).Mode & 0x00000FFF, 0600);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_VERIFICATION_TYPE"] = "empty";
            specificReplace["STATIC_RESOURCE_ACCESS_MODE"] = ToString(EFileAccessMode::Mode_600);

            return specificReplace;
        }
    };

    TTest test("TestStaticResourceWithEmptyVerificationType");
    test.DoTest();
}

Y_UNIT_TEST(TestStaticResourceWithUnmodifiedAccessMode) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_READY;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_READY, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString staticResourceFilePath = PathHolder_->GetStaticResourceDownloadDirectoryFromHash(StaticResourceDownloadHash_, "") + "/tmp_file";
            UNIT_ASSERT_EQUAL((int)TFileStat(staticResourceFilePath).Mode & 0x00000FFF, 0660);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_VERIFICATION_TYPE"] = "empty";
            specificReplace["STATIC_RESOURCE_ACCESS_MODE"] = "";
            // we emulate situation when static resource downloaded with custom permissions(660) and we dont touch it when access permissions are unmodified(empty string for replace map)
            specificReplace["STATIC_RESOURCE_DOWNLOAD_CMD"] = TStringBuilder()
            << "bash -c 'rm -rf downloaded; rm -rf downloaded_result;"
            << "mkdir downloaded; echo some_data > downloaded/data;"
            << "tar -cf downloaded/tmp_file downloaded/data; chmod 660 downloaded/tmp_file; rm downloaded/data;'";

            return specificReplace;
        }
    };

    TTest test("TestStaticResourceWithUnmodifiedAccessMode");
    test.DoTest();
}


Y_UNIT_TEST(TestBadStaticResourceDownloadDirectoryWithEmptyVerificationType) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_INVALID;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.verification_attempts_counter(), 0, status.verification_attempts_counter());
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_INVALID, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT(!SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify")).Success());
            UNIT_ASSERT(!SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download")).Success());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_VERIFICATION_TYPE"] = "empty";
            specificReplace["STATIC_RESOURCE_DOWNLOAD_DIRECTORY"] = "bad_directory";

            return specificReplace;
        }
    };

    TTest test("TestBadStaticResourceDownloadDirectoryWithEmptyVerificationType");
    test.DoTest();
}

Y_UNIT_TEST(TestBadStaticResourceFinalPathWithEmptyVerificationType) {
    class TTest : public ITestStaticResourceCanon {
    public:
        TTest(const TString& testName)
            : ITestStaticResourceCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_).state() == API::EStaticResourceState_INVALID;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = StaticResourceStatusRepository_->GetObjectStatus(StaticResourceId_);
            UNIT_ASSERT_EQUAL_C(status.verification_attempts_counter(), 0, status.verification_attempts_counter());
            UNIT_ASSERT_EQUAL_C(status.state(), API::EStaticResourceState_INVALID, API::EStaticResourceState_Name(status.state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));
            UNIT_ASSERT(!SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "verify")).Success());
            UNIT_ASSERT(SafePorto_->IsContainerExists(PathHolder_->GetStaticResourceContainerWithNameFromHash(StaticResourceDownloadHash_, "download")).Success());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STATIC_RESOURCE_VERIFICATION_TYPE"] = "empty";
            specificReplace["STATIC_RESOURCE_FINAL_PATH"] = PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHash_, SpecificPlace_);

            return specificReplace;
        }
    };

    TTest test("TestBadStaticResourceFinalPathWithEmptyVerificationType");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
