#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/loaders/behavior3_editor_json_reader.h>
#include <infra/pod_agent/libs/behaviour/trees/base/test/test_canon.h>
#include <infra/pod_agent/libs/pod_agent/object_meta/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/system/fs.h>
#include <util/system/shellcommand.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestVolumeCanon: public ITestBehaviourTreeCanon {
public:
    ITestVolumeCanon(const TString &testName)
        : ITestBehaviourTreeCanon(testName, "VolumeTree", "VolumeTree")
        , LayerIdPrefix_("MyLayerId")
        , LayerDownloadHashPrefix_("MyLayerDownloadHash")
        , StaticResourceDownloadHashPrefix_("MyStaticResourceDownloadHash")
        , VolumeId_("MyVolumeId")
    {
    }

    virtual ~ITestVolumeCanon() {
        RemoveStorageAndLayers();
    }

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["VOLUME_ID"] = VolumeId_;
        replace["VOLUME_PATH"] = PathHolder_->GetVolumePath(VolumeId_);
        replace["VOLUME_STORAGE"] = PathHolder_->GetVolumePersistentStorage(VolumeId_);
        replace["VOLUME_PLACE"] = "";
        replace["LAYER_NAME_LIST"] = PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_) + ';';
        replace["LAYER_DOWNLOAD_HASH_LIST"] = LayerDownloadHashPrefix_ + ";";
        replace["STATIC_RESOURCE_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({"my_static_resource"});
        replace["STATIC_RESOURCE_ORIGINAL_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHashPrefix_, "")});

        replace["TREE_HASH"] = "tree_hash";

        replace["IS_VOLUME_PERSISTENT"] = "true";

        return replace;
    }

    void SetupTest() override {
        PrepareVolume();
    }

    void PrepareLayer(const TString& suffix, const TString& data, const TString& place = "") {
        TStringBuilder commandBuilder;
        commandBuilder << "bash -c \"echo \\\"" << data <<  "\\\" > tmp.txt; "
                       << "mkdir -p " << PathHolder_->GetLayersDirectory(place) + "/" + LayerDownloadHashPrefix_ + suffix << "/downloaded_result; "
                       << "tar -czvf " << PathHolder_->GetFinalLayerPathFromHash(LayerDownloadHashPrefix_ + suffix, place) << " tmp.txt\"";

        TShellCommand shellCommand(commandBuilder);
        shellCommand.Run();
        shellCommand.Wait();
        UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());

        TString layerName = PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + suffix);
        SafePorto_->RemoveLayer(layerName, place);
        SafePorto_->ImportLayer(layerName, PathHolder_->GetFinalLayerPathFromHash(LayerDownloadHashPrefix_ + suffix, place), false, place, "").Success();
        SafePorto_->SetLayerPrivate(":" + LayerDownloadHashPrefix_ + suffix, layerName, place).Success();
        LayerStatusRepository_->AddObject(NObjectMetaTestLib::CreateLayerMetaSimple(LayerIdPrefix_ + suffix, LayerDownloadHashPrefix_ + suffix));
        LayerStatusRepository_->UpdateObjectState(LayerDownloadHashPrefix_ + suffix, API::ELayerState_READY);
    }

    void PrepareStaticResource(ui64 checkPeriodMs = 100, const TString& place = "") {
        TStringBuilder commandBuilder;
        commandBuilder << "bash -c \" "
                       << "mkdir -p " << PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHashPrefix_, place) << ";\"";

        TShellCommand shellCommand(commandBuilder);
        shellCommand.Run();
        shellCommand.Wait();
        UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());

        StaticResourceStatusRepository_->AddObject(
            TStaticResourceMeta(
                "my_static_resource"
                , 0
                , 0
                , StaticResourceDownloadHashPrefix_
                , checkPeriodMs
            )
        );
        StaticResourceStatusRepository_->UpdateObjectState(StaticResourceDownloadHashPrefix_, API::EStaticResourceState_READY);
    }

    void RemoveStaticResource(const TString& place = "") {
        TStringBuilder commandBuilder;
        commandBuilder << "bash -c \" "
                       << "rm -rf " << PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHashPrefix_, place) << ";\"";

        TShellCommand shellCommand(commandBuilder);
        shellCommand.Run();
        shellCommand.Wait();
        UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());

        StaticResourceStatusRepository_->RemoveObject("my_static_resource");
    }

private:
    void RemoveStorageAndLayers() {
        {
            const TString volumeName = PathHolder_->GetVolumePath(VolumeId_);
            auto volumeList = SafePorto_->ListVolumes().Success();
            if (!volumeList.empty()) {
                for (auto volume : volumeList) {
                    if (volume.path() == volumeName) {
                        SafePorto_->UnlinkVolume(volumeName).Success();
                    }
                }
            }
        }
        {
            auto list = SafePorto_->ListStorages("", PersistentStoragePrefix_ + "*").Success();
            for (const auto& storage : list) {
                auto result = SafePorto_->RemoveStorage(storage.name(), "");
                if (!result) {
                    Cerr << ToString(result.Error().Message);
                }
            }
        }
        {
            auto list = SafePorto_->ListLayers("", LayerPrefix_+ "*").Success();
            for (const auto& layer : list) {
                auto result = SafePorto_->RemoveLayer(layer.name(), "");
                if (!result) {
                    Cerr << ToString(result.Error().Message);
                }
            }
        }
    }

    void PrepareVolume() {
        VolumeStatusRepository_->AddObject(NObjectMetaTestLib::CreateVolumeMetaSimple(VolumeId_));
    }

protected:
    const TString LayerIdPrefix_;
    const TString LayerDownloadHashPrefix_;
    const TString StaticResourceDownloadHashPrefix_;
    const TString VolumeId_;
};

Y_UNIT_TEST_SUITE(VolumeTreeTestSuite) {

Y_UNIT_TEST(TestCreateVolume) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };

            TickTree(Tree_, 12, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestCreateVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateVolumeWithSpecificPlace) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareSpecificPlace();
            PrepareLayer("", "some_data", SpecificPlace_);
            PrepareStaticResource(100, SpecificPlace_);

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };

            TickTree(Tree_, 12, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, TConsoleRenderer(false).Render(Tree_));

            {
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetVolumePath(VolumeId_)).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetVolumePath(VolumeId_), volumeList[0].path());

                if (NPortoTestLib::IsInsideSandboxPortoIsolation()) {
                    UNIT_ASSERT_C(TString(volumeList[0].place()).EndsWith(SpecificPlacePath_), volumeList[0].place());
                } else {
                    UNIT_ASSERT_EQUAL_C(volumeList[0].place(), SpecificPlacePath_, volumeList[0].place());
                }
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["VOLUME_PLACE"] = SpecificPlace_;
            specificReplace["STATIC_RESOURCE_ORIGINAL_PATHS"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator({PathHolder_->GetFinalStaticResourcePathFromHash(StaticResourceDownloadHashPrefix_, SpecificPlace_)});


            return specificReplace;
        }
    };

    TTest test("TestCreateVolumeWithSpecificPlace");
    test.DoTest();
}

Y_UNIT_TEST(TestWaitingForLayers) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_));

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_WAITING_FOR_LAYERS;
            };

            TickTree(Tree_, 16, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_WAITING_FOR_LAYERS, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWaitingForLayers");
    test.DoTest();
}

Y_UNIT_TEST(TestWaitingForStaticResources) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_WAITING_FOR_STATIC_RESOURCES;
            };

            TickTree(Tree_, 16, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_WAITING_FOR_STATIC_RESOURCES, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWaitingForStaticResources");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveVolume) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };

            TickTree(Tree_, 12, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(VolumeId_)).Success();

            TickTree(Tree_, 12, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestRemoveVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveVolumeAndLayerAndStaticResource) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };

            auto breakHookWaitingForLayers = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_WAITING_FOR_LAYERS;
            };

            auto breakHookWaitingForStaticResources = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_WAITING_FOR_STATIC_RESOURCES;
            };

            TickTree(Tree_, 16, breakHookReady);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(VolumeId_)).Success();
            SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_)).Success();
            RemoveStaticResource();

            TickTree(Tree_, 16, breakHookWaitingForLayers);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_WAITING_FOR_LAYERS, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareLayer("", "some_data");

            TickTree(Tree_, 16, breakHookWaitingForStaticResources);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_WAITING_FOR_STATIC_RESOURCES, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareStaticResource();
            TickTree(Tree_, 16, breakHookReady);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

        }
    };

    TTest test("TestRemoveVolumeAndLayer");
    test.DoTest();
}

Y_UNIT_TEST(TestMultiVolume) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LAYER_NAME_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(
                {
                     PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_A")
                     , PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_B")
                     , PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_ + "_C")
                }
            );
            specificReplace["LAYER_DOWNLOAD_HASH_LIST"] = TBehavior3EditorJsonReader::EscapeCharactersForMemSequenceGenerator(
                {
                    LayerDownloadHashPrefix_ + "_A"
                    , LayerDownloadHashPrefix_ + "_B"
                    , LayerDownloadHashPrefix_ + "_C"
                }
            );

            return specificReplace;
        }

        void Test() override {
            PrepareLayer("_A", "A");
            PrepareLayer("_B", "B");
            PrepareLayer("_C", "some_data");
            PrepareStaticResource();
            TString idA = LayerDownloadHashPrefix_ + "_A";
            TString idB = LayerDownloadHashPrefix_ + "_B";
            TString idC = LayerDownloadHashPrefix_ + "_C";

            auto breakHookReady = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };

            auto breakHookWaitingForLayers = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_WAITING_FOR_LAYERS;
            };

            TickTree(Tree_, 16, breakHookReady);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            TStringBuilder commandBuilder;
            commandBuilder << "echo A | diff - " << PathHolder_->GetVolumePath(VolumeId_) << "/tmp.txt";
            TShellCommand shellCommand(commandBuilder);
            shellCommand.Run();
            shellCommand.Wait();
            UNIT_ASSERT_EQUAL_C(TShellCommand::ECommandStatus::SHELL_FINISHED, shellCommand.GetStatus(), shellCommand.GetError());

            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(VolumeId_)).Success();
            SafePorto_->RemoveLayer(PathHolder_->GetLayerNameFromHash(idB)).Success();

            TickTree(Tree_, 16, breakHookWaitingForLayers);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_WAITING_FOR_LAYERS, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareLayer("_B", "some_data");

            TickTree(Tree_, 16, breakHookReady);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestMultiVolume");
    test.DoTest();
}

Y_UNIT_TEST(TestBadVolumePrivate) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();

            NFs::MakeDirectoryRecursive(PathHolder_->GetVolumePath(VolumeId_));
            SafePorto_->CreateVolume(
                PathHolder_->GetVolumePath(VolumeId_)
                , PathHolder_->GetVolumePersistentStorage(VolumeId_)
                , ""
                , {}
                , 0
                , "bad_hash"
                , EPortoVolumeBackend::Auto
                , {""}
                , {}
                , false
            ).Success();

            {
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetVolumePath(VolumeId_)).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetVolumePath(VolumeId_), volumeList[0].path());
                UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "bad_hash", volumeList[0].private_value());
            }

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };

            TickTree(Tree_, 24, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            {
                auto volumeList = SafePorto_->ListVolumes(PathHolder_->GetVolumePath(VolumeId_)).Success();
                UNIT_ASSERT_C(volumeList.size() == 1, "there should exactly be one volume, but " << volumeList.size() << " found");
                UNIT_ASSERT_EQUAL_C(volumeList[0].path(), PathHolder_->GetVolumePath(VolumeId_), volumeList[0].path());
                UNIT_ASSERT_EQUAL_C(volumeList[0].private_value(), "tree_hash", volumeList[0].private_value());
            }
        }
    };

    TTest test("TestBadVolumePrivate");
    test.DoTest();
}

Y_UNIT_TEST(TestBadLayerPrivateValue) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();
            TString layerName = PathHolder_->GetLayerNameFromHash(LayerDownloadHashPrefix_);
            SafePorto_->SetLayerPrivate(":wrong_hash", layerName);

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_WAITING_FOR_LAYERS;
            };

            TickTree(Tree_, 24, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_WAITING_FOR_LAYERS, API::EVolumeState_Name(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state()) << Endl << TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestBadLayerPrivateValue");
    test.DoTest();
}

Y_UNIT_TEST(TestWrongStorage) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();

            NFs::MakeDirectoryRecursive(PathHolder_->GetVolumePath(VolumeId_));
            SafePorto_->CreateVolume(PathHolder_->GetVolumePath(VolumeId_), PersistentStoragePrefix_ + "wrong_storage", "", {}, 0, "tree_hash", EPortoVolumeBackend::Auto, {""}, {}, false).Success();

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };

            TickTree(Tree_, 12, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY,
                API::EVolumeState_Name(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state()) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL(SafePorto_->GetStoragePrivate("", PathHolder_->GetVolumePersistentStorage(VolumeId_)).Success(), "tree_hash");
        }
    };

    TTest test("TestNoStorage");
    test.DoTest();
}

Y_UNIT_TEST(TestWrongHashStorage) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();
            TString storage = PathHolder_->GetVolumePersistentStorage(VolumeId_);
            NFs::MakeDirectoryRecursive(PathHolder_->GetVolumePath(VolumeId_));
            SafePorto_->CreateVolume(PathHolder_->GetVolumePath(VolumeId_), storage, "", {}, 0, "wrong_hash", EPortoVolumeBackend::Auto, {""}, {}, false).Success();
            SafePorto_->UnlinkVolume(PathHolder_->GetVolumePath(VolumeId_));

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };

            TickTree(Tree_, 12, breakHook);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY,
                API::EVolumeState_Name(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state()) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL(SafePorto_->GetStoragePrivate("", storage).Success(), "tree_hash");
        }
    };

    TTest test("TestWrongHashStorage");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateVolumeFail) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).fail_counter() > 1;
            };

            TickTree(Tree_, 12, breakHook);
            UNIT_ASSERT_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).fail_counter() > 1, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).failed().message(), "CreateVolume", TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).failed().message(), "Private value too log, max 4096 bytes", TConsoleRenderer(false).Render(Tree_));
        }

         TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["TREE_HASH"] = TString(4096 + 1, 'a'); // too big for volume private
            return specificReplace;
        }
    };

    TTest test("TestCreateVolumeFail");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateVolumeNotReady) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TString& volumePathToBeNotReady)
            : TWrapperPortoClient(client)
            , VolumePathToBeNotReady_(volumePathToBeNotReady)
            , PatchVolumeState_(false)
        {
        }

        TExpected<TVector<TPortoVolume>, TPortoError> ListVolumes(const TString& path, const TPortoContainerName& container) override {
            auto result = TWrapperPortoClient::ListVolumes(path, container);

            if (PatchVolumeState_ && result) {
                // Patch specific volume state
                for (auto& volume : result.Success()) {
                    if (volume.path() == VolumePathToBeNotReady_) {
                        volume.set_state("unready");
                    }
                }
            }

            return result;
        }

        void SetPatchVolumeState(bool patchVolumeState) {
            PatchVolumeState_ = patchVolumeState;
        }

    private:
        const TString VolumePathToBeNotReady_;
        bool PatchVolumeState_;
    };

    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };
            auto breakHookCreating = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_CREATING;
            };

            TickTree(Tree_, 12, breakHookReady);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, TConsoleRenderer(false).Render(Tree_));

            ((TTestPortoClient*)TreePorto_.Get())->SetPatchVolumeState(true);
            // One more tick to trigger state check
            TickTree(Tree_, 1, breakHookCreating);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_CREATING, TConsoleRenderer(false).Render(Tree_));
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetVolumePath(VolumeId_));
        }
    };

    TTest test("TestCreateVolumeNotReady");
    test.DoTest();
}

Y_UNIT_TEST(TestUnlinkVolumeFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TString& badVolume)
            : TWrapperPortoClient(client)
            , BadVolume_(badVolume)
        {
        }

        TExpected<void, TPortoError> UnlinkVolume(
            const TString& path
            , const TPortoContainerName& container
            , const TString& target
            , bool strict
        ) override {
            if (path == BadVolume_) {
                return TPortoError{EPortoError::VolumeNotLinked, TStringBuilder() << "UnlinkVolume failed to unlink '" << path << "'"};
            } else {
                return TWrapperPortoClient::UnlinkVolume(path, container, target, strict);
            }
        }

    private:
        const TString BadVolume_;
    };

    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();

            NFs::MakeDirectoryRecursive(PathHolder_->GetVolumePath(VolumeId_));
            SafePorto_->CreateVolume(
                PathHolder_->GetVolumePath(VolumeId_)
                , PathHolder_->GetVolumePersistentStorage(VolumeId_)
                , ""
                , {}
                , 0
                , "bad_hash"
                , EPortoVolumeBackend::Auto
                , {""}
                , {}
                , false
            ).Success();

            auto breakHook = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_INVALID;
            };

            TickTree(Tree_, 12, breakHook);

            auto status = VolumeStatusRepository_->GetObjectStatus(VolumeId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EVolumeState_INVALID
                , API::EVolumeState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_STRING_CONTAINS(
                status.failed().message()
                , "UnlinkVolume failed"
            );
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, PathHolder_->GetVolumePath(VolumeId_));
        }
    };

    TTest test("TestUnlinkVolumeFail");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveStaticResource) {
    class TTest : public ITestVolumeCanon {
    public:
        TTest(const TString& testName)
            : ITestVolumeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareLayer("", "some_data");
            PrepareStaticResource();

            auto breakHookReady = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_READY;
            };

            auto breakHookWaitingForStaticResources = [this]() {
                return VolumeStatusRepository_->GetObjectStatus(VolumeId_).state() == API::EVolumeState_WAITING_FOR_STATIC_RESOURCES;
            };

            TickTree(Tree_, 16, breakHookReady);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

            RemoveStaticResource();

            TickTree(Tree_, 16, breakHookWaitingForStaticResources);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_WAITING_FOR_STATIC_RESOURCES, Endl << TConsoleRenderer(false).Render(Tree_));

            PrepareStaticResource();
            TickTree(Tree_, 16, breakHookReady);
            UNIT_ASSERT_EQUAL_C(VolumeStatusRepository_->GetObjectStatus(VolumeId_).state(), API::EVolumeState_READY, Endl << TConsoleRenderer(false).Render(Tree_));

        }
    };

    TTest test("TestRemoveVolumeAndLayer");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
