#include <infra/pod_agent/libs/behaviour/bt/nodes/base/private_util.h>
#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/trees/workload/base/test/workload_test_canon.h>
#include <infra/pod_agent/libs/network_client/mock_client.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/support_functions.h>
#include <infra/pod_agent/libs/pod_agent/update_holder/test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <infra/libs/http_service/test_common.h>
#include <infra/libs/service_iface/fake_routers.h>

#include <library/cpp/protobuf/json/proto2json.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

#include <util/stream/file.h>
#include <util/system/fs.h>
#include <util/system/user.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestWorkloadTreeCanon: public ITestWorkloadCanon {
public:
    ITestWorkloadTreeCanon(const TString& testName)
        : ITestWorkloadCanon(testName, "WorkloadTreeAll", "WorkloadTree")
    {
    }

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["WORKLOAD_ID"] = WorkloadId_;
        replace["POD_AGENT_MODE"] = "full_mode";
        replace["LOOKUP_CONTAINER_MASK"] = PathHolder_->GetWorkloadContainerLookupMask(BoxId_, WorkloadId_);
        replace["WORKLOAD_ULIMIT"] = "core: unlimited unlimited; ";
        replace["WORKLOAD_CAPABILITIES_AMBIENT"] = "NET_BIND_SERVICE";

        replace["START_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "start");

        replace["START_CMD"] = "bash -c \"echo start; sleep 1000\"";
        replace["START_ENVIRONMENT"] = "";
        replace["START_SECRET_ENVIRONMENT"] = "";
        replace["START_CWD"] = "";
        replace["START_CPU_GUARANTEE"] = "1c";
        replace["START_CPU_LIMIT"] = "1c";
        replace["START_CPU_POLICY"] = "normal";
        replace["START_CPU_WEIGHT"] = "1.01";
        replace["START_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["START_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["START_ANON_LIMIT"] = ToString(1 << 25);
        replace["START_RECHARGE_ON_PGFAULT"] = "true";
        replace["START_THREAD_LIMIT"] = "1001";
        replace["START_CORE_COMMAND"] = "echo start core dumped";
        replace["START_USER"] = GetUsername();
        replace["START_GROUP"] = "porto";
        replace["START_AGING_TIME"] = ToString(1 << 16);
        replace["START_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = PathHolder_->GetWorkloadLogsFilePathAtBoxRootfs(
            BoxId_
            , WorkloadId_
            , "stdout"
        );
        replace["START_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = PathHolder_->GetWorkloadLogsFilePathAtBoxRootfs(
            BoxId_
            , WorkloadId_
            , "stderr"
        );
        replace["START_STDOUT_LOG_PATH"] = PathHolder_->GetWorkloadLogsFilePathInBox(WorkloadId_, "stdout");
        replace["START_STDERR_LOG_PATH"] = PathHolder_->GetWorkloadLogsFilePathInBox(WorkloadId_, "stderr");
        replace["START_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = ToString(1 << 22);
        replace["START_IO_LIMIT"] = "/tmp r: 20000";
        replace["START_IO_OPS_LIMIT"] = "/tmp r: 20001";
        replace["START_IO_POLICY"] = "normal";
        replace["START_IO_WEIGHT"] = "1.02";

        replace["READINESS_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "readiness");
        replace["READINESS_ENVIRONMENT"] = "";
        replace["READINESS_SECRET_ENVIRONMENT"] = "";
        replace["READINESS_HOOK_TYPE"] = "container";
        replace["READINESS_SUCCESS_THRESHOLD"] = "1";
        replace["READINESS_FAILURE_THRESHOLD"] = "1";
        replace["READINESS_CMD"] = "echo readiness";
        replace["READINESS_CWD"] = "";
        replace["READINESS_CPU_GUARANTEE"] = "1c";
        replace["READINESS_CPU_LIMIT"] = "1c";
        replace["READINESS_CPU_POLICY"] = "normal";
        replace["READINESS_CPU_WEIGHT"] = "1.03";
        replace["READINESS_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["READINESS_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["READINESS_ANON_LIMIT"] = ToString(1 << 25);
        replace["READINESS_RECHARGE_ON_PGFAULT"] = "true";
        replace["READINESS_THREAD_LIMIT"] = "1002";
        replace["READINESS_CORE_COMMAND"] = "echo readiness core dumped";
        replace["READINESS_USER"] = GetUsername();
        replace["READINESS_GROUP"] = "porto";
        replace["READINESS_AGING_TIME"] = ToString(1 << 16);
        replace["READINESS_MIN_RESTART_PERIOD"] = "60000";
        replace["READINESS_MAX_RESTART_PERIOD"] = "60000";
        replace["READINESS_RESTART_PERIOD_BACKOFF"] = "1";
        replace["READINESS_RESTART_PERIOD_SCALE"] = "0";
        replace["READINESS_INITIAL_DELAY"] = "0";
        replace["READINESS_MAX_EXECUTION_TIME"] = "60000";
        replace["READINESS_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["READINESS_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["READINESS_STDOUT_LOG_PATH"] = "";
        replace["READINESS_STDERR_LOG_PATH"] = "";
        replace["READINESS_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["READINESS_IO_LIMIT"] = "/tmp r: 20002";
        replace["READINESS_IO_OPS_LIMIT"] = "/tmp r: 20003";
        replace["READINESS_IO_POLICY"] = "normal";
        replace["READINESS_IO_WEIGHT"] = "1.04";

        replace["LIVENESS_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "liveness");
        replace["LIVENESS_ENVIRONMENT"] = "";
        replace["LIVENESS_SECRET_ENVIRONMENT"] = "";
        replace["LIVENESS_HOOK_TYPE"] = "container";
        replace["LIVENESS_SUCCESS_THRESHOLD"] = "1";
        replace["LIVENESS_FAILURE_THRESHOLD"] = "1";
        replace["LIVENESS_CMD"] = "echo liveness";
        replace["LIVENESS_CWD"] = "";
        replace["LIVENESS_CPU_GUARANTEE"] = "1c";
        replace["LIVENESS_CPU_LIMIT"] = "1c";
        replace["LIVENESS_CPU_POLICY"] = "normal";
        replace["LIVENESS_CPU_WEIGHT"] = "1.05";
        replace["LIVENESS_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["LIVENESS_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["LIVENESS_ANON_LIMIT"] = ToString(1 << 25);
        replace["LIVENESS_RECHARGE_ON_PGFAULT"] = "true";
        replace["LIVENESS_THREAD_LIMIT"] = "1003";
        replace["LIVENESS_CORE_COMMAND"] = "echo liveness core dumped";
        replace["LIVENESS_USER"] = GetUsername();
        replace["LIVENESS_GROUP"] = "porto";
        replace["LIVENESS_AGING_TIME"] = ToString(1 << 16);
        replace["LIVENESS_MIN_RESTART_PERIOD"] = "60000";
        replace["LIVENESS_MAX_RESTART_PERIOD"] = "60000";
        replace["LIVENESS_RESTART_PERIOD_BACKOFF"] = "1";
        replace["LIVENESS_RESTART_PERIOD_SCALE"] = "0";
        replace["LIVENESS_INITIAL_DELAY"] = "0";
        replace["LIVENESS_MAX_EXECUTION_TIME"] = "60000";
        replace["LIVENESS_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["LIVENESS_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["LIVENESS_STDOUT_LOG_PATH"] = "";
        replace["LIVENESS_STDERR_LOG_PATH"] = "";
        replace["LIVENESS_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["LIVENESS_IO_LIMIT"] = "/tmp r: 20004";
        replace["LIVENESS_IO_OPS_LIMIT"] = "/tmp r: 20005";
        replace["LIVENESS_IO_POLICY"] = "normal";
        replace["LIVENESS_IO_WEIGHT"] = "1.06";

        replace["STOP_HOOK_TYPE"] = "container";
        replace["STOP_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "stop");
        replace["STOP_ENVIRONMENT"] = "";
        replace["STOP_SECRET_ENVIRONMENT"] = "";
        replace["STOP_CMD"] = "bash -c \"echo stop; sleep 1000\"";
        replace["STOP_CWD"] = "";
        replace["STOP_CPU_GUARANTEE"] = "1c";
        replace["STOP_CPU_LIMIT"] = "1c";
        replace["STOP_CPU_POLICY"] = "normal";
        replace["STOP_CPU_WEIGHT"] = "1.07";
        replace["STOP_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["STOP_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["STOP_ANON_LIMIT"] = ToString(1 << 25);
        replace["STOP_RECHARGE_ON_PGFAULT"] = "true";
        replace["STOP_THREAD_LIMIT"] = "1004";
        replace["STOP_CORE_COMMAND"] = "echo stop core dumped";
        replace["STOP_USER"] = GetUsername();
        replace["STOP_GROUP"] = "porto";
        replace["STOP_AGING_TIME"] = ToString(1 << 16);
        replace["STOP_MIN_RESTART_PERIOD"] = "60000";
        replace["STOP_MAX_RESTART_PERIOD"] = "60000";
        replace["STOP_RESTART_PERIOD_BACKOFF"] = "1";
        replace["STOP_RESTART_PERIOD_SCALE"] = "0";
        replace["STOP_MAX_EXECUTION_TIME"] = "60000";
        replace["STOP_MAX_TRIES"] = "1000";
        replace["STOP_INITIAL_DELAY"] = "0";
        replace["STOP_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["STOP_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["STOP_STDOUT_LOG_PATH"] = "";
        replace["STOP_STDERR_LOG_PATH"] = "";
        replace["STOP_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["STOP_IO_LIMIT"] = "/tmp r: 20006";
        replace["STOP_IO_OPS_LIMIT"] = "/tmp r: 20007";
        replace["STOP_IO_POLICY"] = "normal";
        replace["STOP_IO_WEIGHT"] = "1.08";

        replace["DESTROY_HOOK_TYPE"] = "container";
        replace["DESTROY_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "destroy");
        replace["DESTROY_ENVIRONMENT"] = "";
        replace["DESTROY_SECRET_ENVIRONMENT"] = "";
        replace["DESTROY_CMD"] = "bash -c \"echo destroy; sleep 1000\"";
        replace["DESTROY_CWD"] = "";
        replace["DESTROY_CPU_GUARANTEE"] = "1c";
        replace["DESTROY_CPU_LIMIT"] = "1c";
        replace["DESTROY_CPU_POLICY"] = "normal";
        replace["DESTROY_CPU_WEIGHT"] = "1.09";
        replace["DESTROY_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["DESTROY_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["DESTROY_ANON_LIMIT"] = ToString(1 << 25);
        replace["DESTROY_RECHARGE_ON_PGFAULT"] = "true";
        replace["DESTROY_THREAD_LIMIT"] = "1005";
        replace["DESTROY_CORE_COMMAND"] = "echo destroy core dumped";
        replace["DESTROY_USER"] = GetUsername();
        replace["DESTROY_GROUP"] = "porto";
        replace["DESTROY_AGING_TIME"] = ToString(1 << 16);
        replace["DESTROY_MIN_RESTART_PERIOD"] = "60000";
        replace["DESTROY_MAX_RESTART_PERIOD"] = "60000";
        replace["DESTROY_RESTART_PERIOD_BACKOFF"] = "1";
        replace["DESTROY_RESTART_PERIOD_SCALE"] = "0";
        replace["DESTROY_MAX_EXECUTION_TIME"] = "60000";
        replace["DESTROY_MAX_TRIES"] = "1000";
        replace["DESTROY_INITIAL_DELAY"] = "0";
        replace["DESTROY_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["DESTROY_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["DESTROY_STDOUT_LOG_PATH"] = "";
        replace["DESTROY_STDERR_LOG_PATH"] = "";
        replace["DESTROY_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["DESTROY_IO_LIMIT"] = "/tmp r: 20008";
        replace["DESTROY_IO_OPS_LIMIT"] = "/tmp r: 20009";
        replace["DESTROY_IO_POLICY"] = "normal";
        replace["DESTROY_IO_WEIGHT"] = "1.10";

        replace["INIT_NUM_LIST"] = "";
        replace["INIT_CONTAINER_LIST"] = "";
        replace["INIT_CONTAINER_PREVIOUS_LIST"] = "";
        replace["INIT_ENVIRONMENT_LIST"] = "";
        replace["INIT_SECRET_ENVIRONMENT_LIST"] = "";
        replace["INIT_CMD_LIST"] = "";
        replace["INIT_CWD_LIST"] = "";

        replace["INIT_CPU_GUARANTEE_LIST"] = "";
        replace["INIT_CPU_LIMIT_LIST"] = "";
        replace["INIT_CPU_POLICY_LIST"] = "";
        replace["INIT_CPU_WEIGHT_LIST"] = "";
        replace["INIT_MEMORY_GUARANTEE_LIST"] = "";
        replace["INIT_MEMORY_LIMIT_LIST"] = "";
        replace["INIT_ANON_LIMIT_LIST"] = "";
        replace["INIT_RECHARGE_ON_PGFAULT_LIST"] = "";
        replace["INIT_THREAD_LIMIT_LIST"] = "";
        replace["INIT_CORE_COMMAND_LIST"] = "";
        replace["INIT_USER_LIST"] = "";
        replace["INIT_GROUP_LIST"] = "";
        replace["INIT_AGING_TIME_LIST"] = "";

        replace["INIT_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = "";
        replace["INIT_STDERR_LOG_FILE_FULL_PATH_TO_CREATE_LIST"] = "";
        replace["INIT_STDOUT_LOG_PATH_LIST"] = "";
        replace["INIT_STDERR_LOG_PATH_LIST"] = "";
        replace["INIT_STDOUT_AND_STDERR_FILE_SIZE_LIMIT_LIST"] = "";

        replace["INIT_IO_LIMIT_LIST"] = "";
        replace["INIT_IO_OPS_LIMIT_LIST"] = "";
        replace["INIT_IO_POLICY_LIST"] = "";
        replace["INIT_IO_WEIGHT_LIST"] = "";

        replace["INIT_INITIAL_DELAY_LIST"] = "";
        replace["INIT_RESTART_PERIOD_SCALE_LIST"] = "";
        replace["INIT_RESTART_PERIOD_BACKOFF_LIST"] = "";
        replace["INIT_MAX_RESTART_PERIOD_LIST"] = "";
        replace["INIT_MIN_RESTART_PERIOD_LIST"] = "";
        replace["INIT_MAX_EXECUTION_TIME_LIST"] = "";

        replace["BOX_CONTAINER"] = PathHolder_->GetBoxContainer(BoxId_);
        replace["BOX_TREE_HASH"] = "box_tree_hash";

        replace["TREE_HASH"] = "tree_hash";

        replace["HTTP_READINESS_PATH"] = "/mock_path";
        replace["HTTP_READINESS_PORT"] = "";
        replace["HTTP_READINESS_RESPONSE"] = "Data";
        replace["HTTP_READINESS_ANY_RESPONSE"] = "false";
        replace["HTTP_READINESS_MAX_EXECUTION_TIME"] = "500";
        replace["HTTP_READINESS_INITIAL_DELAY"] = "0";
        replace["HTTP_READINESS_RESTART_PERIOD_SCALE"] = "1";
        replace["HTTP_READINESS_RESTART_PERIOD_BACKOFF"] = "60000";
        replace["HTTP_READINESS_MAX_RESTART_PERIOD"] = "60000";
        replace["HTTP_READINESS_MIN_RESTART_PERIOD"] = "60000";

        replace["HTTP_LIVENESS_PATH"] = "/mock_path";
        replace["HTTP_LIVENESS_PORT"] = "";
        replace["HTTP_LIVENESS_RESPONSE"] = "Data";
        replace["HTTP_LIVENESS_ANY_RESPONSE"] = "false";
        replace["HTTP_LIVENESS_MAX_EXECUTION_TIME"] = "500";
        replace["HTTP_LIVENESS_INITIAL_DELAY"] = "0";
        replace["HTTP_LIVENESS_RESTART_PERIOD_SCALE"] = "1";
        replace["HTTP_LIVENESS_RESTART_PERIOD_BACKOFF"] = "60000";
        replace["HTTP_LIVENESS_MAX_RESTART_PERIOD"] = "60000";
        replace["HTTP_LIVENESS_MIN_RESTART_PERIOD"] = "60000";

        replace["TCP_READINESS_PORT"] = "";
        replace["TCP_READINESS_MAX_EXECUTION_TIME"] = "";
        replace["TCP_READINESS_INITIAL_DELAY"] = "";
        replace["TCP_READINESS_RESTART_PERIOD_SCALE"] = "";
        replace["TCP_READINESS_RESTART_PERIOD_BACKOFF"] = "";
        replace["TCP_READINESS_MAX_RESTART_PERIOD"] = "";
        replace["TCP_READINESS_MIN_RESTART_PERIOD"] = "";

        replace["TCP_LIVENESS_PORT"] = "";
        replace["TCP_LIVENESS_MAX_EXECUTION_TIME"] = "";
        replace["TCP_LIVENESS_INITIAL_DELAY"] = "";
        replace["TCP_LIVENESS_RESTART_PERIOD_SCALE"] = "";
        replace["TCP_LIVENESS_RESTART_PERIOD_BACKOFF"] = "";
        replace["TCP_LIVENESS_MAX_RESTART_PERIOD"] = "";
        replace["TCP_LIVENESS_MIN_RESTART_PERIOD"] = "";

        replace["HTTP_STOP_PATH"] = "/mock_path";
        replace["HTTP_STOP_PORT"] = "";
        replace["HTTP_STOP_RESPONSE"] = "";
        replace["HTTP_STOP_ANY_RESPONSE"] = "true";
        replace["HTTP_STOP_MAX_EXECUTION_TIME"] = "";
        replace["HTTP_STOP_INITIAL_DELAY"] = "";
        replace["HTTP_STOP_RESTART_PERIOD_SCALE"] = "";
        replace["HTTP_STOP_RESTART_PERIOD_BACKOFF"] = "";
        replace["HTTP_STOP_MAX_RESTART_PERIOD"] = "";
        replace["HTTP_STOP_MIN_RESTART_PERIOD"] = "";

        replace["HTTP_DESTROY_PATH"] = "/mock_path";
        replace["HTTP_DESTROY_PORT"] = "";
        replace["HTTP_DESTROY_RESPONSE"] = "";
        replace["HTTP_DESTROY_ANY_RESPONSE"] = "true";
        replace["HTTP_DESTROY_MAX_EXECUTION_TIME"] = "";
        replace["HTTP_DESTROY_INITIAL_DELAY"] = "";
        replace["HTTP_DESTROY_RESTART_PERIOD_SCALE"] = "";
        replace["HTTP_DESTROY_RESTART_PERIOD_BACKOFF"] = "";
        replace["HTTP_DESTROY_MAX_RESTART_PERIOD"] = "";
        replace["HTTP_DESTROY_MIN_RESTART_PERIOD"] = "";

        replace["UNIX_SIGNAL_STOP_SIGNAL"] = ToString(SIGTERM);
        replace["UNIX_SIGNAL_STOP_INITIAL_DELAY"] = "";
        replace["UNIX_SIGNAL_STOP_RESTART_PERIOD_SCALE"] = "";
        replace["UNIX_SIGNAL_STOP_RESTART_PERIOD_BACKOFF"] = "";
        replace["UNIX_SIGNAL_STOP_MAX_RESTART_PERIOD"] = "";
        replace["UNIX_SIGNAL_STOP_MIN_RESTART_PERIOD"] = "";

        return replace;
    }
};

Y_UNIT_TEST_SUITE(WorkloadTreeTestSuite) {

Y_UNIT_TEST(TestWorkloadWaitingForBoxReady) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };
            auto breakHookWaiting = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_WAITING_FOR_BOX;
            };

            TickTree(Tree_, 24, breakHookActive);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, status.state(), TConsoleRenderer(false).Render(Tree_));
            // check whether all conditions for workload are satisfied
            UNIT_ASSERT_EQUAL_C(status.start().current().state(), API::EContainerState_RUNNING, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.readiness_status().container_status().zero_return_code_counter() > 0, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.readiness_status().container_status().positive_return_code_counter() == 0, TConsoleRenderer(false).Render(Tree_));

            auto result = SafePorto_->SetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "box_tree_hash"}));
            UNIT_ASSERT_C((bool)result, result.Error().Message);

            TickTree(Tree_, 24, breakHookWaiting);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_WAITING_FOR_BOX, status.state(), TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWorkloadWaitingForBoxReady");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWaitingForBox) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_WAITING_FOR_BOX;
            };

            TickTree(Tree_, 24, breakHook);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_WAITING_FOR_BOX, WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWorkloadWaitingForBox");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadNotWaitingForBoxWhenBoxAgentMode) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            TickTree(Tree_, 32, breakHookActive);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, status.state(), TConsoleRenderer(false).Render(Tree_));
            // check whether all conditions for workload are satisfied
            UNIT_ASSERT_EQUAL_C(status.start().current().state(), API::EContainerState_RUNNING, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.readiness_status().container_status().zero_return_code_counter() > 0, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.readiness_status().container_status().positive_return_code_counter() == 0, TConsoleRenderer(false).Render(Tree_));

            // Start StdOutPath test
            TString startStdOutPath = SafePorto_->GetProperty(GetFullWorkloadContainerName("start"), EPortoContainerProperty::StdOutPath).Success();
            UNIT_ASSERT_EQUAL_C(startStdOutPath, NFs::CurrentWorkingDirectory() + "/MyWorkloadId_stdout.portolog", startStdOutPath);

            // Start StdErrPath test
            TString startStdErrPath = SafePorto_->GetProperty(GetFullWorkloadContainerName("start"), EPortoContainerProperty::StdErrPath).Success();
            UNIT_ASSERT_EQUAL_C(startStdErrPath, NFs::CurrentWorkingDirectory() + "/MyWorkloadId_stderr.portolog", startStdErrPath);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["POD_AGENT_MODE"] = "box_mode";

            specificReplace["START_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = NFs::CurrentWorkingDirectory() + PathHolder_->GetWorkloadLogsFilePathInBox(WorkloadId_, "stdout");
            specificReplace["START_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = NFs::CurrentWorkingDirectory() + PathHolder_->GetWorkloadLogsFilePathInBox(WorkloadId_, "stderr");
            specificReplace["START_STDOUT_LOG_PATH"] = NFs::CurrentWorkingDirectory() + PathHolder_->GetWorkloadLogsFilePathInBox(WorkloadId_, "stdout");
            specificReplace["START_STDERR_LOG_PATH"] = NFs::CurrentWorkingDirectory() + PathHolder_->GetWorkloadLogsFilePathInBox(WorkloadId_, "stderr");

            return specificReplace;
        }
    };

    TTest test("TestWorkloadNotWaitingForBoxWhenBoxAgentMode");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadContainersProperties) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            const TString path = PathHolder_->GetBoxRootfsPath(BoxId_);
            for (auto prefix : Prefixes_) {
                NFs::MakeDirectoryRecursive(path + "/" + prefix + "_dir");
            }

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };
            auto breakHookDeactivating = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_DEACTIVATING;
            };

            TickTree(Tree_, 32, breakHookActive);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));

            for (const TString container : {"start", "readiness", "liveness"}) {
                TestContainerProperties(container);
            }

            // Start StdOutPath test
            TString startStdOutPath = SafePorto_->GetProperty(GetFullWorkloadContainerName("start"), EPortoContainerProperty::StdOutPath).Success();
            UNIT_ASSERT_EQUAL_C(startStdOutPath, "/MyWorkloadId_stdout.portolog", "Bad start container stdout_path");

            // Start StdErrPath test
            TString startStdErrPath = SafePorto_->GetProperty(GetFullWorkloadContainerName("start"), EPortoContainerProperty::StdErrPath).Success();
            UNIT_ASSERT_EQUAL_C(startStdErrPath, "/MyWorkloadId_stderr.portolog", "Bad start container stderr_path");

            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);
            TickTree(Tree_, 32, breakHookDeactivating);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_DEACTIVATING, TConsoleRenderer(false).Render(Tree_));

            TestContainerProperties("stop");
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            const TString path = PathHolder_->GetBoxRootfsPath(BoxId_);

            for (auto prefix : Prefixes_) {
                specificReplace[to_upper(prefix) + "_CWD"] = "/" + prefix + "_dir";
                specificReplace[to_upper(prefix) + "_ENVIRONMENT"] = SpecificEnvironmentForContainer_.at(prefix);
                specificReplace[to_upper(prefix) + "_SECRET_ENVIRONMENT"] = SpecificSecretEnvironmentForContainer_.at(prefix);
            }

            return specificReplace;
        }

    private:
        void TestContainerProperties(const TString& container) {
            // Env test
            TString containerEnvironment = SafePorto_->GetProperty(GetFullWorkloadContainerName(container), EPortoContainerProperty::Env).Success();
            UNIT_ASSERT_EQUAL_C(containerEnvironment, SpecificEnvironmentForContainer_.at(container), "Bad " << container << " container environment");

            // Secret env test
            TString containerSecretEnvironment = SafePorto_->GetProperty(GetFullWorkloadContainerName(container), EPortoContainerProperty::EnvSecret).Success();
            // secrets have different salt and md5, let's check only substring
            for (const auto& secretEnv: SpecificSecretEnvironmentHiddenForContainer_.at(container)) {
                UNIT_ASSERT_STRING_CONTAINS_C(containerSecretEnvironment, secretEnv, "Bad " << container << " container secret environment");
            }

            // Isolate test
            TString containerIsolate = SafePorto_->GetProperty(GetFullWorkloadContainerName(container), EPortoContainerProperty::Isolate).Success();
            UNIT_ASSERT_EQUAL_C(containerIsolate, "false", "Bad " << container << " container isolate");

            // RechargeOnPgfault test
            TString containerRechPgfault = SafePorto_->GetProperty(GetFullWorkloadContainerName(container), EPortoContainerProperty::RechPgfault).Success();
            UNIT_ASSERT_EQUAL_C(containerRechPgfault, "true", "Bad " << container << " container recharge_on_pgfault");

            // Core command test
            TString containerCoreCommand = SafePorto_->GetProperty(GetFullWorkloadContainerName(container), EPortoContainerProperty::CoreCommand).Success();
            UNIT_ASSERT_EQUAL_C(containerCoreCommand, "echo " + container + " core dumped", "Bad " << container << " container core_command");

            // User test
            TString containerUser = SafePorto_->GetProperty(GetFullWorkloadContainerName(container), EPortoContainerProperty::User).Success();
            UNIT_ASSERT_EQUAL_C(containerUser, GetUsername(), "Bad " << container << " container user");

            // Group test
            TString containerGroup = SafePorto_->GetProperty(GetFullWorkloadContainerName(container), EPortoContainerProperty::Group).Success();
            UNIT_ASSERT_EQUAL_C(containerGroup, "porto", "Bad " << container << " container user");

            // CWD test
            TString containerCWD = SafePorto_->GetProperty(GetFullWorkloadContainerName(container), EPortoContainerProperty::Cwd).Success();
            UNIT_ASSERT_EQUAL_C(containerCWD, "/" + container + "_dir", "Bad " << container << " container cwd");

            // Hostname test
            TString containerHostname = SafePorto_->GetProperty(GetFullWorkloadContainerName(container), EPortoContainerProperty::Hostname).Success();
            UNIT_ASSERT_EQUAL_C(containerHostname, "", "Bad " << container << " container hostname");
        }

    private:
        const TVector<TString> Prefixes_ = {
            "start"
            , "readiness"
            , "liveness"
            , "stop"
        };
        const TString SpecificEnvironment_ = "first_name=first_value;second_name=second_value";
        const TMap<TString, TString> SpecificEnvironmentForContainer_ = {
            {"start", SpecificEnvironment_ + ";container=start"}
            , {"readiness", SpecificEnvironment_ + ";container=readiness"}
            , {"liveness", SpecificEnvironment_ + ";container=liveness"}
            , {"stop", SpecificEnvironment_ + ";container=stop"}
        };
        const TString SpecificSecretEnvironment_ = "secret_base=value_base";
        const TMap<TString, TString> SpecificSecretEnvironmentForContainer_ = {
            {"start", SpecificSecretEnvironment_ + ";secret_start=value_start"}
            , {"readiness", SpecificSecretEnvironment_ + ";secret_readiness=value_readiness"}
            , {"liveness", SpecificSecretEnvironment_ + ";secret_liveness=value_liveness"}
            , {"stop", SpecificSecretEnvironment_ + ";secret_stop=value_stop"}
        };

        const TString SpecificSecretEnvironmentHidden_ = "secret_base=<secret";
        const TMap<TString, TVector<TString>> SpecificSecretEnvironmentHiddenForContainer_ = {
            {"start", {SpecificSecretEnvironmentHidden_, "secret_start=<secret"}}
            , {"readiness", {SpecificSecretEnvironmentHidden_, "secret_readiness=<secret"}}
            , {"liveness", {SpecificSecretEnvironmentHidden_, "secret_liveness=<secret"}}
            , {"stop", {SpecificSecretEnvironmentHidden_, "secret_stop=<secret"}}
        };
    };

    TTest test("TestWorkloadContainersProperties");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadUnactiveRemovedBadBox) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };
            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookActive);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));

            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);
            auto result = SafePorto_->SetProperty(PathHolder_->GetBoxContainer(BoxId_), EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "box_tree_hash"}));
            UNIT_ASSERT_C((bool)result, result.Error().Message);

            TickTree(Tree_, 32, breakHookRemoved);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_REMOVED, status.state(), TConsoleRenderer(false).Render(Tree_));

        }
    };

    TTest test("TestWorkloadUnactiveRemovedBadBox");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedAndRestart) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive1 = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().zero_return_code_counter() == 1
                ;
            };
            auto breakHookActive2 = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().zero_return_code_counter() == 2
                ;
            };
            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookActive1);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().zero_return_code_counter(), 1, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().zero_return_code_counter(), 1, TConsoleRenderer(false).Render(Tree_));
            TInstant readinessFirstTs = NSupport::ToInstant(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().last().start_time());
            TInstant livenessFirstTs = NSupport::ToInstant(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().last().start_time());

            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            TickTree(Tree_, 32, breakHookRemoved);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_KILLED_EXTERNALLY, TConsoleRenderer(false).Render(Tree_));

            {
                auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
                UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "stop container in dead state while start container in running state", stopStatus.last().fail_reason());
                UNIT_ASSERT_EQUAL_C(stopStatus.last().stdout_tail(), "stop\n", stopStatus.last().stdout_tail());
                UNIT_ASSERT_EQUAL_C(stopStatus.last().stderr_tail(), "stop\n", stopStatus.last().stderr_tail());
            }

            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_ACTIVE);

            TickTree(Tree_, 32, breakHookActive2);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().zero_return_code_counter(), 2, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().zero_return_code_counter(), 2, TConsoleRenderer(false).Render(Tree_));
            TInstant readinessSecondTs = NSupport::ToInstant(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().last().start_time());
            TInstant livenessSecondTs = NSupport::ToInstant(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().last().start_time());
            UNIT_ASSERT_GE_C(readinessSecondTs - readinessFirstTs, TDuration::MilliSeconds(RESTART_PERIOD_MS), readinessSecondTs << " " << readinessFirstTs);
            UNIT_ASSERT_GE_C(livenessSecondTs - livenessFirstTs, TDuration::MilliSeconds(RESTART_PERIOD_MS), livenessSecondTs << " " << livenessFirstTs);

            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().current().state(), API::EContainerState_RUNNING, TConsoleRenderer(false).Render(Tree_));

            {
                // Check that status cleared after restart
                auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();

                UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "", stopStatus.last().fail_reason());
                UNIT_ASSERT_EQUAL_C(stopStatus.last().stdout_tail(), "", stopStatus.last().stdout_tail());
                UNIT_ASSERT_EQUAL_C(stopStatus.last().stderr_tail(), "", stopStatus.last().stderr_tail());

                UNIT_ASSERT_EQUAL_C(stopStatus.killed_externally_counter(), 0, stopStatus.killed_externally_counter());
                UNIT_ASSERT_EQUAL_C(stopStatus.oom_counter(), 0, stopStatus.oom_counter());
                UNIT_ASSERT_EQUAL_C(stopStatus.positive_return_code_counter(), 0, stopStatus.positive_return_code_counter());
                UNIT_ASSERT_EQUAL_C(stopStatus.zero_return_code_counter(), 0, stopStatus.zero_return_code_counter());
                UNIT_ASSERT_EQUAL_C(stopStatus.timeout_counter(), 0, stopStatus.timeout_counter());
                UNIT_ASSERT_EQUAL_C(stopStatus.system_failure_counter(), 0, stopStatus.system_failure_counter());
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LIVENESS_MIN_RESTART_PERIOD"] = ToString(RESTART_PERIOD_MS);
            specificReplace["LIVENESS_MAX_RESTART_PERIOD"] = ToString(RESTART_PERIOD_MS);
            specificReplace["WORKLOAD_LIEVENESS_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["LIVENESS_RESTART_PERIOD_SCALE"] = "0";

            specificReplace["READINESS_MIN_RESTART_PERIOD"] = ToString(RESTART_PERIOD_MS);
            specificReplace["READINESS_MAX_RESTART_PERIOD"] = ToString(RESTART_PERIOD_MS);
            specificReplace["READINESS_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["READINESS_RESTART_PERIOD_SCALE"] = "0";

            specificReplace["STOP_MIN_RESTART_PERIOD"] = "1";
            specificReplace["STOP_MAX_RESTART_PERIOD"] = "1";
            specificReplace["STOP_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["STOP_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["STOP_MAX_TRIES"] = "1";
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop\"";
            return specificReplace;
        }

    private:
        const size_t RESTART_PERIOD_MS = 2000;
    };

    TTest test("TestWorkloadRemovedAndRestart");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadFinallyRemoved) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
                : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookActive);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));

            UpdateHolder_->SetWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetSimple(WorkloadId_));
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);

            TickTree(Tree_, 64, breakHookRemoved);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_KILLED_EXTERNALLY, TConsoleRenderer(false).Render(Tree_));

            {
                auto destroyStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status();
                auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();

                UNIT_ASSERT_EQUAL_C(destroyStatus.last().fail_reason(), "", destroyStatus.last().fail_reason());
                UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "stop container in dead state while start container in running state", stopStatus.last().fail_reason());

                UNIT_ASSERT_EQUAL_C(destroyStatus.last().stdout_tail(), "destroy\n", destroyStatus.last().stdout_tail());
                UNIT_ASSERT_EQUAL_C(destroyStatus.last().stderr_tail(), "destroy\n", destroyStatus.last().stderr_tail());

                UNIT_ASSERT_EQUAL_C(stopStatus.last().stdout_tail(), "stop\n", stopStatus.last().stdout_tail());
                UNIT_ASSERT_EQUAL_C(stopStatus.last().stderr_tail(), "stop\n", stopStatus.last().stderr_tail());

                TFile stopDestroyOutputFile(PathHolder_->GetBoxRootfsPath(BoxId_) + "/tmp_file", OpenExisting | RdOnly);
                TString stopDestroyOutput  = TUnbufferedFileInput(stopDestroyOutputFile).ReadAll();

                size_t stopLen = std::strlen("_stop");
                size_t destroyLen = std::strlen("_destroy");
                UNIT_ASSERT_C(stopDestroyOutput.StartsWith("_stop_destroy"), stopDestroyOutput);
                UNIT_ASSERT_C((stopDestroyOutput.size() - stopLen) % destroyLen == 0, stopDestroyOutput);
                for (size_t i = stopLen; i < stopDestroyOutput.size(); i += destroyLen) {
                    UNIT_ASSERT_EQUAL_C(stopDestroyOutput.substr(i, destroyLen), "_destroy", stopDestroyOutput);
                }
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_MIN_RESTART_PERIOD"] = "1";
            specificReplace["DESTROY_MAX_RESTART_PERIOD"] = "1";
            specificReplace["DESTROY_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["DESTROY_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["DESTROY_MAX_TRIES"] = "10";
            specificReplace["DESTROY_CMD"] = "bash -c \"echo destroy; echo -n _destroy >> /tmp_file; >&2 echo destroy\"";

            specificReplace["STOP_MIN_RESTART_PERIOD"] = "1";
            specificReplace["STOP_MAX_RESTART_PERIOD"] = "1";
            specificReplace["STOP_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["STOP_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["STOP_MAX_TRIES"] = "1";
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; echo -n _stop >> /tmp_file; >&2 echo stop\"";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadFinallyRemoved");
    test.DoTest();
}


Y_UNIT_TEST(TestWorkloadWithoutReadiness) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            TickTree(Tree_, 32, breakHookActive);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(
                !status.readiness_status().has_container_status()
                && !status.readiness_status().has_http_get_status()
                && !status.readiness_status().has_tcp_check_status()
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["READINESS_CMD"] = "";
            specificReplace["READINESS_CPU_GUARANTEE"] = "";
            specificReplace["READINESS_CPU_LIMIT"] = "";
            specificReplace["READINESS_MEMORY_GUARANTEE"] = "";
            specificReplace["READINESS_MEMORY_LIMIT"] = "";
            specificReplace["READINESS_ANON_LIMIT"] = "";
            specificReplace["READINESS_MIN_RESTART_PERIOD"] = "";
            specificReplace["READINESS_MAX_RESTART_PERIOD"] = "";
            specificReplace["READINESS_RESTART_PERIOD_BACKOFF"] = "";
            specificReplace["READINESS_RESTART_PERIOD_SCALE"] = "";
            specificReplace["READINESS_INITIAL_DELAY"] = "";
            specificReplace["READINESS_HOOK_TYPE"] = "no_hook";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadWithoutReadiness");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWithoutLiveness) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            TickTree(Tree_, 32, breakHookActive);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(
                !status.liveness_status().has_container_status()
                && !status.liveness_status().has_http_get_status()
                && !status.liveness_status().has_tcp_check_status()
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LIVENESS_CMD"] = "";
            specificReplace["LIVENESS_CPU_GUARANTEE"] = "";
            specificReplace["LIVENESS_CPU_LIMIT"] = "";
            specificReplace["LIVENESS_MEMORY_GUARANTEE"] = "";
            specificReplace["LIVENESS_MEMORY_LIMIT"] = "";
            specificReplace["LIVENESS_ANON_LIMIT"] = "";
            specificReplace["LIVENESS_MIN_RESTART_PERIOD"] = "";
            specificReplace["LIVENESS_MAX_RESTART_PERIOD"] = "";
            specificReplace["LIVENESS_RESTART_PERIOD_BACKOFF"] = "";
            specificReplace["LIVENESS_RESTART_PERIOD_SCALE"] = "";
            specificReplace["LIVENESS_INITIAL_DELAY"] = "";
            specificReplace["LIVENESS_HOOK_TYPE"] = "no_hook";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadWithoutLiveness");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWithoutStopPolicy) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };
            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookActive);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);
            TickTree(Tree_, 32, breakHookRemoved);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_KILLED_EXTERNALLY, TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_CMD"] = "";
            specificReplace["STOP_CPU_GUARANTEE"] = "";
            specificReplace["STOP_CPU_LIMIT"] = "";
            specificReplace["STOP_MEMORY_GUARANTEE"] = "";
            specificReplace["STOP_MEMORY_LIMIT"] = "";
            specificReplace["STOP_ANON_LIMIT"] = "";
            specificReplace["STOP_MIN_RESTART_PERIOD"] = "";
            specificReplace["STOP_MAX_RESTART_PERIOD"] = "";
            specificReplace["STOP_RESTART_PERIOD_BACKOFF"] = "";
            specificReplace["STOP_RESTART_PERIOD_SCALE"] = "";
            specificReplace["STOP_MAX_TRIES"] = "";
            specificReplace["STOP_HOOK_TYPE"] = "no_hook";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadWithoutStopPolicy");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedNewTargetTree) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            UpdateHolder_->SetWorkloadTarget(NObjectTargetTestLib::CreateWorkloadTargetSimple(WorkloadId_));

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookRemoved);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWorkloadRemovedNewTargetTree");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWithHttpReadiness) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART
                ;
            };

            auto breakHookCheckFailed = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().http_get_status().time_limit().consecutive_failures_counter() > 1
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART
                ;
            };

            TickTree(Tree_, 32, breakHookCheckFailed);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto httpStatus = status.readiness_status().http_get_status();
            UNIT_ASSERT_C(httpStatus.time_limit().consecutive_failures_counter() > 1, NProtobufJson::Proto2Json(httpStatus));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_READINESS_FAILED, TConsoleRenderer(false).Render(Tree_));
            TInstant startTs = NSupport::ToInstant(status.start().current().start_time());
            TInstant firstHookTs = NSupport::ToInstant(httpStatus.last().start_time());
            UNIT_ASSERT_GE_C(firstHookTs - startTs, TDuration::MilliSeconds(INITIAL_DELAY_MS), firstHookTs << " " << startTs);
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WAITING_RESTART, httpStatus.current().state()
                , NProtobufJson::Proto2Json(httpStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_FAILURE, httpStatus.last().state()
                , NProtobufJson::Proto2Json(httpStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            THttpService httpService = CreateAndStartHttpService(Config_, MakeSimpleShared<TFakeSuccessRouter>());

            TickTree(Tree_, 32, breakHookActive);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            httpStatus = status.readiness_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(1, httpStatus.time_limit().consecutive_successes_counter(), NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WAITING_RESTART, httpStatus.current().state()
                , NProtobufJson::Proto2Json(httpStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_SUCCESS, httpStatus.last().state()
                , NProtobufJson::Proto2Json(httpStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["READINESS_HOOK_TYPE"] = "http";
            specificReplace["LIVENESS_HOOK_TYPE"] = "no_hook";
            specificReplace["HTTP_READINESS_PORT"] = ToString(Config_.GetPort());
            specificReplace["HTTP_READINESS_INITIAL_DELAY"] = ToString(INITIAL_DELAY_MS);
            specificReplace["HTTP_READINESS_MIN_RESTART_PERIOD"] = "2000";
            return specificReplace;
        }

    private:
        const size_t INITIAL_DELAY_MS = 2000;
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestWorkloadWithHttpReadiness");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWithHttpReadinessTimeout) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookCheckFailed = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().http_get_status().time_limit().consecutive_failures_counter() >= 1;
            };

            THttpService httpService = CreateAndStartHttpService(Config_, MakeSimpleShared<TFakeSlowRouter>());

            TickTree(Tree_, 60, breakHookCheckFailed);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().http_get_status().time_limit().consecutive_failures_counter() >= 1, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_READINESS_FAILED, TConsoleRenderer(false).Render(Tree_));
            TString errorMsg = TStringBuilder()
                << "readiness probe failed: "
                << "HTTP request to http2://localhost:" << Port_ << "/mock_path: "
                << "Request timeout";
            UNIT_ASSERT_EQUAL_C(status.readiness_status().http_get_status().last().inner_fail_reason(), errorMsg, NProtobufJson::Proto2Json(status));

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["READINESS_HOOK_TYPE"] = "http";
            specificReplace["HTTP_READINESS_MAX_EXECUTION_TIME"] = "500";
            specificReplace["HTTP_READINESS_PORT"] = ToString(Config_.GetPort());

            return specificReplace;
        }

    private:
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestWorkloadWithHttpReadinessTimeout");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWithHttpLiveness) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookReady = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().http_get_status().time_limit().consecutive_successes_counter() > 0
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART
                ;
            };

            auto breakHookCheckFailed = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().http_get_status().time_limit().consecutive_failures_counter() > 0
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().http_get_status().current().state() == API::EHttpGetState_WAITING_INIT
                ;
            };

            TickTree(Tree_, 32, breakHookCheckFailed);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto httpStatus = status.liveness_status().http_get_status();
            UNIT_ASSERT_C(httpStatus.time_limit().consecutive_failures_counter() > 0, NProtobufJson::Proto2Json(httpStatus));
            UNIT_ASSERT_C(status.start().killed_externally_counter() > 0, NProtobufJson::Proto2Json(status));
            TInstant startTs = NSupport::ToInstant(status.start().last().start_time());
            TInstant firstHookTs = NSupport::ToInstant(httpStatus.last().start_time());
            UNIT_ASSERT_GE_C(firstHookTs - startTs, TDuration::MilliSeconds(INITIAL_DELAY_MS), firstHookTs << " " << startTs);
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WAITING_INIT, httpStatus.current().state()
                , NProtobufJson::Proto2Json(httpStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_FAILURE, httpStatus.last().state()
                , NProtobufJson::Proto2Json(httpStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            THttpService httpService = CreateAndStartHttpService(Config_, MakeSimpleShared<TFakeSuccessRouter>());

            TickTree(Tree_, 32, breakHookReady);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            httpStatus = status.liveness_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(1, httpStatus.time_limit().consecutive_successes_counter(), NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WAITING_RESTART
                , httpStatus.current().state()
                , NProtobufJson::Proto2Json(httpStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_SUCCESS
                , httpStatus.last().state()
                , NProtobufJson::Proto2Json(httpStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["READINESS_HOOK_TYPE"] = "no_hook";
            specificReplace["LIVENESS_HOOK_TYPE"] = "http";
            specificReplace["HTTP_LIVENESS_PORT"] = ToString(Config_.GetPort());
            specificReplace["HTTP_LIVENESS_INITIAL_DELAY"] = ToString(INITIAL_DELAY_MS);
            specificReplace["HTTP_LIVENESS_MIN_RESTART_PERIOD"] = "2000";
            return specificReplace;
        }

    private:
        const size_t INITIAL_DELAY_MS = 2000;
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestWorkloadWithHttpLiveness");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWithTcpReadiness) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
            , Port_(NTesting::GetFreePort())
        {}

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().tcp_check_status().current().state() == API::ETcpCheckState_WAITING_RESTART
                ;
            };

            auto breakHookCheckFailed = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().tcp_check_status().time_limit().consecutive_failures_counter() >= 2;
            };

            auto breakHookWaitingInit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().tcp_check_status().current().state() == API::ETcpCheckState_WAITING_INIT;
            };

            TickTree(Tree_, 32, breakHookWaitingInit);
            Sleep(TDuration::MilliSeconds(2000));
            auto tcpStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().tcp_check_status();
            UNIT_ASSERT_EQUAL_C(tcpStatus.current().state(), API::ETcpCheckState_WAITING_INIT, NProtobufJson::Proto2Json(tcpStatus) << Endl << TConsoleRenderer(false).Render(Tree_));

            TickTree(Tree_, 32, breakHookCheckFailed);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().tcp_check_status().time_limit().consecutive_failures_counter() >= 2, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_READINESS_FAILED, TConsoleRenderer(false).Render(Tree_));

            TInet6StreamSocket listenSocket = CreateBindedSocket(Port_);
            UNIT_ASSERT_EQUAL_C(listenSocket.Listen(1), 0, strerror(errno));

            TickTree(Tree_, 16, breakHookActive);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(status.readiness_status().tcp_check_status().time_limit().consecutive_successes_counter(), 1, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(
                status.readiness_status().tcp_check_status().current().state()
                , API::ETcpCheckState_WAITING_RESTART
                , TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.readiness_status().tcp_check_status().last().state()
                , API::ETcpCheckState_SUCCESS
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["READINESS_HOOK_TYPE"] = "tcp";
            specificReplace["LIVENESS_HOOK_TYPE"] = "no_hook";
            specificReplace["TCP_READINESS_PORT"] = ToString(Port_);
            specificReplace["TCP_READINESS_MAX_EXECUTION_TIME"] = "1000";
            specificReplace["TCP_READINESS_INITIAL_DELAY"] = "2000";
            specificReplace["TCP_READINESS_RESTART_PERIOD_SCALE"] = "1";
            specificReplace["TCP_READINESS_RESTART_PERIOD_BACKOFF"] = "60000";
            specificReplace["TCP_READINESS_MAX_RESTART_PERIOD"] = "60000";
            specificReplace["TCP_READINESS_MIN_RESTART_PERIOD"] = "2000";

            return specificReplace;
        }
    private:
        NTesting::TPortHolder Port_;
    };

    TTest test("TestWorkloadWithTcpReadiness");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWithTcpLiveness) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
            , Port_(NTesting::GetFreePort())
        {}

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookReady = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().tcp_check_status().time_limit().consecutive_successes_counter() > 0
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().tcp_check_status().current().state() == API::ETcpCheckState_WAITING_RESTART
                ;
            };

            auto breakHookCheckFailed = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().tcp_check_status().time_limit().consecutive_failures_counter() > 0
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().killed_externally_counter() > 0
                ;
            };

            TickTree(Tree_, 32, breakHookCheckFailed);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto tcpStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().tcp_check_status();
            UNIT_ASSERT_C(tcpStatus.time_limit().consecutive_failures_counter() > 0, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_C(status.start().killed_externally_counter() > 0, NProtobufJson::Proto2Json(status));
            TInstant startTs = NSupport::ToInstant(status.start().last().start_time());
            TInstant hookTs = NSupport::ToInstant(tcpStatus.last().start_time());
            UNIT_ASSERT_GE_C(hookTs - startTs, TDuration::MilliSeconds(INITIAL_DELAY_MS), hookTs << " " << startTs);

            TInet6StreamSocket listenSocket = CreateBindedSocket(Port_);
            UNIT_ASSERT_EQUAL_C(listenSocket.Listen(1), 0, strerror(errno));

            TickTree(Tree_, 32, breakHookReady);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            tcpStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().tcp_check_status();
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(tcpStatus.time_limit().consecutive_successes_counter() > 0, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(
                tcpStatus.current().state()
                , API::ETcpCheckState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                tcpStatus.last().state()
                , API::ETcpCheckState_SUCCESS
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["READINESS_HOOK_TYPE"] = "no_hook";
            specificReplace["LIVENESS_HOOK_TYPE"] = "tcp";
            specificReplace["TCP_LIVENESS_PORT"] = ToString(Port_);
            specificReplace["TCP_LIVENESS_MAX_EXECUTION_TIME"] = "200";
            specificReplace["TCP_LIVENESS_INITIAL_DELAY"] = ToString(INITIAL_DELAY_MS);
            specificReplace["TCP_LIVENESS_RESTART_PERIOD_SCALE"] = "1";
            specificReplace["TCP_LIVENESS_RESTART_PERIOD_BACKOFF"] = "60000";
            specificReplace["TCP_LIVENESS_MAX_RESTART_PERIOD"] = "60000";
            specificReplace["TCP_LIVENESS_MIN_RESTART_PERIOD"] = "2000";
            return specificReplace;
        }

    private:
        const size_t INITIAL_DELAY_MS = 2000;
        NTesting::TPortHolder Port_;
    };

    TTest test("TestWorkloadWithTcpLiveness");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWithHttpLivenessPortoKillStartFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client)
            : TWrapperPortoClient(client)
            , LastKillName_(TPortoContainerName(""))
        {
        }

        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int /*sig*/) override {
            LastKillName_ = name;
            return TPortoError{EPortoError::InvalidState, TStringBuilder() << "Container " << TString(name) << " in invalid state"};
        }

        TPortoContainerName GetLastKillName() const {
            return LastKillName_;
        }

    private:
        TPortoContainerName LastKillName_;
    };

    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().system_failure_counter() > 0
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().http_get_status().time_limit().consecutive_failures_counter() >= 2
                ;
            };

            TickTree(Tree_, 32, breakHook);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.liveness_status().http_get_status().time_limit().consecutive_failures_counter() >= 2, NProtobufJson::Proto2Json(status));

            auto startStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start();
            UNIT_ASSERT_C(startStatus.current().state() == API::EContainerState_SYSTEM_FAILURE || startStatus.current().state() == API::EContainerState_RUNNING, NProtobufJson::Proto2Json(startStatus) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(startStatus.current().fail_reason(), "InvalidState", startStatus.current().fail_reason());
            UNIT_ASSERT_C(startStatus.system_failure_counter() > 0, NProtobufJson::Proto2Json(startStatus));

            UNIT_ASSERT_EQUAL_C(GetFullWorkloadContainerName("start"), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["READINESS_HOOK_TYPE"] = "no_hook";
            specificReplace["LIVENESS_HOOK_TYPE"] = "http";

            specificReplace["HTTP_LIVENESS_PORT"] = ToString(Config_.GetPort());
            specificReplace["HTTP_LIVENESS_MIN_RESTART_PERIOD"] = "1";

            return specificReplace;
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto);
        }

    private:
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestWorkloadWithHttpLivenessPortoKillStartFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadWithTcpLivenessPortoKillStartFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client)
            : TWrapperPortoClient(client)
            , LastKillName_(TPortoContainerName(""))
        {
        }

        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int /*sig*/) override {
            LastKillName_ = name;
            return TPortoError{EPortoError::InvalidState, TStringBuilder() << "Container " << TString(name) << " in invalid state"};
        }

        TPortoContainerName GetLastKillName() const {
            return LastKillName_;
        }

    private:
        TPortoContainerName LastKillName_;
    };

    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
            , Port_(NTesting::GetFreePort())
        {}

    protected:
        void Test() override {
            PrepareBox();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().system_failure_counter() > 0
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().tcp_check_status().time_limit().consecutive_failures_counter() >= 2
                ;
            };

            TickTree(Tree_, 32, breakHook);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.liveness_status().tcp_check_status().time_limit().consecutive_failures_counter() >= 2, NProtobufJson::Proto2Json(status));

            auto startStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start();
            UNIT_ASSERT_C(startStatus.current().state() == API::EContainerState_SYSTEM_FAILURE || startStatus.current().state() == API::EContainerState_RUNNING, NProtobufJson::Proto2Json(startStatus) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(startStatus.current().fail_reason(), "InvalidState", startStatus.current().fail_reason());
            UNIT_ASSERT_C(startStatus.system_failure_counter() > 0, NProtobufJson::Proto2Json(startStatus));

            UNIT_ASSERT_EQUAL_C(GetFullWorkloadContainerName("start"), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LIVENESS_HOOK_TYPE"] = "tcp";
            specificReplace["TCP_LIVENESS_PORT"] = ToString(Port_);
            specificReplace["TCP_LIVENESS_MAX_EXECUTION_TIME"] = "1000";
            specificReplace["TCP_LIVENESS_INITIAL_DELAY"] = "0";
            specificReplace["TCP_LIVENESS_RESTART_PERIOD_SCALE"] = "1";
            specificReplace["TCP_LIVENESS_RESTART_PERIOD_BACKOFF"] = "60000";
            specificReplace["TCP_LIVENESS_MAX_RESTART_PERIOD"] = "60000";
            specificReplace["TCP_LIVENESS_MIN_RESTART_PERIOD"] = "1";
            return specificReplace;
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto);
        }

    private:
        NTesting::TPortHolder Port_;
    };

    TTest test("TestWorkloadWithTcpLivenessPortoKillStartFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadShouldBeDestroyedBadBox) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);

            auto breakHookDestroyed = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED
                    && !WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_)
                ;
            };

            TickTree(Tree_, 32, breakHookDestroyed);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_REMOVED, status.state(), TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(false, WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_), TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWorkloadShouldBeDestroyedBadBox");
    test.DoTest();
}

Y_UNIT_TEST(TestRemoveOldContainers) {
    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto readyHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            TPortoContainerName oldInit = PathHolder_->GetWorkloadInitContainer(BoxId_, WorkloadId_, 1);
            SafePorto_->CreateRecursive(oldInit).Success();
            SafePorto_->SetProperty(oldInit, EPortoContainerProperty::Command, "sleep 1000");
            SafePorto_->SetProperty(oldInit, EPortoContainerProperty::Private, PackContainerPrivate({CP_READY, "old_tree_hash"}));
            SafePorto_->Start(oldInit);

            UNIT_ASSERT_EQUAL(SafePorto_->IsContainerExists(oldInit).Success(), true);

            TickTree(Tree_, 32, readyHook);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, status.state(), TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_EQUAL(SafePorto_->IsContainerExists(oldInit).Success(), false);
        }
    };

    TTest test("TestRemoveOldContainers");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStopAndDestroyRemovedBeforeStart) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(
            TPortoClientPtr client
            , const TPortoContainerName& stopContainer
            , const TPortoContainerName& destroyContainer
        )
            : TWrapperPortoClient(client)
            , StopContainer_(stopContainer)
            , DestroyContainer_(destroyContainer)
            , StopRemoved_(false)
            , DestroyRemoved_(false)
        {
        }

        TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override {
            if (name == StopContainer_) {
                StopRemoved_ = true;
            }
            if (name == DestroyContainer_) {
                DestroyRemoved_ = true;
            }

            return TWrapperPortoClient::Destroy(name);
        }

        bool GetStopRemoved() const {
            return StopRemoved_;
        }

        bool GetDestroyRemoved() const {
            return DestroyRemoved_;
        }

    private:
        const TPortoContainerName StopContainer_;
        const TPortoContainerName DestroyContainer_;

        bool StopRemoved_;
        bool DestroyRemoved_;
    };

    // We don't need any real network client functions, so we use completely mock client
    class TTestNetworkClient: public TMockNetworkClient {
    public:
        TTestNetworkClient(
            const TString& stopRequestKey
            , const TString& destroyRequestKey
        )
            : StopRequestKey_(stopRequestKey)
            , DestroyRequestKey_(destroyRequestKey)
            , StopRemoved_(false)
            , DestroyRemoved_(false)
        {
        }

        TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) override {
            if (requestKey == StopRequestKey_) {
                StopRemoved_ = true;
            }
            if (requestKey == DestroyRequestKey_) {
                DestroyRemoved_ = true;
            }

            return TExpected<void, TNetworkClientError>::DefaultSuccess();
        }

        bool GetStopRemoved() const {
            return StopRemoved_;
        }

        bool GetDestroyRemoved() const {
            return DestroyRemoved_;
        }

    private:
        const TString StopRequestKey_;
        const TString DestroyRequestKey_;

        bool StopRemoved_;
        bool DestroyRemoved_;
    };

    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            SafePorto_->Create(GetFullWorkloadContainerName("stop")).Success();
            SafePorto_->Create(GetFullWorkloadContainerName("destroy")).Success();

            TickTree(Tree_, 32, breakHookActive);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_ACTIVE, TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetStopRemoved(), "stop container was not removed");
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetDestroyRemoved(), "destroy container was not removed");

            UNIT_ASSERT_C(((TTestNetworkClient*)TreeNetworkClient_.Get())->GetStopRemoved(), "stop network request was not removed");
            UNIT_ASSERT_C(((TTestNetworkClient*)TreeNetworkClient_.Get())->GetDestroyRemoved(), "destroy network request was not removed");
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(
                porto
                , GetFullWorkloadContainerName("stop")
                , GetFullWorkloadContainerName("destroy")
            );
        }

        TNetworkClientPtr GetSpecificNetworkClient(TNetworkClientPtr /* networkClient */) const override {
            return new TTestNetworkClient(
                // Generation from NetworkBasicNode
                WorkloadId_ + "_" + ToString(NStatusRepositoryTypes::ENetworkHookType::STOP)
                , WorkloadId_ + "_" + ToString(NStatusRepositoryTypes::ENetworkHookType::DESTROY)
            );
        }
    };

    TTest test("TestWorkloadStopAndDestroyRemovedBeforeStart");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStopRemovedBeforeDestroy) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(
            TPortoClientPtr client
            , const TPortoContainerName& stopContainer
            , const TPortoContainerName& destroyContainer
        )
            : TWrapperPortoClient(client)
            , StopContainer_(stopContainer)
            , DestroyContainer_(destroyContainer)
            , StopRemoved_(false)
            , DestroyRemoved_(false)
        {
        }

        TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override {
            if (name == StopContainer_) {
                StopRemoved_ = true;
            }
            if (name == DestroyContainer_) {
                DestroyRemoved_ = true;
            }

            return TWrapperPortoClient::Destroy(name);
        }

        bool GetStopRemoved() const {
            return StopRemoved_;
        }

        bool GetDestroyRemoved() const {
            return DestroyRemoved_;
        }

    private:
        const TPortoContainerName StopContainer_;
        const TPortoContainerName DestroyContainer_;

        bool StopRemoved_;
        bool DestroyRemoved_;
    };

    // We don't need any real network client functions, so we use completely mock client
    class TTestNetworkClient: public TMockNetworkClient {
    public:
        TTestNetworkClient(
            const TString& stopRequestKey
            , const TString& destroyRequestKey
        )
            : StopRequestKey_(stopRequestKey)
            , DestroyRequestKey_(destroyRequestKey)
            , StopRemoved_(false)
            , DestroyRemoved_(false)
        {
        }

        TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) override {
            if (requestKey == StopRequestKey_) {
                StopRemoved_ = true;
            }
            if (requestKey == DestroyRequestKey_) {
                DestroyRemoved_ = true;
            }

            return TExpected<void, TNetworkClientError>::DefaultSuccess();
        }

        bool GetStopRemoved() const {
            return StopRemoved_;
        }

        bool GetDestroyRemoved() const {
            return DestroyRemoved_;
        }

    private:
        const TString StopRequestKey_;
        const TString DestroyRequestKey_;

        bool StopRemoved_;
        bool DestroyRemoved_;
    };

    class TTest : public ITestWorkloadTreeCanon {
    public:
        TTest(const TString& testName)
            : ITestWorkloadTreeCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);

            auto breakHookRemoving = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVING;
            };

            SafePorto_->Create(GetFullWorkloadContainerName("stop")).Success();

            TickTree(Tree_, 32, breakHookRemoving);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_REMOVING, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_), TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetStopRemoved(), "stop container was not removed");
            UNIT_ASSERT_C(!((TTestPortoClient*)TreePorto_.Get())->GetDestroyRemoved(), "destroy container removed");

            UNIT_ASSERT_C(((TTestNetworkClient*)TreeNetworkClient_.Get())->GetStopRemoved(), "stop network request was not removed");
            UNIT_ASSERT_C(!((TTestNetworkClient*)TreeNetworkClient_.Get())->GetDestroyRemoved(), "destroy network request removed");
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(
                porto
                , GetFullWorkloadContainerName("stop")
                , GetFullWorkloadContainerName("destroy")
            );
        }

        TNetworkClientPtr GetSpecificNetworkClient(TNetworkClientPtr /* networkClient */) const override {
            return new TTestNetworkClient(
                // Generation from NetworkBasicNode
                WorkloadId_ + "_" + ToString(NStatusRepositoryTypes::ENetworkHookType::STOP)
                , WorkloadId_ + "_" + ToString(NStatusRepositoryTypes::ENetworkHookType::DESTROY)
            );
        }
    };

    TTest test("TestWorkloadStopRemovedBeforeDestroy");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
