#include <infra/libs/http_service/service.h>
#include <infra/libs/http_service/test_common.h>
#include <infra/libs/service_iface/fake_routers.h>
#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/trees/workload/base/test/workload_test_canon.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/support_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/protobuf/json/proto2json.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestTreeWorkloadDestroyCanon: public ITestWorkloadCanon {
public:
    ITestTreeWorkloadDestroyCanon(const TString& testName)
        : ITestWorkloadCanon(testName, "TreeWorkloadDestroy", "TreeWorkloadDestroy")
    {
    }

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["WORKLOAD_ID"] = WorkloadId_;
        replace["WORKLOAD_ULIMIT"] = "";
        replace["WORKLOAD_CAPABILITIES_AMBIENT"] = "NET_BIND_SERVICE";

        replace["DESTROY_HOOK_TYPE"] = "container";
        replace["DESTROY_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "destroy");
        replace["DESTROY_ENVIRONMENT"] = "";
        replace["DESTROY_SECRET_ENVIRONMENT"] = "";
        replace["DESTROY_CMD"] = "bash -c \"echo destroy; sleep 5000\"";
        replace["DESTROY_CWD"] = "";
        replace["DESTROY_CPU_GUARANTEE"] = "1c";
        replace["DESTROY_CPU_LIMIT"] = "1c";
        replace["DESTROY_CPU_POLICY"] = "normal";
        replace["DESTROY_CPU_WEIGHT"] = "1.01";
        replace["DESTROY_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["DESTROY_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["DESTROY_ANON_LIMIT"] = ToString(1 << 25);
        replace["DESTROY_RECHARGE_ON_PGFAULT"] = "false";
        replace["DESTROY_THREAD_LIMIT"] = "1001";
        replace["DESTROY_CORE_COMMAND"] = "";
        replace["DESTROY_USER"] = "";
        replace["DESTROY_GROUP"] = "";
        replace["DESTROY_AGING_TIME"] = ToString(1 << 16);
        replace["DESTROY_MIN_RESTART_PERIOD"] = "60000";
        replace["DESTROY_MAX_RESTART_PERIOD"] = "60000";
        replace["DESTROY_RESTART_PERIOD_BACKOFF"] = "1";
        replace["DESTROY_RESTART_PERIOD_SCALE"] = "0";
        replace["DESTROY_MAX_EXECUTION_TIME"] = "60000";
        replace["DESTROY_INITIAL_DELAY"] = "0";
        replace["DESTROY_MAX_TRIES"] = "1000";
        replace["DESTROY_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["DESTROY_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["DESTROY_STDOUT_LOG_PATH"] = "";
        replace["DESTROY_STDERR_LOG_PATH"] = "";
        replace["DESTROY_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["DESTROY_IO_LIMIT"] = "/tmp r: 20000";
        replace["DESTROY_IO_OPS_LIMIT"] = "/tmp r: 20001";
        replace["DESTROY_IO_POLICY"] = "normal";
        replace["DESTROY_IO_WEIGHT"] = "1.02";

        replace["BOX_CONTAINER"] = PathHolder_->GetBoxContainer(BoxId_);
        replace["TREE_HASH"] = "tree_hash";

        replace["HTTP_DESTROY_PATH"] = "/mock_path";
        replace["HTTP_DESTROY_PORT"] = "";
        replace["HTTP_DESTROY_RESPONSE"] = "";
        replace["HTTP_DESTROY_ANY_RESPONSE"] = "true";
        replace["HTTP_DESTROY_MAX_EXECUTION_TIME"] = "500";
        replace["HTTP_DESTROY_INITIAL_DELAY"] = "0";
        replace["HTTP_DESTROY_RESTART_PERIOD_SCALE"] = "1";
        replace["HTTP_DESTROY_RESTART_PERIOD_BACKOFF"] = "60000";
        replace["HTTP_DESTROY_MAX_RESTART_PERIOD"] = "60000";
        replace["HTTP_DESTROY_MIN_RESTART_PERIOD"] = "60000";

        return replace;
    }
};

Y_UNIT_TEST_SUITE(TreeWorkloadContainerDestroyTestSuite) {

Y_UNIT_TEST(TestWorkloadDestroyContainerWhenNoContainerCmdAndHttpHookPath) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto breakHookDestroyed = [this]() {
                return !WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_);
            };

            TickTree(Tree_, 32, breakHookDestroyed);
            UNIT_ASSERT_EQUAL_C(
                WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_)
                , false
                , TConsoleRenderer(false).Render(Tree_)
            );

            auto destroyStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status();
            UNIT_ASSERT_EQUAL_C(destroyStatus.last().fail_reason(), "", destroyStatus.last().fail_reason());

            UNIT_ASSERT_EQUAL_C(destroyStatus.last().stdout_tail(), "", destroyStatus.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(destroyStatus.last().stderr_tail(), "", destroyStatus.last().stderr_tail());

            UNIT_ASSERT_EQUAL_C(destroyStatus.zero_return_code_counter(), 0, destroyStatus.zero_return_code_counter());
            UNIT_ASSERT_EQUAL_C(destroyStatus.positive_return_code_counter(), 0, destroyStatus.positive_return_code_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_HOOK_TYPE"] = "no_hook";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadDestroyContainerWhenNoContainerCmdAndHttpHookPath");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyContainerSuccess) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto breakHookDestroyed = [this]() {
                return !WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_);
            };

            TickTree(Tree_, 32, breakHookDestroyed);
            UNIT_ASSERT_EQUAL_C(
                WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_)
                , false
                , TConsoleRenderer(false).Render(Tree_)
            );

            auto destroyStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status();

            UNIT_ASSERT_EQUAL_C(destroyStatus.last().fail_reason(), "", destroyStatus.last().fail_reason());

            UNIT_ASSERT_EQUAL_C(destroyStatus.last().stdout_tail(), "destroy\n", destroyStatus.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(destroyStatus.last().stderr_tail(), "destroy\n", destroyStatus.last().stderr_tail());

            UNIT_ASSERT_EQUAL_C(destroyStatus.zero_return_code_counter(), 1, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(destroyStatus.positive_return_code_counter(), 0, destroyStatus.positive_return_code_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_CMD"] = "bash -c \"echo destroy; >&2 echo destroy;\"";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadDestroyContainerSuccess");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyStateRemoving) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto breakHookRemoving = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVING
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status().current().state() == API::EContainerState_RUNNING
                ;
            };

            TickTree(Tree_, 32, breakHookRemoving);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_REMOVING, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(
                status.destroy_status().container_status().current().state()
                , API::EContainerState_RUNNING
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_CMD"] = "sleep 1000";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadDestroyStateRemoving");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyContainerBackOff) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto breakHookFirstExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status().positive_return_code_counter() == 1;
            };
            auto breakHookThirdExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status().positive_return_code_counter() == 3
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status().current().state() == API::EContainerState_WAITING_RESTART
                ;
            };

            TickTree(Tree_, 32, breakHookFirstExit);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto hookStatus = status.destroy_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                1, hookStatus.positive_return_code_counter()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant firstTs = NSupport::ToInstant(hookStatus.last().start_time());
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_EXITED, hookStatus.last().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TickTree(Tree_, 32, breakHookThirdExit);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            hookStatus = status.destroy_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                3, hookStatus.positive_return_code_counter()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant thirdTs = NSupport::ToInstant(hookStatus.last().start_time());
            UNIT_ASSERT_GE_C(thirdTs - firstTs, TDuration::MilliSeconds(BACKOFF_MS), thirdTs << " " << firstTs);
            UNIT_ASSERT_EQUAL_C(
                hookStatus.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                hookStatus.last().state()
                , API::EContainerState_EXITED
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_MIN_RESTART_PERIOD"] = "0";
            specificReplace["DESTROY_RESTART_PERIOD_BACKOFF"] = ToString(BACKOFF_MS);
            specificReplace["DESTROY_RESTART_PERIOD_SCALE"] = "1";
            specificReplace["DESTROY_CMD"] = "bash -c \"echo destroy; >&2 echo destroy; cd this_dir_does_not_exist\"";
            return specificReplace;
        }
    private:
        const size_t BACKOFF_MS = 2000;
    };

    TTest test("TestWorkloadDestroyContainerBackOff");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyContainerTimeout) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto breakHookExited = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status().killed_externally_counter() > 0
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status().last().state() == API::EContainerState_TIMEOUT
                ;
            };

            TickTree(Tree_, 12, breakHookExited);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(
                status.destroy_status().container_status().killed_externally_counter()
                , 1
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.destroy_status().container_status().timeout_counter()
                , 1
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.destroy_status().container_status().last().state()
                , API::EContainerState_TIMEOUT
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_CMD"] = "sleep 20";
            specificReplace["DESTROY_MAX_EXECUTION_TIME"] = "1000";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadDestroyContainerTimeout");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyContainerRemovedRunningAfterTries) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto breakHookDestroyFinished = [this]() {
                return !WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_)
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status().current().state() == API::EContainerState_UNKNOWN
                ;
            };

            TickTree(Tree_, 32, breakHookDestroyFinished);

            auto destroyStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                destroyStatus.current().state()
                , API::EContainerState_UNKNOWN
                , NProtobufJson::Proto2Json(destroyStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(destroyStatus.last().fail_reason(), "destroy hook failed all attemps 'bash -c \"echo destroy; >&2 echo destroy; cd this_dir_does_not_exist\"'", destroyStatus.last().fail_reason());

            UNIT_ASSERT_EQUAL_C(destroyStatus.last().stdout_tail(), "destroy\n", destroyStatus.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(destroyStatus.last().stderr_tail(), "destroy\nbash: line 0: cd: this_dir_does_not_exist: No such file or directory\n", destroyStatus.last().stderr_tail());

            UNIT_ASSERT_EQUAL_C(destroyStatus.zero_return_code_counter(), 0, destroyStatus.zero_return_code_counter());
            UNIT_ASSERT_EQUAL_C(destroyStatus.positive_return_code_counter(), 3, destroyStatus.positive_return_code_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_MIN_RESTART_PERIOD"] = "1";
            specificReplace["DESTROY_MAX_RESTART_PERIOD"] = "1";
            specificReplace["DESTROY_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["DESTROY_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["DESTROY_MAX_TRIES"] = "3";
            specificReplace["DESTROY_CMD"] = "bash -c \"echo destroy; >&2 echo destroy; cd this_dir_does_not_exist\"";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadDestroyContainerRemovedRunningAfterTries");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyContainerCreateFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , FailCreateCalls_(0)
        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            if (name == ContainerToFail_) {
                ++FailCreateCalls_;
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetFailCreateCalls() {
            return FailCreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 FailCreateCalls_;
    };

    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto breakHookDestroyed = [this]() {
                return !WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_);
            };

            TickTree(Tree_, 4, breakHookDestroyed);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_SYSTEM_FAILURE
                , API::EContainerState_Name(status.last().state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_)
            );

            const TString containerName = GetFullWorkloadContainerName("destroy");

            UNIT_ASSERT_STRING_CONTAINS(status.last().fail_reason(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_EQUAL_C(status.system_failure_counter(), 3, status.system_failure_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(containerName), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_EQUAL_C(((TTestPortoClient*)TreePorto_.Get())->GetFailCreateCalls(), 3, ((TTestPortoClient*)TreePorto_.Get())->GetFailCreateCalls());
            UNIT_ASSERT(!WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, GetFullWorkloadContainerName("destroy"));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_MAX_TRIES"] = "3";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadDestroyContainerCreateFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyFailToStartContainer) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto breakHookDestroyed = [this]() {
                return !WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_);
            };

            TickTree(Tree_, 32, breakHookDestroyed);
            const auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(
                !WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_)
                , TConsoleRenderer(false).Render(Tree_)
            );

            const auto destroyStatus = status.destroy_status().container_status();
            UNIT_ASSERT_STRING_CONTAINS_C(
                destroyStatus.last().fail_reason()
                , "InvalidCommand:(No such file or directory: cannot exec some123)"
                , destroyStatus.last().fail_reason()
            );
            UNIT_ASSERT_EQUAL_C(
                destroyStatus.last().state()
                , API::EContainerState_SYSTEM_FAILURE
                , API::EContainerState_Name(destroyStatus.last().state())
            );
            // Check feedback start_time on PortoStart fail
            UNIT_ASSERT_C(
                destroyStatus.last().start_time().seconds() != 0 || destroyStatus.last().start_time().nanos() != 0
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_CMD"] = "some123 fail command";
            specificReplace["DESTROY_MAX_TRIES"] = "3";
            specificReplace["DESTROY_MIN_RESTART_PERIOD"] = "0";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadDestroyFailToStartContainer");
    test.DoTest();
}

}

Y_UNIT_TEST_SUITE(TreeWorkloadHttpDestroyHookTestSuite) {

Y_UNIT_TEST(TestDestroyWithHttp) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeSuccessRouter>();
            THttpService httpService(Config_, router);
            httpService.Start(fakeFrame);

            auto breakHookDestroyFinished = [this]() {
                return !WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_);
            };

            TickTree(Tree_, 30, breakHookDestroyFinished);

            UNIT_ASSERT_EQUAL_C(
                WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_)
                , false
                , TConsoleRenderer(false).Render(Tree_)
            );

            auto destroyStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(destroyStatus.last().fail_reason(), "", destroyStatus.last().fail_reason());
            UNIT_ASSERT_EQUAL_C(destroyStatus.success_counter(), 1, destroyStatus.success_counter());

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_DESTROY_PORT"] = ToString(Config_.GetPort());
            specificReplace["DESTROY_HOOK_TYPE"] = "http";

            return specificReplace;
        }

    private:
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestDestroyWithHttp");
    test.DoTest();
}

Y_UNIT_TEST(TestFailToDestroyWithHttpWithBadAnswer) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeSuccessRouter>();
            THttpService httpService(Config_, router);
            httpService.Start(fakeFrame);

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART;
            };

            TickTree(Tree_, 10, breakHook);

            auto hookState = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().current().state();
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WAITING_RESTART
                , hookState
                , API::EHttpGetState_Name(hookState) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            auto hookStateLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().last().state();
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WRONG_ANSWER
                , hookStateLast
                , API::EHttpGetState_Name(hookStateLast) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TString failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().current().fail_reason();
            UNIT_ASSERT_C(failReason == "", failReason);
            TString failReasonLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().last().fail_reason();
            UNIT_ASSERT_STRING_CONTAINS(failReasonLast, "OtherData");

            auto wrongAnswerCounter = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().wrong_answer_counter();
            UNIT_ASSERT_EQUAL_C(wrongAnswerCounter, 1, wrongAnswerCounter);

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["HTTP_DESTROY_PORT"] = ToString(Config_.GetPort());
            specificReplace["HTTP_DESTROY_RESPONSE"] = "OtherData";
            specificReplace["HTTP_DESTROY_ANY_RESPONSE"] = "false";
            specificReplace["DESTROY_HOOK_TYPE"] = "http";

            return specificReplace;
        }

    private:
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestFailToDestroyWithHttpWithBadAnswer");
    test.DoTest();
}

Y_UNIT_TEST(TestFailToDestroyWithHttp) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(Config_, router);
            httpService.Start(fakeFrame);

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART;
            };

            TickTree(Tree_, 10, breakHook);

            auto hookState = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().current().state();
            UNIT_ASSERT_EQUAL_C(API::EHttpGetState_WAITING_RESTART, hookState, TConsoleRenderer(false).Render(Tree_));
            auto hookStateLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().last().state();
            UNIT_ASSERT_EQUAL_C(API::EHttpGetState_FAILURE, hookStateLast, TConsoleRenderer(false).Render(Tree_));

            TString failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().current().inner_fail_reason();
            UNIT_ASSERT_C(failReason == "", failReason);
            TString failReasonLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().last().inner_fail_reason();
            UNIT_ASSERT_STRING_CONTAINS(failReasonLast, "404");

            auto errorCounter = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().error_counter();
            UNIT_ASSERT_EQUAL_C(errorCounter, 1, errorCounter);

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_DESTROY_PORT"] = ToString(Config_.GetPort());
            specificReplace["DESTROY_HOOK_TYPE"] = "http";

            return specificReplace;
        }

    private:
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestFailToDestroyWithHttp");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyBackOffWithHttp) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(Config_, router);
            httpService.Start(fakeFrame);

            auto breakHookSecondExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().time_limit().consecutive_failures_counter() == 2
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART
                ;
            };

            auto breakHookThirdExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().time_limit().consecutive_failures_counter() == 3
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART
                ;
            };

            TickTree(Tree_, 32, breakHookSecondExit);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto hookStatus = status.destroy_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(
                2, hookStatus.time_limit().consecutive_failures_counter()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant firstTs = NSupport::ToInstant(hookStatus.last().start_time());
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WAITING_RESTART, hookStatus.current().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_FAILURE, hookStatus.last().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TickTree(Tree_, 32, breakHookThirdExit);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            hookStatus = status.destroy_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(
                3, hookStatus.time_limit().consecutive_failures_counter()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant thirdTs = NSupport::ToInstant(hookStatus.last().start_time());
            UNIT_ASSERT_GE_C(thirdTs - firstTs, TDuration::MilliSeconds(BACKOFF_MS), thirdTs << " " << firstTs);
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WAITING_RESTART, hookStatus.current().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_FAILURE, hookStatus.last().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 3, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[1], "/mock_path", router->GetPaths()[1]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[2], "/mock_path", router->GetPaths()[1]);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["DESTROY_HOOK_TYPE"] = "http";
            specificReplace["HTTP_DESTROY_MIN_RESTART_PERIOD"] = "0";
            specificReplace["HTTP_DESTROY_PORT"] = ToString(Config_.GetPort());
            specificReplace["HTTP_DESTROY_RESTART_PERIOD_BACKOFF"] = ToString(BACKOFF_MS);
            return specificReplace;
        }

    private:
        const size_t BACKOFF_MS = 2000;
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestWorkloadDestroyBackOffWithHttp");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyTimeoutWithHttp) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeSlowRouter>();
            THttpService httpService(Config_, router);
            httpService.Start(fakeFrame);

            auto breakHookFail = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().time_limit().consecutive_failures_counter() > 0;
            };

            TickTree(Tree_, 10, breakHookFail);

            TString failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().last().inner_fail_reason();
            TString errorMsg = TStringBuilder()
                << "destroy policy failed: "
                << "HTTP request to http2://localhost:" << Port_ << "/mock_path: "
                << "Request timeout";
            UNIT_ASSERT_EQUAL_C(failReason, errorMsg, failReason);

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_DESTROY_PORT"] = ToString(Config_.GetPort());
            specificReplace["DESTROY_HOOK_TYPE"] = "http";

            return specificReplace;
        }

    private:
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestWorkloadDestroyTimeoutWithHttp");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDestroyedAfterTriesWithHttp) {
    class TTest : public ITestTreeWorkloadDestroyCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadDestroyCanon(testName)
            , Port_(NTesting::GetFreePort())
            , Config_(NTestCommon::GenerateHttpServiceConfig(Port_))
        {
        }

    protected:

        void Test() override {
            PrepareBox();
            WorkloadStatusRepositoryInternal_->UpdateObjectToBeDestroyedByHook(WorkloadId_, true);
            UNIT_ASSERT(WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_));

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(Config_, router);
            httpService.Start(fakeFrame);

            auto breakHookDestroyed = [this]() {
                return !WorkloadStatusRepositoryInternal_->GetObjectShouldBeDestroyedByHook(WorkloadId_);
            };

            TickTree(Tree_, 32, breakHookDestroyed);

            auto failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status().last().inner_fail_reason();

            TString errorMsg = TStringBuilder()
                << "destroy policy failed: "
                << "HTTP request to http2://localhost:" << Port_ << "/mock_path: "
                << "request failed(HTTP/1.1 404 Not found)";
            UNIT_ASSERT_EQUAL_C(failReason, errorMsg, "expected: " + failReason + ", got: " + errorMsg);

            auto hookStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).destroy_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(hookStatus.time_limit().consecutive_successes_counter(), 0, hookStatus.time_limit().consecutive_successes_counter());
            UNIT_ASSERT_EQUAL_C(hookStatus.time_limit().consecutive_failures_counter(), 3, hookStatus.time_limit().consecutive_failures_counter());

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 3, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[1], "/mock_path", router->GetPaths()[1]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[2], "/mock_path", router->GetPaths()[2]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_DESTROY_PORT"] = ToString(Config_.GetPort());
            specificReplace["DESTROY_MAX_TRIES"] = "3";
            specificReplace["DESTROY_HOOK_TYPE"] = "http";
            specificReplace["HTTP_DESTROY_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["HTTP_DESTROY_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["HTTP_DESTROY_MAX_RESTART_PERIOD"] = "1";

            return specificReplace;
        }

    private:
        NTesting::TPortHolder Port_;
        THttpServiceConfig Config_;
    };

    TTest test("TestWorkloadDestroyedAfterTriesWithHttp");
    test.DoTest();
}

}

}// namespace NInfra::NPodAgent::NTreeTest
