#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/trees/workload/base/test/workload_test_canon.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/testing/unittest/registar.h>

#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestTreeWorkloadInitCanon: public ITestWorkloadCanon {
public:
    ITestTreeWorkloadInitCanon(const TString& testName)
        : ITestWorkloadCanon(testName, "TreeWorkloadInit", "TreeInit")
        , ContainerName_(PathHolder_->GetWorkloadInitContainer(BoxId_, WorkloadId_, 1).GetChild())
        , ContainerNamePrevious_(PathHolder_->GetWorkloadInitContainer(BoxId_, WorkloadId_, 0).GetChild())
    {
    }

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["CONTAINER_NAME"] = ContainerName_;
        replace["CONTAINER_NAME_PREVIOUS"] = ContainerNamePrevious_;

        replace["OBJECT_TYPE"] = "workload";
        replace["OBJECT_INVALID_STATE"] = "EWorkloadState_INVALID";
        replace["OBJECT_INIT_PROCESSES_STATE"] = "EWorkloadState_INIT_PROCESSES";
        replace["OBJECT_ID_OR_HASH"] = WorkloadId_;

        replace["INIT_ANON_LIMIT"] = ToString(1 << 25);
        replace["INIT_CMD"] = "echo init1";
        replace["INIT_CWD"] = "";
        replace["INIT_CPU_GUARANTEE"] = "1c";
        replace["INIT_CPU_LIMIT"] = "1c";
        replace["INIT_CPU_POLICY"] = "normal";
        replace["INIT_CPU_WEIGHT"] = "1.01";
        replace["INIT_HOSTNAME"] = "";
        replace["INIT_INITIAL_DELAY"] = "0";
        replace["INIT_MAX_RESTART_PERIOD"] = "1000";
        replace["INIT_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["INIT_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["INIT_THREAD_LIMIT"] = "1001";
        replace["INIT_MIN_RESTART_PERIOD"] = "1000";
        replace["INIT_NUM"] = "1";
        replace["INIT_RECHARGE_ON_PGFAULT"] = "false";
        replace["INIT_CAPABILITIES_AMBIENT"] = "NET_BIND_SERVICE";
        replace["INIT_ENVIRONMENT"] = "";
        replace["INIT_SECRET_ENVIRONMENT"] = "";
        replace["INIT_ULIMIT"] = "";
        replace["INIT_CORE_COMMAND"] = "";
        replace["INIT_USER"] = "";
        replace["INIT_GROUP"] = "";
        replace["INIT_AGING_TIME"] = ToString(1 << 16);
        replace["INIT_RESTART_PERIOD_BACKOFF"] = "1";
        replace["INIT_RESTART_PERIOD_SCALE"] = "0";
        replace["INIT_MAX_EXECUTION_TIME"] = "60000";
        replace["INIT_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["INIT_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["INIT_STDOUT_LOG_PATH"] = "";
        replace["INIT_STDERR_LOG_PATH"] = "";
        replace["INIT_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["INIT_IO_LIMIT"] = "/tmp r: 20000";
        replace["INIT_IO_OPS_LIMIT"] = "/tmp r: 20001";
        replace["INIT_IO_POLICY"] = "normal";
        replace["INIT_IO_WEIGHT"] = "1.02";
        replace["TREE_HASH"] = "tree_hash";

        return replace;
    }

    void SetupTest() override {
        PrepareWorkload(GetWorkloadInitSize());
        PrepareFirstInit();
    }

private:
    void PrepareFirstInit() {
        SafePorto_->Create({ContainerNamePrevious_});
        SafePorto_->SetProperty({ContainerNamePrevious_}, EPortoContainerProperty::Private, "ready:tree_hash");
    }

protected:
    const TString ContainerName_;
    const TString ContainerNamePrevious_;
};

Y_UNIT_TEST_SUITE(TreeInitTestSuite) {

Y_UNIT_TEST(TestRestartInit) {
    class TTest : public ITestTreeWorkloadInitCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadInitCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookExited = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init_size() > 1
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init(1).positive_return_code_counter() > 1
                ;
            };

            TickTree(Tree_, 32, breakHookExited);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_INVALID, status.state(), TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.init_size() == 2, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.init(1).positive_return_code_counter() > 1, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.init(1).zero_return_code_counter() == 0, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.init(1).last().return_code() == 1, Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["INIT_CMD"] = "false";

            return specificReplace;
        }

        ui32 GetWorkloadInitSize() const override {
            return 2;
        }
    };

    TTest test("TestRestartInit");
    test.DoTest();
}

Y_UNIT_TEST(TestInitInvalidCommand) {
    class TTest : public ITestTreeWorkloadInitCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadInitCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init_size() > 1
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_INVALID
                ;
            };

            TickTree(Tree_, 32, breakHook);

            const auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(
                API::EWorkloadState_INVALID
                , status.state()
                , TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(status.init_size(), 2, status.init_size());

            const TString expectedFailReason = "InvalidCommand:(No such file or directory: cannot exec invalid_command)";
            const auto initStatus = status.init(1);
            UNIT_ASSERT_STRING_CONTAINS_C(
                initStatus.last().fail_reason()
                , expectedFailReason
                , initStatus.last().fail_reason()
            );
            UNIT_ASSERT_EQUAL_C(
                initStatus.last().state()
                , API::EContainerState_SYSTEM_FAILURE
                , API::EContainerState_Name(initStatus.last().state())
            );
            // Check feedback start_time on PortoStart fail
            UNIT_ASSERT_C(
                initStatus.last().start_time().seconds() != 0 || initStatus.last().start_time().nanos() != 0
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["INIT_CMD"] = "invalid_command";

            return specificReplace;
        }

        ui32 GetWorkloadInitSize() const override {
            return 2;
        }
    };

    TTest test("TestInitInvalidCommand");
    test.DoTest();
}

Y_UNIT_TEST(TestTimeoutInit) {
    class TTest : public ITestTreeWorkloadInitCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadInitCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init_size() > 1
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init(1).current().state() == API::EContainerState_WAITING_RESTART
                ;
            };

            TickTree(Tree_, 32, breakHookWaitingRestart);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_INVALID, status.state(), TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(status.init_size(), 2, status.init_size());
            UNIT_ASSERT_C(status.init(1).killed_externally_counter() > 0, status.init(1).killed_externally_counter());
            UNIT_ASSERT_EQUAL_C(
                status.init(1).current().state()
                , API::EContainerState_WAITING_RESTART
                , Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.init(1).last().state()
                , API::EContainerState_TIMEOUT
                , Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["INIT_CMD"] = "sleep 1000";

            specificReplace["INIT_MIN_RESTART_PERIOD"] = "60000";
            specificReplace["INIT_MAX_RESTART_PERIOD"] = "60000";
            specificReplace["INIT_MAX_EXECUTION_TIME"] = "1000";

            return specificReplace;
        }

        ui32 GetWorkloadInitSize() const override {
            return 2;
        }
    };

    TTest test("TestTimeoutInit");
    test.DoTest();
}

Y_UNIT_TEST(TestInitFileWrite) {
    class TTest : public ITestTreeWorkloadInitCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadInitCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookExited = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init_size() == 2
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init(1).last().state() == API::EContainerState_EXITED
                ;
            };

            TickTree(Tree_, 32, breakHookExited);

            UNIT_ASSERT_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init_size() == 2, Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(
                WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init(1).last().state() == API::EContainerState_EXITED
                , Endl << API::EContainerState_Name(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init(1).last().state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto cwd = SafePorto_->GetProperty({ContainerName_}, EPortoContainerProperty::Cwd).Success();
            UNIT_ASSERT(NFs::Exists(cwd + "/init.txt"));
            auto content = TFileInput(cwd + "/init.txt").ReadAll();
            UNIT_ASSERT_EQUAL_C(content, "init\n", content);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["INIT_CMD"] = "bash -c \"echo init > init.txt\"";

            return specificReplace;
        }

        ui32 GetWorkloadInitSize() const override {
            return 2;
        }
    };

    TTest test("TestInitFileWrite");
    test.DoTest();
}

Y_UNIT_TEST(TestCreateInitContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)

        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            if (name == ContainerToFail_) {
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestTreeWorkloadInitCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadInitCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init_size() > 1
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).init(1).system_failure_counter() > 0
                ;
            };

            TickTree(Tree_, 32, breakHook);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_INVALID, status.state(), TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.init_size() == 2, Endl << TConsoleRenderer(false).Render(Tree_));

            auto initStatus = status.init(1);
            UNIT_ASSERT_STRING_CONTAINS(initStatus.last().fail_reason(), TStringBuilder() << "Can't create " << ContainerName_ << " container");
            UNIT_ASSERT_C(initStatus.system_failure_counter() > 0, initStatus.system_failure_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(ContainerName_), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls() > 0, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, ContainerName_);
        }

        ui32 GetWorkloadInitSize() const override {
            return 2;
        }
    };

    TTest test("TestCreateInitContainerFail");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
