#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/trees/workload/base/test/workload_test_canon.h>
#include <infra/pod_agent/libs/network_client/mock_client.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/test_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/testing/unittest/registar.h>

#include <util/folder/path.h>
#include <util/folder/pathsplit.h>
#include <util/stream/file.h>
#include <util/system/fs.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestTreeWorkloadStartCanon: public ITestWorkloadCanon {
public:
    ITestTreeWorkloadStartCanon(const TString& testName)
        : ITestWorkloadCanon(testName, "TreeWorkloadStart", "TreeWorkloadStart")
    {
    }

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["WORKLOAD_ID"] = WorkloadId_;
        replace["WORKLOAD_ULIMIT"] = "core: unlimited unlimited; ";
        replace["WORKLOAD_CAPABILITIES_AMBIENT"] = "NET_BIND_SERVICE";

        replace["START_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "start");
        replace["START_ENVIRONMENT"] = "";
        replace["START_SECRET_ENVIRONMENT"] = "";
        replace["START_CMD"] = "bash -c \"echo start; sleep 1000\"";
        replace["START_CWD"] = "";
        replace["START_CPU_GUARANTEE"] = "1c";
        replace["START_CPU_LIMIT"] = "1c";
        replace["START_CPU_POLICY"] = "normal";
        replace["START_CPU_WEIGHT"] = "1.01";
        replace["START_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["START_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["START_ANON_LIMIT"] = ToString(1 << 25);
        replace["START_RECHARGE_ON_PGFAULT"] = "false";
        replace["START_THREAD_LIMIT"] = "1001";
        replace["START_CORE_COMMAND"] = "bash -c \"echo dumped\"";
        replace["START_USER"] = "";
        replace["START_GROUP"] = "";
        replace["START_AGING_TIME"] = ToString(1 << 16);
        replace["START_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["START_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["START_STDOUT_LOG_PATH"] = "";
        replace["START_STDERR_LOG_PATH"] = "";
        replace["START_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["START_IO_LIMIT"] = "/tmp r: 20000";
        replace["START_IO_OPS_LIMIT"] = "/tmp r: 20001";
        replace["START_IO_POLICY"] = "normal";
        replace["START_IO_WEIGHT"] = "1.02";

        replace["READINESS_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "readiness");
        replace["READINESS_ENVIRONMENT"] = "";
        replace["READINESS_SECRET_ENVIRONMENT"] = "";
        replace["READINESS_HOOK_TYPE"] = "container";
        replace["READINESS_SUCCESS_THRESHOLD"] = "1";
        replace["READINESS_FAILURE_THRESHOLD"] = "1";
        replace["READINESS_CMD"] = "echo readiness";
        replace["READINESS_CWD"] = "";
        replace["READINESS_CPU_GUARANTEE"] = "1c";
        replace["READINESS_CPU_LIMIT"] = "1c";
        replace["READINESS_CPU_POLICY"] = "normal";
        replace["READINESS_CPU_WEIGHT"] = "1.03";
        replace["READINESS_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["READINESS_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["READINESS_ANON_LIMIT"] = ToString(1 << 25);
        replace["READINESS_RECHARGE_ON_PGFAULT"] = "false";
        replace["READINESS_THREAD_LIMIT"] = "1002";
        replace["READINESS_CORE_COMMAND"] = "";
        replace["READINESS_USER"] = "";
        replace["READINESS_GROUP"] = "";
        replace["READINESS_AGING_TIME"] = ToString(1 << 16);
        replace["READINESS_MIN_RESTART_PERIOD"] = "60000";
        replace["READINESS_MAX_RESTART_PERIOD"] = "60000";
        replace["READINESS_RESTART_PERIOD_BACKOFF"] = "1";
        replace["READINESS_RESTART_PERIOD_SCALE"] = "0";
        replace["READINESS_INITIAL_DELAY"] = "0";
        replace["READINESS_MAX_EXECUTION_TIME"] = "60000";
        replace["READINESS_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["READINESS_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["READINESS_STDOUT_LOG_PATH"] = "";
        replace["READINESS_STDERR_LOG_PATH"] = "";
        replace["READINESS_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["READINESS_IO_LIMIT"] = "/tmp r: 20002";
        replace["READINESS_IO_OPS_LIMIT"] = "/tmp r: 20003";
        replace["READINESS_IO_POLICY"] = "normal";
        replace["READINESS_IO_WEIGHT"] = "1.04";

        replace["LIVENESS_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "liveness");
        replace["LIVENESS_ENVIRONMENT"] = "";
        replace["LIVENESS_SECRET_ENVIRONMENT"] = "";
        replace["LIVENESS_HOOK_TYPE"] = "container";
        replace["LIVENESS_SUCCESS_THRESHOLD"] = "1";
        replace["LIVENESS_FAILURE_THRESHOLD"] = "1";
        replace["LIVENESS_CMD"] = "echo liveness";
        replace["LIVENESS_CWD"] = "";
        replace["LIVENESS_CPU_GUARANTEE"] = "1c";
        replace["LIVENESS_CPU_LIMIT"] = "1c";
        replace["LIVENESS_CPU_POLICY"] = "normal";
        replace["LIVENESS_CPU_WEIGHT"] = "1.05";
        replace["LIVENESS_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["LIVENESS_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["LIVENESS_ANON_LIMIT"] = ToString(1 << 25);
        replace["LIVENESS_RECHARGE_ON_PGFAULT"] = "false";
        replace["LIVENESS_THREAD_LIMIT"] = "1003";
        replace["LIVENESS_CORE_COMMAND"] = "";
        replace["LIVENESS_USER"] = "";
        replace["LIVENESS_GROUP"] = "";
        replace["LIVENESS_AGING_TIME"] = ToString(1 << 16);
        replace["LIVENESS_MIN_RESTART_PERIOD"] = "60000";
        replace["LIVENESS_MAX_RESTART_PERIOD"] = "60000";
        replace["LIVENESS_RESTART_PERIOD_BACKOFF"] = "1";
        replace["LIVENESS_RESTART_PERIOD_SCALE"] = "0";
        replace["LIVENESS_INITIAL_DELAY"] = "0";
        replace["LIVENESS_MAX_EXECUTION_TIME"] = "60000";
        replace["LIVENESS_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["LIVENESS_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["LIVENESS_STDOUT_LOG_PATH"] = "";
        replace["LIVENESS_STDERR_LOG_PATH"] = "";
        replace["LIVENESS_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["LIVENESS_IO_LIMIT"] = "/tmp r: 20004";
        replace["LIVENESS_IO_OPS_LIMIT"] = "/tmp r: 20005";
        replace["LIVENESS_IO_POLICY"] = "normal";
        replace["LIVENESS_IO_WEIGHT"] = "1.06";

        replace["TREE_HASH"] = "tree_hash";

        replace["HTTP_READINESS_PATH"] = "";
        replace["HTTP_READINESS_PORT"] = "90";
        replace["HTTP_READINESS_RESPONSE"] = "http_readiness_response";
        replace["HTTP_READINESS_ANY_RESPONSE"] = "false";
        replace["HTTP_READINESS_MAX_EXECUTION_TIME"] = "1000";
        replace["HTTP_READINESS_INITIAL_DELAY"] = "0";
        replace["HTTP_READINESS_RESTART_PERIOD_SCALE"] = "0";
        replace["HTTP_READINESS_RESTART_PERIOD_BACKOFF"] = "1";
        replace["HTTP_READINESS_MAX_RESTART_PERIOD"] = "60000";
        replace["HTTP_READINESS_MIN_RESTART_PERIOD"] = "60000";

        replace["HTTP_LIVENESS_PATH"] = "";
        replace["HTTP_LIVENESS_PORT"] = "90";
        replace["HTTP_LIVENESS_RESPONSE"] = "http_liveness_response";
        replace["HTTP_LIVENESS_ANY_RESPONSE"] = "false";
        replace["HTTP_LIVENESS_MAX_EXECUTION_TIME"] = "1000";
        replace["HTTP_LIVENESS_INITIAL_DELAY"] = "0";
        replace["HTTP_LIVENESS_RESTART_PERIOD_SCALE"] = "0";
        replace["HTTP_LIVENESS_RESTART_PERIOD_BACKOFF"] = "1";
        replace["HTTP_LIVENESS_MAX_RESTART_PERIOD"] = "60000";
        replace["HTTP_LIVENESS_MIN_RESTART_PERIOD"] = "60000";

        replace["TCP_READINESS_PATH"] = "";
        replace["TCP_READINESS_PORT"] = "";
        replace["TCP_READINESS_RESPONSE"] = "";
        replace["TCP_READINESS_MAX_EXECUTION_TIME"] = "";
        replace["TCP_READINESS_INITIAL_DELAY"] = "";
        replace["TCP_READINESS_RESTART_PERIOD_SCALE"] = "";
        replace["TCP_READINESS_RESTART_PERIOD_BACKOFF"] = "";
        replace["TCP_READINESS_MAX_RESTART_PERIOD"] = "";
        replace["TCP_READINESS_MIN_RESTART_PERIOD"] = "";

        replace["TCP_LIVENESS_PATH"] = "";
        replace["TCP_LIVENESS_PORT"] = "";
        replace["TCP_LIVENESS_RESPONSE"] = "";
        replace["TCP_LIVENESS_MAX_EXECUTION_TIME"] = "";
        replace["TCP_LIVENESS_INITIAL_DELAY"] = "";
        replace["TCP_LIVENESS_RESTART_PERIOD_SCALE"] = "";
        replace["TCP_LIVENESS_RESTART_PERIOD_BACKOFF"] = "";
        replace["TCP_LIVENESS_MAX_RESTART_PERIOD"] = "";
        replace["TCP_LIVENESS_MIN_RESTART_PERIOD"] = "";

        return replace;
    }
};

Y_UNIT_TEST_SUITE(TreeWorkloadStartTestSuite) {

Y_UNIT_TEST(TestDestroyContainers) {
    class TTest : public ITestTreeWorkloadStartCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadStartCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            const TVector<TString> containers = {"readiness", "liveness", "start"};

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            TickTree(Tree_, 24, breakHook);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), TConsoleRenderer(false).Render(Tree_));
            for (auto container : containers) {
                auto result = SafePorto_->Destroy(GetFullWorkloadContainerName(container));
                UNIT_ASSERT_C((bool)result, result.Error().Message);

                TickTree(Tree_, 24, breakHook);
                auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
                UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, status.state(), EWorkloadState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_));
            }
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["READINESS_MIN_RESTART_PERIOD"] = "500";
            specificReplace["READINESS_MAX_RESTART_PERIOD"] = "500";
            return specificReplace;
        }
    };

    TTest test("TestDestroyContainers");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStartPositiveReturnCodeCounterTick) {
    class TTest : public ITestTreeWorkloadStartCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadStartCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().positive_return_code_counter() > 0;
            };

            TickTree(Tree_, 40, breakHook);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_DEAD, TConsoleRenderer(false).Render(Tree_));
            TString expectedStderr =  "bash: line 0: cd: this_dir_does_not_exist: No such file or directory\n";
            UNIT_ASSERT_EQUAL_C(status.start().last().stderr_tail(), expectedStderr, status.start().last().stderr_tail());
            UNIT_ASSERT_EQUAL_C(status.start().last().stdout_tail(), "start\n", status.start().last().stdout_tail());
            UNIT_ASSERT_C(status.start().positive_return_code_counter() > 0,  status.start().positive_return_code_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["START_CMD"] = "bash -c 'echo start; cd this_dir_does_not_exist'";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadStartPositiveReturnCodeCounterTick");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStartFailReasonAndFeedbackStartTime) {
    class TTest : public ITestTreeWorkloadStartCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadStartCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().fail_reason() != "";
            };

            TickTree(Tree_, 24, breakHook);
            const auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            const TString expectedFailReason = "InvalidCommand:(No such file or directory: cannot exec invalid_command)";
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EWorkloadState_INVALID
                , TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_STRING_CONTAINS_C(
                status.start().last().fail_reason()
                , expectedFailReason
                , status.start().last().fail_reason()
            );
            UNIT_ASSERT_EQUAL_C(
                status.start().last().state()
                , API::EContainerState_SYSTEM_FAILURE
                , API::EContainerState_Name(status.start().last().state())
            );
            // Check feedback start_time on PortoStart fail
            UNIT_ASSERT_C(
                status.start().last().start_time().seconds() != 0 || status.start().last().start_time().nanos() != 0
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["START_CMD"] = "invalid_command";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadStartFailReasonAndFeedbackStartTime");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadMemoryLimit) {
    class TTest : public ITestTreeWorkloadStartCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadStartCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().oom_counter() > 0;
            };

            TickTree(Tree_, 32, breakHook);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start();
            UNIT_ASSERT_C(
                status.oom_counter() > 0
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["START_CMD"] = "dd if=/dev/zero bs=64M count=1 of=/dev/null";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadMemoryLimit");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadCPULimit) {
    class TTest : public ITestTreeWorkloadStartCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadStartCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            TickTree(Tree_, 24, breakHookActive);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, status.state(), TConsoleRenderer(false).Render(Tree_));
            auto startWorkload = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "start");
            ui64 cpuUsage = FromString(SafePorto_->GetProperty(startWorkload, EPortoContainerProperty::CpuUsage).Success());

            UNIT_ASSERT_C(cpuUsage < 250 * 1000 * 1000, cpuUsage);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["START_CPU_GUARANTEE"] = "0.1c";
            specificReplace["START_CPU_LIMIT"] = "0.1c";
            specificReplace["START_CMD"] = "bash -c '\
                start=$(date +%s%N);\
                while [ $(($(date +%s%N) - $start)) -le 500000000 ]; do\
                    for i in {1..10000}; do\
                        var=$((var+1));\
                    done;\
                    echo $(($(date +%s%N) - $start));\
                done;\
                sleep 10\
            '";
            specificReplace["READINESS_CMD"] = "sleep 0.5";
            specificReplace["LIVENESS_CMD"] = "sleep 0.5";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadCPULimit");
    test.DoTest();
}

Y_UNIT_TEST(TestStartContainerPositiveExitCodeCount) {
    class TTest : public ITestTreeWorkloadStartCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadStartCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookDeactivated = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_DEACTIVATED_BY_LIVENESS
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().killed_externally_counter() > 0
                ;
            };

            TickTree(Tree_, 32, breakHookDeactivated);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.start().killed_externally_counter() > 0, NProtobufJson::Proto2Json(status));

            auto workloadState = status.state();
            UNIT_ASSERT_EQUAL_C(
                workloadState
                , API::EWorkloadState_DEACTIVATED_BY_LIVENESS
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto returnCode = status.start().last().return_code();
            UNIT_ASSERT_C(returnCode < 0, returnCode);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["LIVENESS_CMD"] = "wget";
            specificReplace["LIVENESS_MAX_RESTART_PERIOD"] = "0";

            return specificReplace;
        }
    };

    TTest test("TestStartContainerPositiveExitCodeCount");
    test.DoTest();
}

Y_UNIT_TEST(TestCoreDump) {
    class TTest : public ITestTreeWorkloadStartCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadStartCanon(testName)
        {
        }

    protected:
        void Test() override {
            if (NPortoTestLib::IsInsideSandboxPortoIsolation()) {
                // TODO(DEPLOY-4308): Remove after solving blockers
                return;
            }

            PrepareBox();

            TFsPath(JoinPaths(TStringBuf(NFs::CurrentWorkingDirectory()), BINARY_NAME)).CopyTo(
                JoinPaths(TStringBuf(PathHolder_->GetBoxRootfsPath(BoxId_)), BINARY_NAME), 0
            );
            Chmod(JoinPaths(TStringBuf(PathHolder_->GetBoxRootfsPath(BoxId_)), BINARY_NAME).data(), MODE0755);

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state() == API::EContainerState_KILLED_EXTERNALLY;
            };

            TickTree(Tree_, 12, breakHook);

            TFileInput output(JoinPaths(TStringBuf(PathHolder_->GetBoxRootfsPath(BoxId_)), TStringBuf("output")), EOpenModeFlag::RdOnly);
            TString text = output.ReadAll();
            UNIT_ASSERT_EQUAL_C(text, BINARY_NAME, text);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["START_CMD"] = JoinPaths(TStringBuf("/"), BINARY_NAME);
            specificReplace["START_CORE_COMMAND"] = R"(bash -c 'printf "${CORE_EXE_NAME}" > /output')";

            return specificReplace;
        }

    private:
        const TStringBuf BINARY_NAME = "main";
    };

    TTest test("TestCoreDump");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStartCreateContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)

        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            if (name == ContainerToFail_) {
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestTreeWorkloadStartCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadStartCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state() == API::EContainerState_SYSTEM_FAILURE;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start();
            UNIT_ASSERT_EQUAL_C(status.last().state(), API::EContainerState_SYSTEM_FAILURE, API::EContainerState_Name(status.last().state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString containerName = GetFullWorkloadContainerName("start");

            UNIT_ASSERT_STRING_CONTAINS(status.last().fail_reason(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_C(status.system_failure_counter() > 0, status.system_failure_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(containerName), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls() > 0, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, GetFullWorkloadContainerName("start"));
        }
    };

    TTest test("TestWorkloadStartCreateContainerFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessAndLivenessRemovedBeforeStart) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(
            TPortoClientPtr client
            , const TPortoContainerName& readinessContainer
            , const TPortoContainerName& livenessContainer
        )
            : TWrapperPortoClient(client)
            , ReadinessContainer_(readinessContainer)
            , LivenessContainer_(livenessContainer)
            , ReadinessRemoved_(false)
            , LivenessRemoved_(false)
        {
        }

        TExpected<void, TPortoError> Destroy(const TPortoContainerName& name) override {
            if (name == ReadinessContainer_) {
                ReadinessRemoved_ = true;
            }
            if (name == LivenessContainer_) {
                LivenessRemoved_ = true;
            }

            return TWrapperPortoClient::Destroy(name);
        }

        bool GetReadinessRemoved() const {
            return ReadinessRemoved_;
        }

        bool GetLivenessRemoved() const {
            return LivenessRemoved_;
        }

    private:
        const TPortoContainerName ReadinessContainer_;
        const TPortoContainerName LivenessContainer_;

        bool ReadinessRemoved_;
        bool LivenessRemoved_;
    };

    // We don't need any real network client functions, so we use completely mock client
    class TTestNetworkClient: public TMockNetworkClient {
    public:
        TTestNetworkClient(
            const TString& readinessRequestKey
            , const TString& livenessRequestKey
        )
            : ReadinessRequestKey_(readinessRequestKey)
            , LivenessRequestKey_(livenessRequestKey)
            , ReadinessRemoved_(false)
            , LivenessRemoved_(false)
        {
        }

        TExpected<void, TNetworkClientError> RemoveRequest(const TString& requestKey) override {
            if (requestKey == ReadinessRequestKey_) {
                ReadinessRemoved_= true;
            }
            if (requestKey == LivenessRequestKey_) {
                LivenessRemoved_= true;
            }

            return TExpected<void, TNetworkClientError>::DefaultSuccess();
        }

        bool GetReadinessRemoved() const {
            return ReadinessRemoved_;
        }

        bool GetLivenessRemoved() const {
            return LivenessRemoved_;
        }

    private:
        const TString ReadinessRequestKey_;
        const TString LivenessRequestKey_;

        bool ReadinessRemoved_;
        bool LivenessRemoved_;
    };

    class TTest : public ITestTreeWorkloadStartCanon{
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadStartCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            auto breakHookRunning = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().current().state() == API::EContainerState_RUNNING
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().current().state() == API::EContainerState_RUNNING
                ;
            };

            TickTree(Tree_, 32, breakHookRunning);

            UNIT_ASSERT_C(!((TTestPortoClient*)TreePorto_.Get())->GetReadinessRemoved(), "readiness container was removed");
            UNIT_ASSERT_C(!((TTestPortoClient*)TreePorto_.Get())->GetLivenessRemoved(), "liveness container was removed");

            SafePorto_->Kill(GetFullWorkloadContainerName("start"), 9).Success();
            SafePorto_->WaitContainers({GetFullWorkloadContainerName("start")}).Success();

            WorkloadStatusRepository_->ClearContainerStatus(
                NStatusRepositoryTypes::TContainerDescription(
                    WorkloadId_
                    , NStatusRepositoryTypes::EObjectType::WORKLOAD
                    , NStatusRepositoryTypes::TContainerDescription::EContainerType::READINESS
                )
            );
            WorkloadStatusRepository_->ClearContainerStatus(
                NStatusRepositoryTypes::TContainerDescription(
                    WorkloadId_
                    , NStatusRepositoryTypes::EObjectType::WORKLOAD
                    , NStatusRepositoryTypes::TContainerDescription::EContainerType::LIVENESS
                )
            );

            TickTree(Tree_, 32, breakHookRunning);

            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetReadinessRemoved(), "readiness container was not removed");
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetLivenessRemoved(), "liveness container was not removed");

            UNIT_ASSERT_C(((TTestNetworkClient*)TreeNetworkClient_.Get())->GetReadinessRemoved(), "readiness network request was not removed");
            UNIT_ASSERT_C(((TTestNetworkClient*)TreeNetworkClient_.Get())->GetLivenessRemoved(), "liveness network request was not removed");
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(
                porto
                , GetFullWorkloadContainerName("readiness")
                , GetFullWorkloadContainerName("liveness")
            );
        }

        TNetworkClientPtr GetSpecificNetworkClient(TNetworkClientPtr /* networkClient */) const override {
            return new TTestNetworkClient(
                // Generation from NetworkBasicNode
                WorkloadId_ + "_" + ToString(NStatusRepositoryTypes::ENetworkHookType::READINESS)
                , WorkloadId_ + "_" + ToString(NStatusRepositoryTypes::ENetworkHookType::LIVENESS)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["READINESS_CMD"] = "bash -c \"echo readiness; sleep 1000\"";
            specificReplace["LIVENESS_CMD"] = "bash -c \"echo liveness; sleep 1000\"";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessAndLivenessRemovedBeforeStart");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
