#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/trees/workload/base/test/workload_test_canon.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/support_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestReadinessCanon: public ITestWorkloadCanon {
public:
    ITestReadinessCanon(const TString& testName)
        : ITestWorkloadCanon(testName, "TreeWorkloadReadiness", "TreeContainerStatusHook")
    {
    }

    void DoTest() {
        ITestWorkloadCanon::DoTest();
        // readiness must not kill start
        CheckStartContainerState(EPortoContainerState::Running);
    }

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["WORKLOAD_ID"] = WorkloadId_;
        replace["START_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "start");
        replace["WORKLOAD_ULIMIT"] = "";
        replace["WORKLOAD_CAPABILITIES_AMBIENT"] = "NET_BIND_SERVICE";

        replace["CONTAINER_AGING_TIME"] = ToString(1 << 16);
        replace["CONTAINER_ANON_LIMIT"] = ToString(1 << 25);
        replace["CONTAINER_CORE_COMMAND"] = "";
        replace["CONTAINER_CPU_GUARANTEE"] = "0.5c";
        replace["CONTAINER_CPU_LIMIT"] = "0.5c";
        replace["CONTAINER_CPU_POLICY"] = "normal";
        replace["CONTAINER_CPU_WEIGHT"] = "1.01";
        replace["CONTAINER_ENVIRONMENT"] = "";
        replace["CONTAINER_SECRET_ENVIRONMENT"] = "";
        replace["CONTAINER_GROUP"] = "";
        replace["CONTAINER_IO_LIMIT"] = "/tmp r: 20000";
        replace["CONTAINER_IO_OPS_LIMIT"] = "/tmp r: 20001";
        replace["CONTAINER_IO_POLICY"] = "normal";
        replace["CONTAINER_IO_WEIGHT"] = "1.02";
        replace["CONTAINER_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["CONTAINER_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["CONTAINER_NAME"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "readiness");
        replace["CONTAINER_RECHARGE_ON_PGFAULT"] = "false";
        replace["CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["CONTAINER_STDERR_LOG_PATH"] = "";
        replace["CONTAINER_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["CONTAINER_STDOUT_LOG_PATH"] = "";
        replace["CONTAINER_THREAD_LIMIT"] = "1001";
        replace["CONTAINER_USER"] = "";
        replace["FAILURE_THRESHOLD"] = "1";
        replace["HOOK_CMD"] = "echo readiness";
        replace["HOOK_CWD"] = "";
        replace["SUCCESS_THRESHOLD"] = "1";

        replace["HOOK_MIN_RESTART_PERIOD"] = "100";
        replace["HOOK_MAX_RESTART_PERIOD"] = "100";
        replace["HOOK_RESTART_PERIOD_BACKOFF"] = "1";
        replace["HOOK_RESTART_PERIOD_SCALE"] = "0";
        replace["HOOK_INITIAL_DELAY"] = "0";
        replace["HOOK_MAX_EXECUTION_TIME"] = "60000";

        replace["CONTAINER_TYPE"] = "readiness";
        replace["FEEDBACK_WHEN_HOOK_FINISHED_OK"] = "EWorkloadState_ACTIVE";
        replace["FEEDBACK_WHEN_HOOK_FAILED"] = "EWorkloadState_READINESS_FAILED";
        replace["FEEDBACK_WHEN_HOOK_FAILED_TO_START"] = "EWorkloadState_FAILED_TO_START_READINESS";
        replace["DESTROY_CONTAINER_WHEN_HOOK_FAILS"] = "";

        replace["TREE_HASH"] = "tree_hash";

        return replace;
    }

    void PrepareWorkloadStartContainer() {
        const TPortoContainerName workloadStartName = GetFullWorkloadContainerName("start");

        SafePorto_->Create(workloadStartName).Success();
        SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c \"echo start; sleep 1000\"").Success();
        SafePorto_->Start(workloadStartName).Success();
    }

    void CheckStartContainerState(const EPortoContainerState& state) {
        const TPortoContainerName workloadStartName = GetFullWorkloadContainerName("start");
        const EPortoContainerState containerState = FromString(SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success());

        UNIT_ASSERT_EQUAL_C(containerState, state, containerState);
    }
};

class ITestLivenessCanon: public ITestWorkloadCanon {
public:
    ITestLivenessCanon(const TString& testName)
        : ITestWorkloadCanon(testName, "TreeWorkloadLiveness", "TreeContainerStatusHook")
    {
    }

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;
        replace["WORKLOAD_ID"] = WorkloadId_;
        replace["START_CONTAINER"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "start");
        replace["WORKLOAD_ULIMIT"] = "";
        replace["WORKLOAD_CAPABILITIES_AMBIENT"] = "NET_BIND_SERVICE";

        replace["CONTAINER_AGING_TIME"] = ToString(1 << 16);
        replace["CONTAINER_ANON_LIMIT"] = ToString(1 << 25);
        replace["CONTAINER_CORE_COMMAND"] = "";
        replace["CONTAINER_CPU_GUARANTEE"] = "0.5c";
        replace["CONTAINER_CPU_LIMIT"] = "0.5c";
        replace["CONTAINER_CPU_POLICY"] = "normal";
        replace["CONTAINER_CPU_WEIGHT"] = "1.01";
        replace["CONTAINER_ENVIRONMENT"] = "";
        replace["CONTAINER_SECRET_ENVIRONMENT"] = "";
        replace["CONTAINER_GROUP"] = "";
        replace["CONTAINER_IO_LIMIT"] = "/tmp r: 20000";
        replace["CONTAINER_IO_OPS_LIMIT"] = "/tmp r: 20001";
        replace["CONTAINER_IO_POLICY"] = "normal";
        replace["CONTAINER_IO_WEIGHT"] = "1.02";
        replace["CONTAINER_MEMORY_GUARANTEE"] = ToString(1 << 25);
        replace["CONTAINER_MEMORY_LIMIT"] = ToString(1 << 25);
        replace["CONTAINER_NAME"] = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "liveness");
        replace["CONTAINER_RECHARGE_ON_PGFAULT"] = "false";
        replace["CONTAINER_STDERR_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["CONTAINER_STDERR_LOG_PATH"] = "";
        replace["CONTAINER_STDOUT_AND_STDERR_FILE_SIZE_LIMIT"] = "";
        replace["CONTAINER_STDOUT_LOG_FILE_FULL_PATH_TO_CREATE"] = "";
        replace["CONTAINER_STDOUT_LOG_PATH"] = "";
        replace["CONTAINER_THREAD_LIMIT"] = "1001";
        replace["CONTAINER_USER"] = "";
        replace["FAILURE_THRESHOLD"] = "1";
        replace["HOOK_CMD"] = "echo liveness";
        replace["HOOK_CWD"] = "";
        replace["SUCCESS_THRESHOLD"] = "1";

        replace["HOOK_MIN_RESTART_PERIOD"] = "100";
        replace["HOOK_MAX_RESTART_PERIOD"] = "100";
        replace["HOOK_RESTART_PERIOD_BACKOFF"] = "1";
        replace["HOOK_RESTART_PERIOD_SCALE"] = "0";
        replace["HOOK_INITIAL_DELAY"] = "0";
        replace["HOOK_MAX_EXECUTION_TIME"] = "60000";

        replace["CONTAINER_TYPE"] = "liveness";
        replace["FEEDBACK_WHEN_HOOK_FINISHED_OK"] = "EWorkloadState_UNKNOWN";
        replace["FEEDBACK_WHEN_HOOK_FAILED"] = "EWorkloadState_DEACTIVATED_BY_LIVENESS";
        replace["FEEDBACK_WHEN_HOOK_FAILED_TO_START"] = "EWorkloadState_FAILED_TO_START_LIVENESS";
        replace["DESTROY_CONTAINER_WHEN_HOOK_FAILS"] = "yes";

        replace["TREE_HASH"] = "tree_hash";

        return replace;
    }

    void PrepareWorkloadStartContainer() {
        const TPortoContainerName workloadStartName = PathHolder_->GetWorkloadContainerWithName(BoxId_, WorkloadId_, "start");

        SafePorto_->Create(workloadStartName).Success();
        SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c \"echo start; sleep 1000\"").Success();
        SafePorto_->Start(workloadStartName).Success();
    }

    void CheckStartContainerState(const EPortoContainerState& state) {
        const TPortoContainerName workloadStartName = GetFullWorkloadContainerName("start");
        const EPortoContainerState containerState = FromString(SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success());

        UNIT_ASSERT_EQUAL_C(containerState, state, containerState);
    }
};

Y_UNIT_TEST_SUITE(TreeContainerStatusHookTestSuite) {

Y_UNIT_TEST(TestWorkloadReadinessRestartPeriod) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().zero_return_code_counter() > 2;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status();
             UNIT_ASSERT_C(
                status.zero_return_code_counter() > 2
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_EXITED
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }
    };

    TTest test("TestWorkloadReadinessRestartPeriod");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessRestartPeriod) {
    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
                : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().zero_return_code_counter() > 2;
            };

            TickTree(Tree_, 24, breakHook);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status();
            UNIT_ASSERT_C(
                status.zero_return_code_counter() > 2
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_EXITED
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            CheckStartContainerState(EPortoContainerState::Running);
        }
    };

    TTest test("TestWorkloadLivenessRestartPeriod");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessSuccessThreshold) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHookOneTry = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().time_limit().consecutive_successes_counter() == 1;
            };
            auto breakHookTwoTries = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().time_limit().consecutive_successes_counter() >= 2;
            };

            TickTree(Tree_, 16, breakHookOneTry);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().container_status().time_limit().consecutive_successes_counter() == 1, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_UNEQUAL_C(API::EWorkloadState_ACTIVE, status.state(), TConsoleRenderer(false).Render(Tree_));

            TickTree(Tree_, 16, breakHookTwoTries);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().container_status().time_limit().consecutive_successes_counter() >= 2, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, status.state(), TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["SUCCESS_THRESHOLD"] = "2";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessSuccessThreshold");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessFailureThreshold) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHookOneTry = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().time_limit().consecutive_failures_counter() == 1;
            };
            auto breakHookTwoTries = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().time_limit().consecutive_failures_counter() >= 2;
            };

            TickTree(Tree_, 16, breakHookOneTry);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().container_status().time_limit().consecutive_failures_counter() == 1, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_UNEQUAL_C(API::EWorkloadState_READINESS_FAILED, status.state(), TConsoleRenderer(false).Render(Tree_));

            TickTree(Tree_, 16, breakHookTwoTries);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().container_status().time_limit().consecutive_failures_counter() >= 2, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_READINESS_FAILED, status.state(), TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["FAILURE_THRESHOLD"] = "2";
            specificReplace["HOOK_CMD"] = "bash -c 'echo readiness; cd this_dir_does_not_exist'";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessFailureThreshold");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessBackOff) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHookDoubleExit = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().zero_return_code_counter() >= 2;
            };
            auto breakHookUnexpectedExit = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().zero_return_code_counter() > 2;
            };

            TickTree(Tree_, 16, breakHookDoubleExit);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                status.zero_return_code_counter()
                , 2
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART,
                NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_EXITED,
                NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TickTree(Tree_, 4, breakHookUnexpectedExit);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                status.zero_return_code_counter()
                , 2
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART,
                NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_EXITED,
                NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_MAX_RESTART_PERIOD"] = "100000";
            specificReplace["HOOK_RESTART_PERIOD_BACKOFF"] = "100000";
            specificReplace["HOOK_RESTART_PERIOD_SCALE"] = "1";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessBackOff");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessBackOff) {
    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
                : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHookDoubleExit = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().zero_return_code_counter() >= 2;
            };
            auto breakHookUnexpectedExit = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().zero_return_code_counter() > 2;
            };

            TickTree(Tree_, 20, breakHookDoubleExit);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                status.zero_return_code_counter()
                , 2
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_EXITED
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TickTree(Tree_, 4, breakHookUnexpectedExit);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                status.zero_return_code_counter()
                , 2
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_EXITED
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            CheckStartContainerState(EPortoContainerState::Running);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_MAX_RESTART_PERIOD"] = "100000";
            specificReplace["HOOK_RESTART_PERIOD_BACKOFF"] = "100000";
            specificReplace["HOOK_RESTART_PERIOD_SCALE"] = "1";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadLivenessBackOff");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessInitialDelayAndBackOff) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHookFirstExit = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().zero_return_code_counter() == 1;
            };
            auto breakHookThirdExit = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().zero_return_code_counter() == 3;
            };

            TickTree(Tree_, 32, breakHookFirstExit);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto hookStatus = status.readiness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                1
                , hookStatus.zero_return_code_counter()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant startTs = NSupport::ToInstant(status.start().last().start_time());
            TInstant firstHookTs = NSupport::ToInstant(hookStatus.last().start_time());
            UNIT_ASSERT_GE_C(firstHookTs - startTs, TDuration::MilliSeconds(INITIAL_DELAY_MS), firstHookTs << " " << startTs);
            CheckStartContainerState(EPortoContainerState::Running);
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_WAITING_RESTART
                , hookStatus.current().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_EXITED
                , hookStatus.last().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TickTree(Tree_, 32, breakHookThirdExit);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            hookStatus = status.readiness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                3
                , hookStatus.zero_return_code_counter()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant thirdHookTs = NSupport::ToInstant(hookStatus.last().start_time());
            UNIT_ASSERT_GE_C(thirdHookTs - firstHookTs, TDuration::MilliSeconds(BACKOFF_MS), thirdHookTs << " " << firstHookTs);
            CheckStartContainerState(EPortoContainerState::Running);
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_WAITING_RESTART
                , hookStatus.current().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_EXITED
                , hookStatus.last().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_INITIAL_DELAY"] = ToString(INITIAL_DELAY_MS);
            specificReplace["HOOK_RESTART_PERIOD_BACKOFF"] = ToString(BACKOFF_MS);
            specificReplace["HOOK_RESTART_PERIOD_SCALE"] = "1";
            return specificReplace;
        }

    private:
        const size_t INITIAL_DELAY_MS = 2000;
        const size_t BACKOFF_MS = 2000;
    };

    TTest test("TestWorkloadReadinessInitialDelayAndBackOff");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessInitialDelayAndBackOff) {
    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
            : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHookFirstExit = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().zero_return_code_counter() == 1;
            };
            auto breakHookThirdExit = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().zero_return_code_counter() == 3;
            };

            TickTree(Tree_, 32, breakHookFirstExit);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto hookStatus = status.liveness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                1
                , hookStatus.zero_return_code_counter()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant startTs = NSupport::ToInstant(status.start().last().start_time());
            TInstant firstHookTs = NSupport::ToInstant(hookStatus.last().start_time());
            UNIT_ASSERT_GE_C(firstHookTs - startTs, TDuration::MilliSeconds(INITIAL_DELAY_MS), firstHookTs << " " << startTs);
            CheckStartContainerState(EPortoContainerState::Running);
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_WAITING_RESTART
                , hookStatus.current().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_EXITED
                , hookStatus.last().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TickTree(Tree_, 32, breakHookThirdExit);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            hookStatus = status.liveness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                3
                , hookStatus.zero_return_code_counter()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant thirdHookTs = NSupport::ToInstant(hookStatus.last().start_time());
            UNIT_ASSERT_GE_C(thirdHookTs - firstHookTs, TDuration::MilliSeconds(BACKOFF_MS), thirdHookTs << " " << firstHookTs);
            CheckStartContainerState(EPortoContainerState::Running);
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_WAITING_RESTART
                , hookStatus.current().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_EXITED
                , hookStatus.last().state()
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            CheckStartContainerState(EPortoContainerState::Running);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_INITIAL_DELAY"] = ToString(INITIAL_DELAY_MS);
            specificReplace["HOOK_RESTART_PERIOD_BACKOFF"] = ToString(BACKOFF_MS);
            specificReplace["HOOK_RESTART_PERIOD_SCALE"] = "1";
            return specificReplace;
        }

    private:
        const size_t INITIAL_DELAY_MS = 2000;
        const size_t BACKOFF_MS = 2000;
    };

    TTest test("TestWorkloadLivenessInitialDelayAndBackOff");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessInitialDelayAndContainerStatus) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
                : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHookWaiting = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().current().state() == API::EContainerState_WAITING_INIT;
            };

            TickTree(Tree_, 2, breakHookWaiting);

            auto breakHookExited = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().zero_return_code_counter() > 0;
            };

            TickTree(Tree_, 4, breakHookExited);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status();
            UNIT_ASSERT_C(status.zero_return_code_counter() == 0, NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(status.current().state(), API::EContainerState_WAITING_INIT, NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_INITIAL_DELAY"] = "5000";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessInitialDelayAndContainerStatus");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessTimeout) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
                : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHookExited = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().killed_externally_counter() > 0;
            };

            TickTree(Tree_, 12, breakHookExited);
            auto workloadStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto status = workloadStatus.readiness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                workloadStatus.state()
                , API::EWorkloadState_READINESS_FAILED
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.killed_externally_counter()
                , 1
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.timeout_counter()
                , 1
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_TIMEOUT
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "sleep 20";
            specificReplace["HOOK_MAX_EXECUTION_TIME"] = "1000";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessTimeout");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessContainerStatusRunning) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
                : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().current().state() == API::EContainerState_RUNNING;
            };

            TickTree(Tree_, 20, breakHook);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status();
            UNIT_ASSERT_C(status.current().state() == API::EContainerState_RUNNING, NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "sleep 100";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessInitialDelayAndContainerStatus");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessContainerStatusRunning) {
    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
                : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this](){
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().current().state() == API::EContainerState_RUNNING;
            };

            TickTree(Tree_, 16, breakHook);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status();
            UNIT_ASSERT_C(status.current().state() == API::EContainerState_RUNNING, NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_));

            CheckStartContainerState(EPortoContainerState::Running);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "sleep 100";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadLivenessInitialDelayAndContainerStatus");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessMemoryLimit) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().oom_counter() > 0;
            };

            TickTree(Tree_, 32, breakHook);
            auto workloadStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto status = workloadStatus.readiness_status().container_status();
            UNIT_ASSERT_C(status.oom_counter() > 0, NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(0, workloadStatus.start().oom_counter(), NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_OUT_OF_MEMORY
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "dd if=/dev/zero bs=64M count=1 of=/dev/null";
            specificReplace["HOOK_MIN_RESTART_PERIOD"] = "60000";
            specificReplace["HOOK_MAX_RESTART_PERIOD"] = "60000";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessMemoryLimit");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessMemoryLimit) {
    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
                : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().oom_counter() > 0
                    && FromString(SafePorto_->GetProperty(GetFullWorkloadContainerName("start"), EPortoContainerProperty::State).Success()) == EPortoContainerState::Dead
                ;
            };

            TickTree(Tree_, 32, breakHook);

            auto workloadStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto status = workloadStatus.liveness_status().container_status();
            UNIT_ASSERT_C(status.oom_counter() > 0, NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(0, workloadStatus.start().oom_counter(), NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_OUT_OF_MEMORY
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            CheckStartContainerState(EPortoContainerState::Dead);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "dd if=/dev/zero bs=64M count=1 of=/dev/null";
            specificReplace["HOOK_MIN_RESTART_PERIOD"] = "60000";
            specificReplace["HOOK_MAX_RESTART_PERIOD"] = "60000";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadLivenessMemoryLimit");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessTick) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            TickTree(Tree_, 16, breakHookActive);
            auto workloadStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto status = workloadStatus.readiness_status().container_status();
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, workloadStatus.state(), TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(status.zero_return_code_counter() > 0, status.zero_return_code_counter());
            UNIT_ASSERT_EQUAL_C(status.last().stdout_tail(), "readiness\n", status.current().stdout_tail());
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_EXITED
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }
    };

    TTest test("TestWorkloadReadinessTick");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessPositiveReturnCodeCounterTick) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_READINESS_FAILED;
            };

            TickTree(Tree_, 32, breakHook);
            auto workloadStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto status = workloadStatus.readiness_status().container_status();
            TString expectedStderr =  "bash: line 0: cd: this_dir_does_not_exist: No such file or directory\n";
            UNIT_ASSERT_EQUAL_C(workloadStatus.state(), API::EWorkloadState_READINESS_FAILED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(status.last().stderr_tail(), expectedStderr, status.last().stderr_tail());
            UNIT_ASSERT_EQUAL_C(status.last().stdout_tail(), "readiness\n", status.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(status.last().fail_reason(), "readiness probe failed: command 'bash -c 'echo readiness; cd this_dir_does_not_exist'': exit_code = '1'", status.last().fail_reason());
            UNIT_ASSERT_C(status.positive_return_code_counter() > 0, status.positive_return_code_counter());
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_EXITED
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "bash -c 'echo readiness; cd this_dir_does_not_exist'";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessPositiveReturnCodeCounterTick");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessPositiveReturnCodeCounterTick) {
    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
                : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_DEACTIVATED_BY_LIVENESS
                    && FromString(SafePorto_->GetProperty(GetFullWorkloadContainerName("start"), EPortoContainerProperty::State).Success()) == EPortoContainerState::Dead
                ;
            };

            TickTree(Tree_, 12, breakHook);

            auto workloadStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto status = workloadStatus.liveness_status().container_status();
            i32 cnt = status.positive_return_code_counter();
            TString expectedStderr =  "bash: line 0: cd: this_dir_does_not_exist: No such file or directory\n";
            UNIT_ASSERT_EQUAL_C(workloadStatus.state(), API::EWorkloadState_DEACTIVATED_BY_LIVENESS, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(status.last().stderr_tail(), expectedStderr, status.last().stderr_tail());
            UNIT_ASSERT_EQUAL_C(status.last().stdout_tail(), "liveness\n", status.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(status.last().fail_reason(), "liveness probe failed: command 'bash -c 'echo liveness; cd this_dir_does_not_exist'': exit_code = '1'", status.last().fail_reason());
            UNIT_ASSERT_C(cnt > 0, "Expected cnt > 0, got " << cnt);

            CheckStartContainerState(EPortoContainerState::Dead);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "bash -c 'echo liveness; cd this_dir_does_not_exist'";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadLivenessPositiveReturnCodeCounterTick");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessFailReasonAndFeedbackStartTime) {
    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().last().state() == API::EContainerState_SYSTEM_FAILURE
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_FAILED_TO_START_READINESS
                ;
            };

            TickTree(Tree_, 16, breakHook);
            const auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            const auto readinessStatus = status.readiness_status().container_status();

            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EWorkloadState_FAILED_TO_START_READINESS
                , API::EWorkloadState_Name(status.state())
            );
            UNIT_ASSERT_EQUAL_C(
                readinessStatus.last().state()
                , API::EContainerState_SYSTEM_FAILURE
                , API::EContainerState_Name(readinessStatus.last().state())
            );
            UNIT_ASSERT_STRING_CONTAINS_C(
                readinessStatus.last().fail_reason()
                , "InvalidCommand:(No such file or directory: cannot exec invalid_command)"
                , readinessStatus.last().fail_reason()
            );
            // Check feedback start_time on PortoStart fail
            UNIT_ASSERT_C(
                readinessStatus.last().start_time().seconds() != 0 || readinessStatus.last().start_time().nanos() != 0
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "invalid_command";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadReadinessFailReasonAndFeedbackStartTime");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessFailReasonAndFeedbackStartTime) {
    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
                : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().last().state() == API::EContainerState_SYSTEM_FAILURE
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_FAILED_TO_START_LIVENESS
                ;
            };

            TickTree(Tree_, 16, breakHook);
            const auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            const auto livenessStatus = status.liveness_status().container_status();

            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EWorkloadState_FAILED_TO_START_LIVENESS
                , API::EWorkloadState_Name(status.state())
            );
            UNIT_ASSERT_EQUAL_C(
                livenessStatus.last().state()
                , API::EContainerState_SYSTEM_FAILURE
                , API::EContainerState_Name(livenessStatus.last().state())
            );
            UNIT_ASSERT_STRING_CONTAINS_C(
                livenessStatus.last().fail_reason()
                , "InvalidCommand:(No such file or directory: cannot exec invalid_command)"
                , livenessStatus.last().fail_reason()
            );
            // Check feedback start_time on PortoStart fail
            UNIT_ASSERT_C(
                livenessStatus.last().start_time().seconds() != 0 || livenessStatus.last().start_time().nanos() != 0
                , TConsoleRenderer(false).Render(Tree_)
            );

            CheckStartContainerState(EPortoContainerState::Running);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "invalid_command";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadLivenessFailReasonAndFeedbackStartTime");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessPortoKillStartFailOnHookFail) {
    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
                : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();

            const TPortoContainerName workloadStartName = GetFullWorkloadContainerName("start");
            SafePorto_->Create(workloadStartName).Success();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_DEACTIVATED_BY_LIVENESS;
            };

            TickTree(Tree_, 12, breakHook);

            auto workloadStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto status = workloadStatus.liveness_status().container_status();

            TString expectedStderr =  "bash: line 0: cd: this_dir_does_not_exist: No such file or directory\n";
            UNIT_ASSERT_EQUAL_C(workloadStatus.state(), API::EWorkloadState_DEACTIVATED_BY_LIVENESS, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(status.last().stderr_tail(), expectedStderr, status.last().stderr_tail());
            UNIT_ASSERT_EQUAL_C(status.last().stdout_tail(), "liveness\n", status.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EContainerState_WAITING_RESTART
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EContainerState_EXITED
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_C(status.positive_return_code_counter() > 0, status.positive_return_code_counter());

            CheckStartContainerState(EPortoContainerState::Stopped);

            UNIT_ASSERT_STRING_CONTAINS_C(workloadStatus.start().current().fail_reason(), "InvalidState", workloadStatus.start().current().fail_reason());
            UNIT_ASSERT_EQUAL_C(workloadStatus.start().current().state(), API::EContainerState_SYSTEM_FAILURE, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_C(workloadStatus.start().system_failure_counter() > 0, NProtobufJson::Proto2Json(status));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_CMD"] = "bash -c 'echo liveness; cd this_dir_does_not_exist'";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadLivenessPortoKillStartFailOnHookFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessPortoKillHookFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client)
            : TWrapperPortoClient(client)
            , LastKillName_(TPortoContainerName(""))
        {
        }

        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int /*sig*/) override {
            LastKillName_ = name;
            return TPortoError{EPortoError::InvalidState, TStringBuilder() << "Container " << TString(name) << " in invalid state"};
        }

        TPortoContainerName GetLastKillName() const {
            return LastKillName_;
        }

    private:
        TPortoContainerName LastKillName_;
    };

    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
                : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().system_failure_counter() > 0;
            };

            TickTree(Tree_, 20, breakHook);

            auto hookStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                hookStatus.last().state()
                , API::EContainerState_SYSTEM_FAILURE
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_STRING_CONTAINS_C(hookStatus.last().fail_reason(), "InvalidState", hookStatus.last().fail_reason());
            UNIT_ASSERT_C(hookStatus.system_failure_counter() > 0, NProtobufJson::Proto2Json(hookStatus));

            UNIT_ASSERT_EQUAL_C(GetFullWorkloadContainerName("readiness"), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["HOOK_CMD"] = "sleep 20";
            specificReplace["HOOK_MAX_EXECUTION_TIME"] = "1000";

            return specificReplace;
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto);
        }
    };

    TTest test("TestWorkloadReadinessPortoKillHookFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessPortoKillHookFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client)
            : TWrapperPortoClient(client)
            , LastKillName_(TPortoContainerName(""))
        {
        }

        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int /*sig*/) override {
            LastKillName_ = name;
            return TPortoError{EPortoError::InvalidState, TStringBuilder() << "Container " << TString(name) << " in invalid state"};
        }

        TPortoContainerName GetLastKillName() const {
            return LastKillName_;
        }

    private:
        TPortoContainerName LastKillName_;
    };

    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
                : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().system_failure_counter() > 0;
            };

            TickTree(Tree_, 20, breakHook);

            auto hookStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                hookStatus.last().state()
                , API::EContainerState_SYSTEM_FAILURE
                , NProtobufJson::Proto2Json(hookStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_STRING_CONTAINS_C(hookStatus.last().fail_reason(), "InvalidState", hookStatus.last().fail_reason());
            UNIT_ASSERT_C(hookStatus.system_failure_counter() > 0, NProtobufJson::Proto2Json(hookStatus));

            UNIT_ASSERT_EQUAL_C(GetFullWorkloadContainerName("liveness"), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["HOOK_CMD"] = "sleep 20";
            specificReplace["HOOK_MAX_EXECUTION_TIME"] = "1000";

            return specificReplace;
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto);
        }
    };

    TTest test("TestWorkloadLivenessPortoKillHookFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadReadinessCreateContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)

        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            if (name == ContainerToFail_) {
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestReadinessCanon {
    public:
        TTest(const TString& testName)
            : ITestReadinessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status().last().state() == API::EContainerState_SYSTEM_FAILURE;
            };

            TickTree(Tree_, 16, breakHook);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().container_status();
            UNIT_ASSERT_EQUAL_C(status.last().state(), API::EContainerState_SYSTEM_FAILURE, API::EContainerState_Name(status.last().state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString containerName = GetFullWorkloadContainerName("readiness");

            UNIT_ASSERT_STRING_CONTAINS(status.last().fail_reason(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_C(status.system_failure_counter() > 0, status.system_failure_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(containerName), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls() > 0, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, GetFullWorkloadContainerName("readiness"));
        }
    };

    TTest test("TestWorkloadReadinessCreateContainerFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadLivenessCreateContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , CreateCalls_(0)

        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            ++CreateCalls_;
            if (name == ContainerToFail_) {
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetCreateCalls() {
            return CreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 CreateCalls_;
    };

    class TTest : public ITestLivenessCanon {
    public:
        TTest(const TString& testName)
            : ITestLivenessCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status().last().state() == API::EContainerState_SYSTEM_FAILURE;
            };

            TickTree(Tree_, 16, breakHook);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).liveness_status().container_status();
            UNIT_ASSERT_EQUAL_C(status.last().state(), API::EContainerState_SYSTEM_FAILURE, API::EContainerState_Name(status.last().state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString containerName = GetFullWorkloadContainerName("liveness");

            UNIT_ASSERT_STRING_CONTAINS(status.last().fail_reason(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_C(status.system_failure_counter() > 0, status.system_failure_counter());
            UNIT_ASSERT_EQUAL_C(TPortoContainerName::NoEscape(containerName), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName());
            UNIT_ASSERT_C(((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls() > 0, ((TTestPortoClient*)TreePorto_.Get())->GetCreateCalls());

            CheckStartContainerState(EPortoContainerState::Running);
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, GetFullWorkloadContainerName("liveness"));
        }
    };

    TTest test("TestWorkloadLivenessCreateContainerFail");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
