#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/behaviour/trees/workload/base/test/workload_test_canon.h>

#include <infra/libs/http_service/test_common.h>
#include <infra/libs/service_iface/fake_routers.h>

#include <library/cpp/protobuf/json/proto2json.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestTreeNetworkStatusHookCanon: public ITestWorkloadCanon {
public:
    ITestTreeNetworkStatusHookCanon(const TString& testName)
        : ITestWorkloadCanon(testName, "TreeNetworkStatusHook", "TreeNetworkStatusHook")
        , Port_(NTesting::GetFreePort())
    {
    }

protected:
    TMap<TString, TString> GetReplace() const final {
        TMap<TString, TString> replace;

        replace["WORKLOAD_ID"] = WorkloadId_;

        replace["TREE_HASH"] = "tree_hash";

        replace["NETWORK_HOOK_TYPE"] = "readiness";
        replace["HOOK_BACKEND"] = "tcp";

        replace["ANY_RESPONSE"] =  "false";
        replace["FAILURE_THRESHOLD"] = "1";
        replace["INITIAL_DELAY"] = "0";
        replace["MAX_EXECUTION_TIME"] = "200";
        replace["MAX_RESTART_PERIOD"] = "100";
        replace["MIN_RESTART_PERIOD"] = "100";
        replace["PATH"] =  "";
        replace["PORT"] = ToString(Port_);
        replace["RESPONSE"] =  "";
        replace["RESTART_PERIOD_BACKOFF"] = "1";
        replace["RESTART_PERIOD_SCALE"] = "0";
        replace["STATE_WAITING_INIT"] = "ETcpCheckState_WAITING_INIT";
        replace["STATE_WAITING_RESTART"] = "ETcpCheckState_WAITING_RESTART";
        replace["SUCCESS_THRESHOLD"] = "1";

        replace["DESTROY_CONTAINER_WHEN_HOOK_FAILS"] = "";
        replace["FEEDBACK_WHEN_HOOK_FAILED"] = "EWorkloadState_READINESS_FAILED";
        replace["FEEDBACK_WHEN_HOOK_FINISHED_OK"] = "EWorkloadState_ACTIVE";
        replace["START_CONTAINER"] = "";

        return replace;
    }

protected:
    NTesting::TPortHolder Port_;
};

Y_UNIT_TEST_SUITE(TreeNetworkStatusHookTestSuite) {

Y_UNIT_TEST(TestNetworkHookTcpSuccess) {
    class TTest : public ITestTreeNetworkStatusHookCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeNetworkStatusHookCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookFailed = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_READINESS_FAILED;
            };
            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            TickTree(Tree_, 16, breakHookFailed);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_READINESS_FAILED, WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), TConsoleRenderer(false).Render(Tree_));

            TInet6StreamSocket listenSocket = CreateBindedSocket(Port_);
            UNIT_ASSERT_EQUAL_C(listenSocket.Listen(1), 0, strerror(errno));

            TickTree(Tree_, 16, breakHookActive);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestNetworkHookTcpSuccess");
    test.DoTest();
}

Y_UNIT_TEST(TestNetworkHookHttpSuccess) {
    class TTest : public ITestTreeNetworkStatusHookCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeNetworkStatusHookCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookFailed = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_READINESS_FAILED;
            };
            auto breakHookActive = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_ACTIVE;
            };

            TickTree(Tree_, 16, breakHookFailed);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_READINESS_FAILED, WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), TConsoleRenderer(false).Render(Tree_));

            THttpServiceConfig config = NTestCommon::GenerateHttpServiceConfig(Port_);
            THttpService httpService = CreateAndStartHttpService(config, MakeSimpleShared<TFakeSuccessRouter>());

            TickTree(Tree_, 16, breakHookActive);
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HOOK_BACKEND"] = "http";
            specificReplace["PATH"] =  "/mock_path";
            specificReplace["RESPONSE"] =  "Data";
            specificReplace["STATE_WAITING_INIT"] = "EHttpGetState_WAITING_INIT";
            specificReplace["STATE_WAITING_RESTART"] = "EHttpGetState_WAITING_RESTART";
            return specificReplace;
        }
    };

    TTest test("TestNetworkHookHttpSuccess");
    test.DoTest();
}

Y_UNIT_TEST(TestNetworkHookSuccessThreshold) {
    class TTest : public ITestTreeNetworkStatusHookCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeNetworkStatusHookCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookTwoTries = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().tcp_check_status().time_limit().consecutive_successes_counter() == 2;
            };
            auto breakHookThreeTries = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().tcp_check_status().time_limit().consecutive_successes_counter() >= 3;
            };

            TInet6StreamSocket listenSocket = CreateBindedSocket(Port_);
            // Listen(2) is enough for this test
            // However, we use Listen(10) because we want to reduce flap probability
            UNIT_ASSERT_EQUAL_C(listenSocket.Listen(10), 0, strerror(errno));

            TickTree(Tree_, 16, breakHookTwoTries);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().tcp_check_status().time_limit().consecutive_successes_counter() == 2, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_UNEQUAL_C(API::EWorkloadState_ACTIVE, status.state(), TConsoleRenderer(false).Render(Tree_));

            TickTree(Tree_, 16, breakHookThreeTries);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().tcp_check_status().time_limit().consecutive_successes_counter() >= 3, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_ACTIVE, status.state(), TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["SUCCESS_THRESHOLD"] = "3";
            return specificReplace;
        }
    };

    TTest test("TestNetworkHookSuccessThreshold");
    test.DoTest();
}

Y_UNIT_TEST(TestNetworkHookFailureThreshold) {
    class TTest : public ITestTreeNetworkStatusHookCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeNetworkStatusHookCanon(testName)
        {
        }

    protected:
        void Test() override {
            auto breakHookTwoTries = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().tcp_check_status().time_limit().consecutive_failures_counter() == 2;
            };
            auto breakHookThreeTries = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).readiness_status().tcp_check_status().time_limit().consecutive_failures_counter() >= 3;
            };

            TickTree(Tree_, 16, breakHookTwoTries);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().tcp_check_status().time_limit().consecutive_failures_counter() == 2, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_UNEQUAL_C(API::EWorkloadState_READINESS_FAILED, status.state(), TConsoleRenderer(false).Render(Tree_));

            TickTree(Tree_, 16, breakHookThreeTries);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_C(status.readiness_status().tcp_check_status().time_limit().consecutive_failures_counter() >= 3, NProtobufJson::Proto2Json(status));
            UNIT_ASSERT_EQUAL_C(API::EWorkloadState_READINESS_FAILED, status.state(), TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["FAILURE_THRESHOLD"] = "3";
            return specificReplace;
        }
    };

    TTest test("TestNetworkHookFailureThreshold");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
