#include "tree_workload_stop_test_canon.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/private_util.h>
#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/support_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestTreeWorkloadContainerStopCanon: public ITestTreeWorkloadStopCanon {
public:
    ITestTreeWorkloadContainerStopCanon(const TString& testName)
        : ITestTreeWorkloadStopCanon(testName, "TreeWorkloadContainerStop")
    {
    }
};

Y_UNIT_TEST_SUITE(TreeWorkloadContainerStopTestSuite) {

Y_UNIT_TEST(TestWorkloadDeactivating) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookDeactivating = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_DEACTIVATING;
            };

            TickTree(Tree_, 32, breakHookDeactivating);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_DEACTIVATING, TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWorkloadDeactivating");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemoved) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            SafePorto_->Kill(GetFullWorkloadContainerName("start"), 9).Success();
            SafePorto_->WaitContainers({GetFullWorkloadContainerName("start")}).Success();

            TickTree(Tree_, 32, breakHookRemoved);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(status.last().stdout_tail(), "stop\n", status.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(status.last().stderr_tail(), "stop\n", status.last().stderr_tail());
            UNIT_ASSERT_EQUAL_C(status.last().fail_reason(), "", status.last().fail_reason());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop\"";
            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            auto workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c \"echo start; >&2 echo start; sleep 1000\"").Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
        }
    };

    TTest test("TestWorkloadRemoved");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedStopped) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            SafePorto_->Stop(GetFullWorkloadContainerName("start"), TDuration::MilliSeconds(1000)).Success();
            TickTree(Tree_, 32, breakHookRemoved);

            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().last().state() == API::EContainerState_EXITED, API::EContainerState_EXITED, TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().last().stdout_tail(), "stop\n", WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().last().stderr_tail(), "stop\n", WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().last().stderr_tail());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop\"";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedStopped");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedIncorrectPrivate) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 2, breakHookRemoved);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
        }

        void PrepareWorkloadStartContainer() final {
            auto workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c \"echo start; sleep 1000\"").Success();
            SafePorto_->Start(workloadStartName).Success();
        }
    };

    TTest test("TestWorkloadRemovedIncorrectPrivate");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedPaused) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().current().state() == API::EContainerState_WAITING_RESTART;
            };

            SafePorto_->Pause(GetFullWorkloadContainerName("start")).Success();

            TickTree(Tree_, 32, breakHookWaitingRestart);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_DEACTIVATING, TConsoleRenderer(false).Render(Tree_));

            auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(stopStatus.current().state(), API::EContainerState_WAITING_RESTART, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(stopStatus.last().state(), API::EContainerState_EXITED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "stop container in dead state while start container in paused state", stopStatus.last().fail_reason());

            UNIT_ASSERT_EQUAL_C(stopStatus.last().stdout_tail(), "stop\n", stopStatus.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(stopStatus.last().stderr_tail(), "stop\nbash: line 0: cd: this_dir_does_not_exist: No such file or directory\n", stopStatus.last().stderr_tail());

            UNIT_ASSERT_EQUAL(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_UNKNOWN);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop; cd this_dir_does_not_exist\"";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedPaused");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedMeta) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().current().state() == API::EContainerState_WAITING_RESTART;
            };

            TickTree(Tree_, 32, breakHookWaitingRestart);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_DEACTIVATING, TConsoleRenderer(false).Render(Tree_));

            auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(stopStatus.current().state(), API::EContainerState_WAITING_RESTART, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(stopStatus.last().state(), API::EContainerState_EXITED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "stop container in dead state while start container in meta state", stopStatus.last().fail_reason());

            UNIT_ASSERT_EQUAL_C(stopStatus.last().stdout_tail(), "stop\n", stopStatus.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(stopStatus.last().stderr_tail(), "stop\nbash: line 0: cd: this_dir_does_not_exist: No such file or directory\n", stopStatus.last().stderr_tail());

            UNIT_ASSERT_EQUAL(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_UNKNOWN);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop; cd this_dir_does_not_exist\"";
            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            const auto workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
            UNIT_ASSERT_EQUAL_C(ToString(EPortoContainerState::Meta), SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success(), "failed to start 'start' container");
        }
    };

    TTest test("TestWorkloadRemovedMeta");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedRunning) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().current().state() == API::EContainerState_WAITING_RESTART;
            };

            TickTree(Tree_, 32, breakHookWaitingRestart);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_DEACTIVATING, TConsoleRenderer(false).Render(Tree_));

            auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(stopStatus.current().state(), API::EContainerState_WAITING_RESTART, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(stopStatus.last().state(), API::EContainerState_EXITED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "stop container in dead state while start container in running state", stopStatus.last().fail_reason());

            UNIT_ASSERT_EQUAL_C(stopStatus.last().stdout_tail(), "stop\n", stopStatus.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(stopStatus.last().stderr_tail(), "stop\nbash: line 0: cd: this_dir_does_not_exist: No such file or directory\n", stopStatus.last().stderr_tail());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop; cd this_dir_does_not_exist\"";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedRunning");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedFailToStop) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookRemoved);
            const auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));

            const auto stopStatus = status.stop_status().container_status();
            UNIT_ASSERT_STRING_CONTAINS_C(
                stopStatus.last().fail_reason()
                , "InvalidCommand:(No such file or directory: cannot exec some123)"
                , stopStatus.last().fail_reason()
            );
            UNIT_ASSERT_EQUAL_C(
                stopStatus.last().state()
                , API::EContainerState_SYSTEM_FAILURE
                , API::EContainerState_Name(stopStatus.last().state())
            );
            // Check feedback start_time on PortoStart fail
            UNIT_ASSERT_C(
                stopStatus.last().start_time().seconds() != 0 || stopStatus.last().start_time().nanos() != 0
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_CMD"] = "some123 fail command";
            specificReplace["STOP_MAX_TRIES"] = "10";
            specificReplace["STOP_MIN_RESTART_PERIOD"] = "0";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedFailToStop");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStopInitialDelayAndBackOff) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookFirstExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().positive_return_code_counter() == 1;
            };
            auto breakHookThirdExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().positive_return_code_counter() == 3
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().current().state() == API::EContainerState_WAITING_RESTART
                ;
            };

            TickTree(Tree_, 32, breakHookFirstExit);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto stopStatus = status.stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                1
                , stopStatus.positive_return_code_counter()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant startTs = NSupport::ToInstant(status.start().last().start_time());
            TInstant firstTs = NSupport::ToInstant(stopStatus.last().start_time());
            UNIT_ASSERT_GE_C(firstTs - startTs, TDuration::MilliSeconds(INITIAL_DELAY_MS), firstTs << " " << startTs);
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_EXITED
                , stopStatus.last().state()
                , NProtobufJson::Proto2Json(stopStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TickTree(Tree_, 32, breakHookThirdExit);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            stopStatus = status.stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(
                3
                , stopStatus.positive_return_code_counter()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant thirdTs = NSupport::ToInstant(stopStatus.last().start_time());
            UNIT_ASSERT_GE_C(thirdTs - firstTs, TDuration::MilliSeconds(BACKOFF_MS), thirdTs << " " << firstTs);
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_WAITING_RESTART
                , stopStatus.current().state()
                , NProtobufJson::Proto2Json(stopStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EContainerState_EXITED
                , stopStatus.last().state()
                , NProtobufJson::Proto2Json(stopStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_INITIAL_DELAY"] = ToString(INITIAL_DELAY_MS);
            specificReplace["STOP_MIN_RESTART_PERIOD"] = "0";
            specificReplace["STOP_RESTART_PERIOD_BACKOFF"] = ToString(BACKOFF_MS);
            specificReplace["STOP_RESTART_PERIOD_SCALE"] = "1";
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop; cd this_dir_does_not_exist\"";
            return specificReplace;
        }

    private:
        const size_t INITIAL_DELAY_MS = 2000;
        const size_t BACKOFF_MS = 2000;
    };

    TTest test("TestWorkloadStopInitialDelayAndBackOff");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStopTimeout) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookExited = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().killed_externally_counter() > 0;
            };

            TickTree(Tree_, 12, breakHookExited);

            auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(status.state(), API::EWorkloadState_DEACTIVATING, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_C(
                status.stop_status().container_status().killed_externally_counter() == 1
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_C(
                status.stop_status().container_status().timeout_counter() == 1
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(API::EContainerState_TIMEOUT, status.stop_status().container_status().last().state(), NProtobufJson::Proto2Json(status));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_CMD"] = "sleep 20";
            specificReplace["STOP_MAX_EXECUTION_TIME"] = "1000";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadStopTimeout");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedRunningAfterTries) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookRemoved);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_KILLED_EXTERNALLY, TConsoleRenderer(false).Render(Tree_));

            auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "stop container in dead state while start container in running state", stopStatus.last().fail_reason());

            UNIT_ASSERT_EQUAL_C(stopStatus.last().stdout_tail(), "stop\n", stopStatus.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(stopStatus.last().stderr_tail(), "stop\nbash: line 0: cd: this_dir_does_not_exist: No such file or directory\n", stopStatus.last().stderr_tail());

            UNIT_ASSERT_EQUAL_C(stopStatus.zero_return_code_counter(), 0, stopStatus.zero_return_code_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.positive_return_code_counter(), 3, stopStatus.positive_return_code_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_MIN_RESTART_PERIOD"] = "1";
            specificReplace["STOP_MAX_RESTART_PERIOD"] = "1";
            specificReplace["STOP_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["STOP_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["STOP_MAX_TRIES"] = "3";
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop; cd this_dir_does_not_exist\"";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedRunningAfterTries");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedRunningAfterStopZeroExitCode) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookRemoved);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_KILLED_EXTERNALLY, TConsoleRenderer(false).Render(Tree_));

            auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "stop container in dead state while start container in running state", stopStatus.last().fail_reason());

            UNIT_ASSERT_EQUAL_C(stopStatus.last().stdout_tail(), "stop\n", stopStatus.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(stopStatus.last().stderr_tail(), "stop\n", stopStatus.last().stderr_tail());

            UNIT_ASSERT_EQUAL_C(stopStatus.zero_return_code_counter(), 1, stopStatus.zero_return_code_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.positive_return_code_counter(), 0, stopStatus.positive_return_code_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_MIN_RESTART_PERIOD"] = "1";
            specificReplace["STOP_MAX_RESTART_PERIOD"] = "1";
            specificReplace["STOP_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["STOP_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["STOP_MAX_TRIES"] = "10";
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop\"";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedRunningAfterStopZeroExitCode");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedNoStart) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookRemoved);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
        }
    };

    TTest test("TestWorkloadRemovedNoStart");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadPortoKillStartFail) {
    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            SafePorto_->Pause(GetFullWorkloadContainerName("start")).Success();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookFail = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state() == API::EContainerState_SYSTEM_FAILURE;
            };

            TickTree(Tree_, 32, breakHookFail);

            auto startStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start();
            UNIT_ASSERT_EQUAL_C(startStatus.last().state(), API::EContainerState_SYSTEM_FAILURE, NProtobufJson::Proto2Json(startStatus) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(startStatus.last().fail_reason(), "InvalidState", startStatus.last().fail_reason());
            UNIT_ASSERT_C(startStatus.system_failure_counter() > 0, NProtobufJson::Proto2Json(startStatus));

            auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "stop container in dead state while start container in paused state", stopStatus.last().fail_reason());

            UNIT_ASSERT_EQUAL_C(stopStatus.last().stdout_tail(), "stop\n", stopStatus.last().stdout_tail());
            UNIT_ASSERT_EQUAL_C(stopStatus.last().stderr_tail(), "stop\nbash: line 0: cd: this_dir_does_not_exist: No such file or directory\n", stopStatus.last().stderr_tail());

            UNIT_ASSERT_EQUAL_C(stopStatus.zero_return_code_counter(), 0, stopStatus.zero_return_code_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.positive_return_code_counter(), 3, stopStatus.positive_return_code_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_MIN_RESTART_PERIOD"] = "1";
            specificReplace["STOP_MAX_RESTART_PERIOD"] = "1";
            specificReplace["STOP_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["STOP_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["STOP_MAX_TRIES"] = "3";
            specificReplace["STOP_CMD"] = "bash -c \"echo stop; >&2 echo stop; cd this_dir_does_not_exist\"";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadPortoKillStartFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadPortoKillStopFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client)
            : TWrapperPortoClient(client)
            , LastKillName_(TPortoContainerName(""))
        {
        }

        TExpected<void, TPortoError> Kill(const TPortoContainerName& name, int /*sig*/) override {
            LastKillName_ = name;
            return TPortoError{EPortoError::InvalidState, TStringBuilder() << "Container " << TString(name) << " in invalid state"};
        }

        TPortoContainerName GetLastKillName() const {
            return LastKillName_;
        }

    private:
        TPortoContainerName LastKillName_;
    };

    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookFail = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status().system_failure_counter() > 0;
            };

            TickTree(Tree_, 32, breakHookFail);

            auto stopStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_C(stopStatus.last().state() == API::EContainerState_SYSTEM_FAILURE || stopStatus.last().state() == API::EContainerState_RUNNING, NProtobufJson::Proto2Json(stopStatus) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(stopStatus.last().fail_reason(), "InvalidState", stopStatus.last().fail_reason());
            UNIT_ASSERT_C(stopStatus.system_failure_counter() > 0, NProtobufJson::Proto2Json(stopStatus));

            UNIT_ASSERT_EQUAL_C(GetFullWorkloadContainerName("stop"), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName(), ((TTestPortoClient*)TreePorto_.Get())->GetLastKillName());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["STOP_MAX_EXECUTION_TIME"] = "1000";

            return specificReplace;
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto);
        }
    };

    TTest test("TestWorkloadPortoKillStopFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStopCreateContainerFail) {
    class TTestPortoClient: public TWrapperPortoClient {
    public:
        TTestPortoClient(TPortoClientPtr client, const TPortoContainerName& containerToFail)
            : TWrapperPortoClient(client)
            , ContainerToFail_(containerToFail)
            , LastCreateName_(TPortoContainerName(""))
            , FailCreateCalls_(0)
        {
        }

        TExpected<void, TPortoError> CreateRecursive(const TPortoContainerName& name) override {
            LastCreateName_ = name;
            if (name == ContainerToFail_) {
                ++FailCreateCalls_;
                return TPortoError{EPortoError::Unknown, TStringBuilder() << "Can't create " << TString(name) << " container"};
            } else {
                return TWrapperPortoClient::CreateRecursive(name);
            }
        }

        TPortoContainerName GetLastCreateName() const {
            return LastCreateName_;
        }

        ui32 GetFailCreateCalls() {
            return FailCreateCalls_;
        }

    private:
        TPortoContainerName ContainerToFail_;
        TPortoContainerName LastCreateName_;
        ui32 FailCreateCalls_;
    };

    class TTest : public ITestTreeWorkloadContainerStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadContainerStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHook = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 24, breakHook);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().container_status();
            UNIT_ASSERT_EQUAL_C(status.last().state(), API::EContainerState_SYSTEM_FAILURE, API::EContainerState_Name(status.last().state()) << Endl << TConsoleRenderer(false).RenderAll(Tree_));

            const TString containerName = GetFullWorkloadContainerName("stop");

            UNIT_ASSERT_STRING_CONTAINS(status.last().fail_reason(), TStringBuilder() << "Can't create " << containerName << " container");
            UNIT_ASSERT_EQUAL_C(status.system_failure_counter(), 10, status.system_failure_counter());
            UNIT_ASSERT_EQUAL_C(
                WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state()
                , API::EWorkloadState_REMOVED
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                TPortoContainerName::NoEscape(containerName)
                , ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName()
                , ((TTestPortoClient*)TreePorto_.Get())->GetLastCreateName()
            );
            UNIT_ASSERT_EQUAL_C(((TTestPortoClient*)TreePorto_.Get())->GetFailCreateCalls(), 10, ((TTestPortoClient*)TreePorto_.Get())->GetFailCreateCalls());
        }

        TPortoClientPtr GetSpecificPorto(TPortoClientPtr porto) const override {
            return new TTestPortoClient(porto, GetFullWorkloadContainerName("stop"));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_MAX_TRIES"] = "10";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadStopCreateContainerFail");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
