#include "tree_workload_stop_test_canon.h"

#include <infra/libs/http_service/service.h>
#include <infra/libs/http_service/test_common.h>
#include <infra/libs/service_iface/fake_routers.h>
#include <infra/pod_agent/libs/behaviour/bt/nodes/base/private_util.h>
#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/support_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/protobuf/json/proto2json.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestTreeWorkloadHttpStopCanon: public ITestTreeWorkloadStopCanon {
public:
    ITestTreeWorkloadHttpStopCanon(const TString& testName)
        : ITestTreeWorkloadStopCanon(testName, "TreeWorkloadHttpStop")
        , Port_(NTesting::GetFreePort())
        , HttpServiceConfig_(NTestCommon::GenerateHttpServiceConfig(Port_))
    {
    }

protected:
    NTesting::TPortHolder Port_;
    THttpServiceConfig HttpServiceConfig_;
};

Y_UNIT_TEST_SUITE(TreeWorkloadHttpStopTestSuite) {

Y_UNIT_TEST(TestWorkloadRemoved) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeSuccessRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            SafePorto_->Kill(GetFullWorkloadContainerName("start"), 9).Success();
            SafePorto_->WaitContainers({GetFullWorkloadContainerName("start")}).Success();

            TickTree(Tree_, 32, breakHookRemoved);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(status.last().fail_reason(), "", status.last().fail_reason());
            UNIT_ASSERT_EQUAL_C(status.success_counter(), 0, status.success_counter());
            UNIT_ASSERT_EQUAL_C(status.error_counter(), 0, status.error_counter());
            UNIT_ASSERT_EQUAL_C(status.timeout_counter(), 0, status.timeout_counter());
            UNIT_ASSERT_EQUAL_C(status.wrong_answer_counter(), 0, status.wrong_answer_counter());

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 0, router->GetRequestsCount());

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";
            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemoved");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadFailToStopWithBadAnswer) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeSuccessRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART;
            };

            TickTree(Tree_, 32, breakHookWaitingRestart);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status();

            UNIT_ASSERT_EQUAL_C(
                status.current().state()
                , API::EHttpGetState_WAITING_RESTART
                , API::EHttpGetState_Name(status.current().state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.last().state()
                , API::EHttpGetState_WRONG_ANSWER
                , API::EHttpGetState_Name(status.last().state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            UNIT_ASSERT_EQUAL_C(status.wrong_answer_counter(), 1, status.wrong_answer_counter());
            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;

            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["HTTP_STOP_RESPONSE"] = "OtherData";
            specificReplace["HTTP_STOP_ANY_RESPONSE"] = "false";
            specificReplace["STOP_HOOK_TYPE"] = "http";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadFailToStopWithBadAnswer");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedStopped) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:

        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeSuccessRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            SafePorto_->Stop(GetFullWorkloadContainerName("start"), TDuration::MilliSeconds(1000)).Success();
            TickTree(Tree_, 32, breakHookRemoved);

            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));

            auto successCount = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().success_counter();
            auto errorCount = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().error_counter();
            auto timeoutCount = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().timeout_counter();
            auto wrongAnswerCount = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().wrong_answer_counter();
            UNIT_ASSERT_EQUAL_C(successCount, 0, successCount);
            UNIT_ASSERT_EQUAL_C(errorCount, 0, errorCount);
            UNIT_ASSERT_EQUAL_C(timeoutCount, 0, timeoutCount);
            UNIT_ASSERT_EQUAL_C(wrongAnswerCount, 0, wrongAnswerCount);

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 0, router->GetRequestsCount());

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedStopped");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedPaused) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART;
            };

            SafePorto_->Pause(GetFullWorkloadContainerName("start")).Success();
            TickTree(Tree_, 10, breakHookWaitingRestart);

            UNIT_ASSERT_EQUAL(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_UNKNOWN);

            auto hookState = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state();
            UNIT_ASSERT_EQUAL_C(hookState, API::EHttpGetState_WAITING_RESTART, TConsoleRenderer(false).Render(Tree_));

            auto hookStateLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().state();
            UNIT_ASSERT_EQUAL_C(hookStateLast, API::EHttpGetState_FAILURE, TConsoleRenderer(false).Render(Tree_));


            auto failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReason, "", failReason);

            auto failReasonLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReasonLast, "http stop hook finished while start container in paused state", failReasonLast);

            auto errorCounter = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().error_counter();
            UNIT_ASSERT_EQUAL_C(errorCounter, 1, errorCounter);

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedPaused");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemoveMeta) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART;
            };

            TickTree(Tree_, 10, breakHookWaitingRestart);

            UNIT_ASSERT_EQUAL(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_UNKNOWN);

            auto hookState = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state();
            UNIT_ASSERT_EQUAL_C(hookState, API::EHttpGetState_WAITING_RESTART, TConsoleRenderer(false).Render(Tree_));

            auto hookStateLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().state();
            UNIT_ASSERT_EQUAL_C(hookStateLast, API::EHttpGetState_FAILURE, TConsoleRenderer(false).Render(Tree_));

            auto failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReason, "", failReason);

            auto failReasonLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReasonLast, "http stop hook finished while start container in meta state", failReasonLast);


            auto errorCounter = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().error_counter();
            UNIT_ASSERT_EQUAL_C(errorCounter, 1, errorCounter);

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";

            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            const auto workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
            UNIT_ASSERT_EQUAL_C(ToString(EPortoContainerState::Meta), SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success(), "failed to start 'start' container");
        }
    };

    TTest test("TestWorkloadRemovedMeta");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedRunning) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART;
            };

            TickTree(Tree_, 10, breakHookWaitingRestart);

            auto hookState = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state();
            UNIT_ASSERT_EQUAL_C(hookState, API::EHttpGetState_WAITING_RESTART, TConsoleRenderer(false).Render(Tree_));

            auto hookStateLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().state();
            UNIT_ASSERT_EQUAL_C(hookStateLast, API::EHttpGetState_FAILURE, TConsoleRenderer(false).Render(Tree_));

            auto failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReason, "", failReason);

            auto failReasonLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReasonLast, "http stop hook finished while start container in running state", failReasonLast);


            auto errorCounter = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().error_counter();
            UNIT_ASSERT_EQUAL_C(errorCounter, 1, errorCounter);

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedRunning");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedFailToStop) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART;
            };

            TickTree(Tree_, 10, breakHookWaitingRestart);

            auto hookState = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state();
            UNIT_ASSERT_EQUAL_C(API::EHttpGetState_WAITING_RESTART, hookState, TConsoleRenderer(false).Render(Tree_));
            auto hookStateLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().state();
            UNIT_ASSERT_EQUAL_C(API::EHttpGetState_FAILURE, hookStateLast, TConsoleRenderer(false).Render(Tree_));

            TString failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().inner_fail_reason();
            UNIT_ASSERT_C(failReason == "", failReason);
            TString failReasonLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().inner_fail_reason();
            UNIT_ASSERT_STRING_CONTAINS(failReasonLast, "404");

            auto errorCounter = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().error_counter();
            UNIT_ASSERT_EQUAL_C(errorCounter, 1, errorCounter);

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedFailToStop");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStopInitialDelayAndBackOff) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookSecondExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().time_limit().consecutive_failures_counter() == 2
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART
                ;
            };
            auto breakHookThirdExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().time_limit().consecutive_failures_counter() == 3
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().state() == API::EHttpGetState_WAITING_RESTART
                ;
            };

            TickTree(Tree_, 32, breakHookSecondExit);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto stopStatus = status.stop_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(
                2
                , stopStatus.time_limit().consecutive_failures_counter()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant startTs = NSupport::ToInstant(status.start().last().start_time());
            TInstant firstTs = NSupport::ToInstant(stopStatus.last().start_time());
            UNIT_ASSERT_GE_C(firstTs - startTs, TDuration::MilliSeconds(INITIAL_DELAY_MS), firstTs << " " << startTs);
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WAITING_RESTART
                , stopStatus.current().state()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_FAILURE
                , stopStatus.last().state()
                , NProtobufJson::Proto2Json(stopStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TickTree(Tree_, 32, breakHookThirdExit);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            stopStatus = status.stop_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(
                3
                , stopStatus.time_limit().consecutive_failures_counter()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant thirdTs = NSupport::ToInstant(stopStatus.last().start_time());
            UNIT_ASSERT_GE_C(thirdTs - firstTs, TDuration::MilliSeconds(BACKOFF_MS), thirdTs << " " << firstTs);
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_WAITING_RESTART
                , stopStatus.current().state()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EHttpGetState_FAILURE
                , stopStatus.last().state()
                , NProtobufJson::Proto2Json(stopStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 3, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[1], "/mock_path", router->GetPaths()[1]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[2], "/mock_path", router->GetPaths()[2]);
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_INITIAL_DELAY"] = ToString(INITIAL_DELAY_MS);
            specificReplace["HTTP_STOP_MIN_RESTART_PERIOD"] = "0";
            specificReplace["HTTP_STOP_RESTART_PERIOD_BACKOFF"] = ToString(BACKOFF_MS);
            specificReplace["HTTP_STOP_RESTART_PERIOD_SCALE"] = "1";
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";
            return specificReplace;
        }

    private:
        const size_t INITIAL_DELAY_MS = 2000;
        const size_t BACKOFF_MS = 2000;
    };

    TTest test("TestWorkloadStopInitialDelayAndBackOff");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStopTimeout) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeSlowRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookFail = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().time_limit().consecutive_failures_counter() > 0;
            };

            TickTree(Tree_, 4, breakHookFail);

            TString failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().current().inner_fail_reason();
            UNIT_ASSERT_C(failReason == "", failReason);

            TString failReasonLast = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().inner_fail_reason();
            TString errorMsg = TStringBuilder()
                << "stop policy failed: "
                << "HTTP request to http2://localhost:" << Port_ << "/mock_path: "
                << "Request timeout";
            UNIT_ASSERT_EQUAL_C(failReasonLast, errorMsg, failReasonLast);

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadStopTimeout");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedRunningAfterTries) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookRemoved);

            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state(), API::EContainerState_KILLED_EXTERNALLY, TConsoleRenderer(false).Render(Tree_));

            auto failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReason, "http stop hook finished while start container in running state", failReason);

            auto hookStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(hookStatus.time_limit().consecutive_successes_counter(), 0, hookStatus.time_limit().consecutive_successes_counter());
            UNIT_ASSERT_EQUAL_C(hookStatus.time_limit().consecutive_failures_counter(), 3, hookStatus.time_limit().consecutive_failures_counter());

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 3, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[1], "/mock_path", router->GetPaths()[1]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[2], "/mock_path", router->GetPaths()[2]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";
            specificReplace["STOP_MAX_TRIES"] = "3";
            specificReplace["HTTP_STOP_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["HTTP_STOP_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["HTTP_STOP_MAX_RESTART_PERIOD"] = "1";

            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            auto workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c \"echo start; >&2 echo start; sleep 5000\"").Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
            UNIT_ASSERT_EQUAL_C(ToString(EPortoContainerState::Running), SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success(), "failed to start 'start' container");
        }
    };

    TTest test("TestWorkloadRemovedRunningAfterTries");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedNoStart) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeSuccessRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 60, breakHookRemoved);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 0, router->GetRequestsCount());

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedNoStart");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadPortoKillStartFailSuccess) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeSuccessRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookKill = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().last().state() == API::EContainerState_KILLED_EXTERNALLY;
            };

            TickTree(Tree_, 60, breakHookKill);

            auto startStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start();
            UNIT_ASSERT_EQUAL_C(startStatus.last().state(), API::EContainerState_KILLED_EXTERNALLY, NProtobufJson::Proto2Json(startStatus) << Endl << TConsoleRenderer(false).Render(Tree_));

            auto hookStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(hookStatus.time_limit().consecutive_successes_counter(), 1, hookStatus.time_limit().consecutive_successes_counter());
            UNIT_ASSERT_EQUAL_C(hookStatus.time_limit().consecutive_failures_counter(), 0, hookStatus.time_limit().consecutive_failures_counter());

            // STOP_MAX_TRIES = 3, but on Success must kill immediately
            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 1, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[2]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";
            specificReplace["STOP_MAX_TRIES"] = "3";
            specificReplace["HTTP_STOP_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["HTTP_STOP_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["HTTP_STOP_MAX_RESTART_PERIOD"] = "1";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadPortoKillStartFailSuccess");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadPortoKillStartFailFail) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            SafePorto_->Pause(GetFullWorkloadContainerName("start")).Success();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            auto router = MakeSimpleShared<TFakeFailureRouter>();
            THttpService httpService(HttpServiceConfig_, router);
            httpService.Start(fakeFrame);

            auto breakHookFail = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start().current().state() == API::EContainerState_SYSTEM_FAILURE;
            };

            TickTree(Tree_, 60, breakHookFail);

            auto startStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).start();
            UNIT_ASSERT_EQUAL_C(startStatus.current().state(), API::EContainerState_SYSTEM_FAILURE, NProtobufJson::Proto2Json(startStatus) << Endl << TConsoleRenderer(false).Render(Tree_));
            UNIT_ASSERT_STRING_CONTAINS_C(startStatus.current().fail_reason(), "InvalidState", startStatus.current().fail_reason());
            UNIT_ASSERT_C(startStatus.system_failure_counter() > 0, NProtobufJson::Proto2Json(startStatus));

            auto failReason = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status().last().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReason, "http stop hook finished while start container in paused state", failReason);

            auto hookStatus = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().http_get_status();
            UNIT_ASSERT_EQUAL_C(hookStatus.time_limit().consecutive_successes_counter(), 0, hookStatus.time_limit().consecutive_successes_counter());
            UNIT_ASSERT_EQUAL_C(hookStatus.time_limit().consecutive_failures_counter(), 3, hookStatus.time_limit().consecutive_failures_counter());

            UNIT_ASSERT_EQUAL_C(router->GetRequestsCount(), 3, router->GetRequestsCount());
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[0], "/mock_path", router->GetPaths()[0]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[1], "/mock_path", router->GetPaths()[1]);
            UNIT_ASSERT_EQUAL_C(router->GetPaths()[2], "/mock_path", router->GetPaths()[2]);

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";
            specificReplace["STOP_MAX_TRIES"] = "3";
            specificReplace["HTTP_STOP_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["HTTP_STOP_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["HTTP_STOP_MAX_RESTART_PERIOD"] = "1";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadPortoKillStartFailFail");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDeactivating) {
    class TTest : public ITestTreeWorkloadHttpStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadHttpStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto fakeFrame = logger.SpawnFrame();
            THttpService httpService(HttpServiceConfig_, MakeSimpleShared<TFakeSuccessRouter>());
            httpService.Start(fakeFrame);

            auto breakHookDeactivating = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_DEACTIVATING;
            };

            TickTree(Tree_, 32, breakHookDeactivating);

            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_DEACTIVATING, TConsoleRenderer(false).Render(Tree_));

            httpService.ShutDown();
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["HTTP_STOP_PORT"] = ToString(HttpServiceConfig_.GetPort());
            specificReplace["STOP_HOOK_TYPE"] = "http";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadDeactivating");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
