#include "tree_workload_stop_test_canon.h"

#include <infra/pod_agent/libs/behaviour/bt/nodes/base/private_util.h>
#include <infra/pod_agent/libs/behaviour/bt/render/console_renderer.h>
#include <infra/pod_agent/libs/pod_agent/status_repository/support_functions.h>
#include <infra/pod_agent/libs/porto_client/porto_test_lib/wrapper_client.h>

#include <library/cpp/protobuf/json/proto2json.h>
#include <library/cpp/testing/unittest/registar.h>

namespace NInfra::NPodAgent::NTreeTest {

class ITestTreeWorkloadUnixSignalStopCanon: public ITestTreeWorkloadStopCanon {
public:
    ITestTreeWorkloadUnixSignalStopCanon(const TString& testName)
        : ITestTreeWorkloadStopCanon(testName, "TreeWorkloadUnixSignalStop")
    {
    }
};

Y_UNIT_TEST_SUITE(TreeWorkloadUnixSignalStopTestSuite) {

Y_UNIT_TEST(TestWorkloadRemovedSuccessKill) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookRemoved);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EWorkloadState_REMOVED
                , TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.start().last().state()
                , API::EContainerState_KILLED_EXTERNALLY
                , TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.start().last().return_code()
                , -SIGTERM
                , status.start().last().return_code() << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto stopStatus = status.stop_status().unix_signal_status();
            UNIT_ASSERT_EQUAL_C(stopStatus.success_counter(), 1, stopStatus.success_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.error_counter(), 0, stopStatus.error_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedSuccessKill");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedWaitMinRestartPeriodAfterLastTry) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            // Number of ticks is so big here because
            // in this test start container sleeps 5 seconds after receiving the SIGTERM,
            // and only then dies with return code 7.
            TickTree(Tree_, 200, breakHookRemoved);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EWorkloadState_REMOVED
                , TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.start().last().state()
                , API::EContainerState_EXITED
                , TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.start().last().return_code()
                , 7
                , status.start().last().return_code() << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto stopStatus = status.stop_status().unix_signal_status();
            UNIT_ASSERT_EQUAL_C(stopStatus.success_counter(), 1, stopStatus.success_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.error_counter(), 0, stopStatus.error_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";
            specificReplace["STOP_MAX_TRIES"] = "1";
            specificReplace["UNIX_SIGNAL_STOP_MIN_RESTART_PERIOD"] = "60000";
            specificReplace["UNIX_SIGNAL_STOP_MAX_RESTART_PERIOD"] = "60000";

            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            const TPortoContainerName workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c 'trap \"sleep 5; exit 7\" SIGTERM; for((;;)); do echo none > /dev/null; done'").Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
            UNIT_ASSERT_EQUAL_C(ToString(EPortoContainerState::Running), SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success(), "failed to start 'start' container");
        }
    };

    TTest test("TestWorkloadRemovedWaitMinRestartPeriodAfterLastTry");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemoved) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            SafePorto_->Kill(GetFullWorkloadContainerName("start"), 9).Success();
            SafePorto_->WaitContainers({GetFullWorkloadContainerName("start")}).Success();

            TickTree(Tree_, 32, breakHookRemoved);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto stopStatus = status.stop_status().unix_signal_status();
            UNIT_ASSERT_EQUAL_C(
                WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state()
                , API::EWorkloadState_REMOVED
                , TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "", stopStatus.last().fail_reason());
            UNIT_ASSERT_EQUAL_C(stopStatus.success_counter(), 0, stopStatus.success_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.error_counter(), 0, stopStatus.error_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemoved");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedStopped) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:

        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            SafePorto_->Stop(GetFullWorkloadContainerName("start"), TDuration::MilliSeconds(1000)).Success();
            TickTree(Tree_, 32, breakHookRemoved);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto stopStatus = status.stop_status().unix_signal_status();
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EWorkloadState_REMOVED
                , API::EWorkloadState_Name(status.state()) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(stopStatus.last().fail_reason(), "", stopStatus.last().fail_reason());
            UNIT_ASSERT_EQUAL_C(stopStatus.success_counter(), 0, stopStatus.success_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.error_counter(), 0, stopStatus.error_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedStopped");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedPaused) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().unix_signal_status().current().state() == API::EUnixSignalState_WAITING_RESTART;
            };

            SafePorto_->Pause(GetFullWorkloadContainerName("start")).Success();
            TickTree(Tree_, 10, breakHookWaitingRestart);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto stopStatus = status.stop_status().unix_signal_status();

            UNIT_ASSERT_EQUAL(status.start().last().state(), API::EContainerState_UNKNOWN);

            auto hookState = stopStatus.current().state();
            UNIT_ASSERT_EQUAL_C(
                hookState
                , API::EUnixSignalState_WAITING_RESTART
                , API::EUnixSignalState_Name(hookState) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto hookStateLast = stopStatus.last().state();
            UNIT_ASSERT_EQUAL_C(
                hookStateLast
                , API::EUnixSignalState_FAILURE
                , API::EUnixSignalState_Name(hookStateLast) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto failReason = stopStatus.current().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReason, "", failReason);

            auto failReasonLast = stopStatus.last().fail_reason();
            UNIT_ASSERT_STRING_CONTAINS(failReasonLast, "InvalidState(8):Kill");

            UNIT_ASSERT_EQUAL_C(stopStatus.success_counter(), 0, stopStatus.success_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.error_counter(), 1, stopStatus.error_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedPaused");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemoveMeta) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().unix_signal_status().current().state() == API::EUnixSignalState_WAITING_RESTART;
            };

            TickTree(Tree_, 10, breakHookWaitingRestart);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto stopStatus = status.stop_status().unix_signal_status();

            UNIT_ASSERT_EQUAL(status.start().last().state(), API::EContainerState_UNKNOWN);

            auto hookState = stopStatus.current().state();
            UNIT_ASSERT_EQUAL_C(
                hookState
                , API::EUnixSignalState_WAITING_RESTART
                , API::EUnixSignalState_Name(hookState) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto hookStateLast = stopStatus.last().state();
            UNIT_ASSERT_EQUAL_C(
                hookStateLast
                , API::EUnixSignalState_FAILURE
                , API::EUnixSignalState_Name(hookStateLast) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto failReason = stopStatus.current().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReason, "", failReason);

            auto failReasonLast = stopStatus.last().fail_reason();
            UNIT_ASSERT_STRING_CONTAINS(failReasonLast, "InvalidState(8):Kill");

            UNIT_ASSERT_EQUAL_C(stopStatus.success_counter(), 0, stopStatus.success_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.error_counter(), 1, stopStatus.error_counter());
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            const auto workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
            UNIT_ASSERT_EQUAL_C(ToString(EPortoContainerState::Meta), SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success(), "failed to start 'start' container");
        }
    };

    TTest test("TestWorkloadRemovedMeta");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedRunning) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookWaitingRestart = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().unix_signal_status().current().state() == API::EUnixSignalState_WAITING_RESTART;
            };

            TickTree(Tree_, 10, breakHookWaitingRestart);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto stopStatus = status.stop_status().unix_signal_status();

            UNIT_ASSERT_EQUAL(status.start().last().state(), API::EContainerState_UNKNOWN);

            auto hookState = stopStatus.current().state();
            UNIT_ASSERT_EQUAL_C(
                hookState
                , API::EUnixSignalState_WAITING_RESTART
                , API::EUnixSignalState_Name(hookState) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto hookStateLast = stopStatus.last().state();
            UNIT_ASSERT_EQUAL_C(
                hookStateLast
                , API::EUnixSignalState_SUCCESS
                , API::EUnixSignalState_Name(hookStateLast) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            auto failReason = stopStatus.current().fail_reason();
            UNIT_ASSERT_EQUAL_C(failReason, "", failReason);

            auto failReasonLast = stopStatus.last().fail_reason();
            UNIT_ASSERT_STRING_CONTAINS(failReasonLast, "");

            UNIT_ASSERT_EQUAL_C(stopStatus.success_counter(), 1, stopStatus.success_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.error_counter(), 0, stopStatus.error_counter());

            const TString startStdout = SafePorto_->GetStdout(GetFullWorkloadContainerName("start")).Success();
            UNIT_ASSERT_EQUAL_C(
                "You have no power here\n"
                , startStdout
                , startStdout
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            const TPortoContainerName workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c 'trap \"echo You have no power here\" SIGTERM; for((;;)); do echo none > /dev/null; done'").Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
            UNIT_ASSERT_EQUAL_C(ToString(EPortoContainerState::Running), SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success(), "failed to start 'start' container");
        }
    };

    TTest test("TestWorkloadRemovedRunning");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadStopInitialDelayAndBackOff) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookSecondExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().unix_signal_status().time_limit().consecutive_successes_counter() == 2
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().unix_signal_status().current().state() == API::EUnixSignalState_WAITING_RESTART
                ;
            };
            auto breakHookThirdExit = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().unix_signal_status().time_limit().consecutive_successes_counter() == 3
                    && WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).stop_status().unix_signal_status().current().state() == API::EUnixSignalState_WAITING_RESTART
                ;
            };

            TickTree(Tree_, 32, breakHookSecondExit);
            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            auto stopStatus = status.stop_status().unix_signal_status();
            UNIT_ASSERT_EQUAL_C(
                2
                , stopStatus.time_limit().consecutive_successes_counter()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant startTs = NSupport::ToInstant(status.start().last().start_time());
            TInstant firstTs = NSupport::ToInstant(stopStatus.last().send_time());
            UNIT_ASSERT_GE_C(firstTs - startTs, TDuration::MilliSeconds(INITIAL_DELAY_MS), firstTs << " " << startTs);
            UNIT_ASSERT_EQUAL_C(
                API::EUnixSignalState_WAITING_RESTART
                , stopStatus.current().state()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EUnixSignalState_SUCCESS
                , stopStatus.last().state()
                , NProtobufJson::Proto2Json(stopStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            TickTree(Tree_, 32, breakHookThirdExit);
            status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            stopStatus = status.stop_status().unix_signal_status();
            UNIT_ASSERT_EQUAL_C(
                3
                , stopStatus.time_limit().consecutive_successes_counter()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            TInstant thirdTs = NSupport::ToInstant(stopStatus.last().send_time());
            UNIT_ASSERT_GE_C(thirdTs - firstTs, TDuration::MilliSeconds(BACKOFF_MS), thirdTs << " " << firstTs);
            UNIT_ASSERT_EQUAL_C(
                API::EUnixSignalState_WAITING_RESTART
                , stopStatus.current().state()
                , NProtobufJson::Proto2Json(status) << Endl << TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                API::EUnixSignalState_SUCCESS
                , stopStatus.last().state()
                , NProtobufJson::Proto2Json(stopStatus) << Endl << TConsoleRenderer(false).Render(Tree_)
            );

            UNIT_ASSERT_EQUAL_C(stopStatus.success_counter(), 3, stopStatus.success_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.error_counter(), 0, stopStatus.error_counter());

            const TString startStdout = SafePorto_->GetStdout(GetFullWorkloadContainerName("start")).Success();
            const TString expectedStdoutPerSignal = "You have no power here\n";
            UNIT_ASSERT_EQUAL_C(
                expectedStdoutPerSignal + expectedStdoutPerSignal + expectedStdoutPerSignal
                , startStdout
                , startStdout
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["UNIX_SIGNAL_STOP_INITIAL_DELAY"] = ToString(INITIAL_DELAY_MS);
            specificReplace["UNIX_SIGNAL_STOP_MIN_RESTART_PERIOD"] = "0";
            specificReplace["UNIX_SIGNAL_STOP_RESTART_PERIOD_BACKOFF"] = ToString(BACKOFF_MS);
            specificReplace["UNIX_SIGNAL_STOP_RESTART_PERIOD_SCALE"] = "1";

            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            const TPortoContainerName workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c 'trap \"echo You have no power here\" SIGTERM; for((;;)); do echo none > /dev/null; done'").Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
            UNIT_ASSERT_EQUAL_C(ToString(EPortoContainerState::Running), SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success(), "failed to start 'start' container");
        }

    private:
        const size_t INITIAL_DELAY_MS = 2000;
        const size_t BACKOFF_MS = 2000;
    };

    TTest test("TestWorkloadStopInitialDelayAndBackOff");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedRunningAfterTries) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 32, breakHookRemoved);

            auto status = WorkloadStatusRepository_->GetObjectStatus(WorkloadId_);
            UNIT_ASSERT_EQUAL_C(
                status.state()
                , API::EWorkloadState_REMOVED
                , TConsoleRenderer(false).Render(Tree_)
            );
            UNIT_ASSERT_EQUAL_C(
                status.start().last().state()
                , API::EContainerState_KILLED_EXTERNALLY
                , TConsoleRenderer(false).Render(Tree_)
            );

            auto stopStatus = status.stop_status().unix_signal_status();
            UNIT_ASSERT_EQUAL_C(stopStatus.success_counter(), 3, stopStatus.success_counter());
            UNIT_ASSERT_EQUAL_C(stopStatus.error_counter(), 0, stopStatus.error_counter());
            UNIT_ASSERT_EQUAL_C(
                stopStatus.time_limit().consecutive_successes_counter()
                , 3
                , stopStatus.time_limit().consecutive_successes_counter()
            );
            UNIT_ASSERT_EQUAL_C(
                stopStatus.time_limit().consecutive_failures_counter()
                , 0
                , stopStatus.time_limit().consecutive_failures_counter()
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["UNIX_SIGNAL_STOP_RESTART_PERIOD_SCALE"] = "0";
            specificReplace["UNIX_SIGNAL_STOP_RESTART_PERIOD_BACKOFF"] = "1";
            specificReplace["UNIX_SIGNAL_STOP_MAX_RESTART_PERIOD"] = "1";

            specificReplace["STOP_MAX_TRIES"] = "3";
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            const TPortoContainerName workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c 'trap \"echo You have no power here\" SIGTERM; for((;;)); do echo none > /dev/null; done'").Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
            UNIT_ASSERT_EQUAL_C(ToString(EPortoContainerState::Running), SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success(), "failed to start 'start' container");
        }
    };

    TTest test("TestWorkloadRemovedRunningAfterTries");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadRemovedNoStart) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookRemoved = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_REMOVED;
            };

            TickTree(Tree_, 60, breakHookRemoved);
            UNIT_ASSERT_EQUAL_C(WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state(), API::EWorkloadState_REMOVED, TConsoleRenderer(false).Render(Tree_));
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }
    };

    TTest test("TestWorkloadRemovedNoStart");
    test.DoTest();
}

Y_UNIT_TEST(TestWorkloadDeactivating) {
    class TTest : public ITestTreeWorkloadUnixSignalStopCanon {
    public:
        TTest(const TString& testName)
            : ITestTreeWorkloadUnixSignalStopCanon(testName)
        {
        }

    protected:
        void Test() override {
            PrepareBox();
            PrepareWorkloadStartContainer();
            WorkloadStatusRepository_->UpdateObjectTargetState(WorkloadId_, API::EWorkloadTarget_REMOVED);

            auto breakHookDeactivating = [this]() {
                return WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state() == API::EWorkloadState_DEACTIVATING;
            };

            TickTree(Tree_, 32, breakHookDeactivating);
            UNIT_ASSERT_EQUAL_C(
                WorkloadStatusRepository_->GetObjectStatus(WorkloadId_).state()
                , API::EWorkloadState_DEACTIVATING
                , TConsoleRenderer(false).Render(Tree_)
            );
        }

        TMap<TString, TString> GetSpecificReplace() const override {
            TMap<TString, TString> specificReplace;
            specificReplace["STOP_HOOK_TYPE"] = "unix_signal";

            return specificReplace;
        }

        void PrepareWorkloadStartContainer() final {
            const TPortoContainerName workloadStartName = GetFullWorkloadContainerName("start");

            SafePorto_->Create(workloadStartName).Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Command, "bash -c 'trap \"echo You have no power here\" SIGTERM; for((;;)); do echo none > /dev/null; done'").Success();
            SafePorto_->SetProperty(workloadStartName, EPortoContainerProperty::Private, PackContainerPrivate({CP_EMPTY, "tree_hash"}));
            SafePorto_->Start(workloadStartName).Success();
            UNIT_ASSERT_EQUAL_C(ToString(EPortoContainerState::Running), SafePorto_->GetProperty(workloadStartName, EPortoContainerProperty::State).Success(), "failed to start 'start' container");
        }
    };

    TTest test("TestWorkloadDeactivating");
    test.DoTest();
}

}

} // namespace NInfra::NPodAgent::NTreeTest
