#include "client.h"
#include "test_service.h"

#include <infra/pod_agent/libs/grpc_service/service.h>

#include <infra/libs/logger/logger.h>
#include <infra/libs/service_iface/str_iface.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/tests_data.h>

namespace NInfra::NPodAgent::NTestClient {

TGrpcServiceConfig GenerateGrpcServiceConfig(ui16 port) {
    TGrpcServiceConfig cfg;
    cfg.SetPort(port);
    return cfg;
}

template <typename TRequest, typename TReply, typename Callback>
void CheckMethod(TClient& client, TTestService& service,  Callback&& callback, TStringBuf command, const TString& postData = "") {
    TString resposne;
    TAttributes replyAttributes;
    UNIT_ASSERT(command == "/set_log_level" || !service.WasAction(TString{command}));
    (client.*callback)(RequestPtr<TRequest>(command, postData, TAttributes()), ReplyPtr<TReply>(resposne, replyAttributes));
    UNIT_ASSERT(command == "/set_log_level" || service.WasAction(TString{command}));
}

Y_UNIT_TEST_SUITE(ClientMethods) {

Y_UNIT_TEST(TestNoArgumentsMethodsAndHelp) {
    NTesting::TPortHolder port = NTesting::GetFreePort();
    const TGrpcServiceConfig config = GenerateGrpcServiceConfig(port);

    TLoggerConfig loggerConfig;
    TLogger logger(loggerConfig);
    TTestService service;
    auto fakeFrame = logger.SpawnFrame();

    const TString address = "localhost:" + ToString(static_cast<i32>(config.GetPort()));

    TGrpcService grpcService(config, service);
    grpcService.Start(fakeFrame);

    TClient client(address);

    CheckMethod<TEmptyRequest<TReqPing>, TRawDataReply<TRspPing>>(client, service, &TClient::Ping, "/ping");

    CheckMethod<TEmptyRequest<TReqSensors>, TRawDataReply<TRspSensors>>(client, service, &TClient::Sensors, "/sensors");

    CheckMethod<TEmptyRequest<TReqSensors>, TRawDataReply<TRspSensors>>(client, service, &TClient::UserSensors, "/user_sensors");

    CheckMethod<TEmptyRequest<TReqVersion>, TRawDataReply<TRspVersion>>(client, service, &TClient::Version, "/version");

    CheckMethod<TEmptyRequest<TReqReopenLog>, TEmptyReply<TRspReopenLog>>(client, service, &TClient::ReopenLog, "/reopen_log");

    CheckMethod<TEmptyRequest<TReqConfig>, TProto2JsonReply<TRspConfig>>(client, service, &TClient::Config, "/config");

    CheckMethod<TJson2ProtoRequest<API::TPodAgentRequest, true>, TProto2JsonReply<API::TPodAgentStatus>>(client, service, &TClient::UpdatePodAgentRequest, "/update_pod_request", "{}");

    CheckMethod<TEmptyRequest<TReqGetPodAgentStatus>, TProto2JsonReply<API::TPodAgentStatus>>(client, service, &TClient::GetPodAgentStatus, "/get_pod_agent_status");

    CheckMethod<TEmptyRequest<TReqShutdown>, TEmptyReply<TRspShutdown>>(client, service, &TClient::Shutdown, "/shutdown");

    grpcService.Shutdown();
}

Y_UNIT_TEST(TestUpdatePodAgentRequestWithUnkownJsonFields) {
    NTesting::TPortHolder port = NTesting::GetFreePort();
    const TGrpcServiceConfig config = GenerateGrpcServiceConfig(port);

    TLoggerConfig loggerConfig;
    TLogger logger(loggerConfig);
    TTestService service;
    auto fakeFrame = logger.SpawnFrame();

    const TString address = "localhost:" + ToString(static_cast<i32>(config.GetPort()));

    TGrpcService grpcService(config, service);
    grpcService.Start(fakeFrame);

    TClient client(address);

    CheckMethod<TJson2ProtoRequest<API::TPodAgentRequest, true>, TProto2JsonReply<API::TPodAgentStatus>>(client, service, &TClient::UpdatePodAgentRequest, "/update_pod_request", "{\"unknowwn_field\": \"true\"}");

    grpcService.Shutdown();
}

Y_UNIT_TEST(TestSetLogLevel) {
    NTesting::TPortHolder port = NTesting::GetFreePort();
    const TGrpcServiceConfig config = GenerateGrpcServiceConfig(port);

    TLoggerConfig loggerConfig;
    TLogger logger(loggerConfig);
    TTestService service;
    auto fakeFrame = logger.SpawnFrame();

    const TString address = "localhost:" + ToString(static_cast<i32>(config.GetPort()));

    TGrpcService grpcService(config, service);
    grpcService.Start(fakeFrame);

    TClient client(address);

    UNIT_ASSERT_STRINGS_EQUAL("Unknown", service.LogLevel());

    CheckMethod<TJson2ProtoRequest<TReqSetLogLevel>, TEmptyReply<TRspSetLogLevel>>(client, service, &TClient::SetLogLevel, "/set_log_level", R"({"Level":"42"})");
    UNIT_ASSERT_STRINGS_EQUAL("42", service.LogLevel());

    CheckMethod<TJson2ProtoRequest<TReqSetLogLevel>, TEmptyReply<TRspSetLogLevel>>(client, service, &TClient::SetLogLevel, "/set_log_level", R"({"Level":"Debug"})");
    UNIT_ASSERT_STRINGS_EQUAL("Debug", service.LogLevel());

    CheckMethod<TEmptyRequest<TReqShutdown>, TEmptyReply<TRspShutdown>>(client, service, &TClient::Shutdown, "/shutdown");

    grpcService.Shutdown();
}

}

} // namespace NInfra::NPodAgent::NTestClient
