#include "common.h"

#include <contrib/libs/re2/re2/re2.h>
#include <library/cpp/json/yson/json2yson.h>
#include <util/generic/maybe.h>

namespace NInfra::NPodAgent {

inline static const TString SALT_PREFIX = "salt=";
inline static const ui64 SALT_SIZE = 8;
inline static const TString MD5_PREFIX = "md5=";

ui64 TimestampToMicros(const google::protobuf::Timestamp& timestamp) {
    return timestamp.seconds() * (ui64)1000000 + timestamp.nanos() / 1000;
}

TMaybe<std::pair<TString, TString>> ExtractSaltAndMd5FromSecret(const TString& secret) {
    if (!RE2::FullMatch(secret, "^<secret salt=.{8} md5=[a-zA-Z0-9]{32}>$")) {
        return Nothing();
    }

    auto saltBegin = secret.find(SALT_PREFIX) + SALT_PREFIX.size();
    auto saltEnd = secret.find(MD5_PREFIX, saltBegin + SALT_SIZE) - 1;
    auto md5Begin = secret.find(MD5_PREFIX, saltEnd) + MD5_PREFIX.size();
    auto md5End = secret.size() - 1;

    return std::pair(
        secret.substr(saltBegin, saltEnd - saltBegin),
        secret.substr(md5Begin, md5End - md5Begin)
    );
}

bool GetBooleanDeployLabel(const API::TPodAgentRequest& podAgentRequest, const TString& labelKey) {
    bool result = false;
    for (const auto& it : podAgentRequest.pod_dynamic_attributes().labels().attributes()) {
        if (it.key() == "deploy") {
            NJson::TJsonValue deployLabels = NJson::TJsonValue(NJson::JSON_MAP);
            if (NJson2Yson::DeserializeYsonAsJsonValue(it.value(), &deployLabels)) {
                result = deployLabels.GetMap().Value(labelKey, NJson::TJsonValue{}).GetBoolean();
            }
        }
    }
    return result;
}

} // namespace NInfra::NPodAgent
